/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;


/**
 * A class to represent a log messenger.
 */
public class LogMessenger {

	private int meSeverity = LogMessage.MSG_INFORMATION;

	private final List<LogMessageHandler> mMsgHandlerArray = new ArrayList<LogMessageHandler>();

	private final Map<Integer, Integer> mSeverityMap = new HashMap<Integer, Integer>();

	private final List<LogMessage> mStoredMsgArray = new ArrayList<LogMessage>();

	/**
	 * Instantiates a log messenger.
	 */
	public LogMessenger() {
	}

	/**
	 * Constructor
	 * @param handler
	 *            pointer to a message handler
	 * @param sAppContext
	 *            a string representing the application context/name
	 *
	 * @exclude from published api.
	 */
	public LogMessenger(LogMessageHandler handler, String sAppContext /* = "" */) {
		handler.setAppContext(sAppContext);
		addHandler(handler, "");
	}

	/**
	 * Add a message handler.
	 * @param handler
	 *            a message handler
	 * @param sAppContext
	 *            a string representing the application context/name
	 *
	 * @exclude from published api.
	 */
	public void addHandler(LogMessageHandler handler, String sAppContext /* = "" */) {
	    //if there are stored messages, add them to the new handler
	    int n = mStoredMsgArray.size();
	    for (int i = 0; i < n; i++) {
	        handler.sendMessage(mStoredMsgArray.get(i));
	    }
	    //
	    // add the handler
	    //
	    handler.setAppContext(sAppContext);
	    mMsgHandlerArray.add(handler);
    }

	/**
	 * Check through the message configuration elements to determine the
	 * severity of the given error.
	 * @param sContext
	 *            the context of the model; it should be the parent node of
	 *            "messaging"
	 * @param model
	 *            the configuration model containing the messaging information
	 * @param exFull
	 *            the error.
	 * @remarks oExFull could be thrown from this function, if the severity is
	 *          FATAL_ERROR
	 */
	void checkMessaging(String sContext, Model model, ExFull exFull) {
		boolean bFound = false;

		if (model == null)
			throw exFull;
		
		Node messaging = model.resolveNode(sContext + ".messaging");
		if (messaging == null || messaging.getNodes().length() == 0)
			throw exFull;

		NodeList nodeList = messaging.getNodes();
		for (int nExInd = 0; nExInd < exFull.count(); nExInd++) {
			
			int nExId = exFull.getResId(nExInd);

			for (int nMsInd = nodeList.length(); nMsInd > 0; nMsInd--) {
				
				Element msgId = (Element)(((Element)nodeList.item(nMsInd - 1)).resolveNode("msgId"));
				if (msgId == null)
					break;

				int nMessageId = 0;
				String sValue = msgId.getText(false, false, false).getValue();
				try { nMessageId = Integer.parseInt(sValue); }
				catch (NumberFormatException ex) { }

				if (nMessageId == nExId) {

					bFound = true;
					int eSeverity = LogMessage.MSG_FATAL_ERROR;
					Element severity = (Element)(((Node)nodeList.item(nMsInd - 1)).resolveNode(XFA.SEVERITY));
					if (severity != null) {
						
						String sSeverity = severity.getText(false, false, false).getValue();

						if (sSeverity.equals("ignore"))
							eSeverity = LogMessage.MSG_IGNORE;
						else if (sSeverity.equals("warning"))
							eSeverity = LogMessage.MSG_WARNING;
						else if (sSeverity.equals("information"))
							eSeverity = LogMessage.MSG_INFORMATION;
						else if (sSeverity.equals("error"))
							eSeverity = LogMessage.MSG_FATAL_ERROR;
						else if (sSeverity.equals("trace"))
							eSeverity = LogMessage.MSG_TRACE;
						else if (!StringUtils.isEmpty(sSeverity)) {
							MsgFormatPos oFormat = new MsgFormatPos(ResId.InvalidOptionValueException);
							oFormat.format(XFA.SEVERITY);
							oFormat.format(sSeverity);

							throw new ExFull(oFormat);
						}
					}

					if (eSeverity == LogMessage.MSG_FATAL_ERROR)
						throw exFull;

					if (eSeverity != LogMessage.MSG_IGNORE) {
						
						LogMessage exceptionMessage = new LogMessage(exFull, eSeverity);
						sendMessage(exceptionMessage);
					}

					break;
				}
			}
			
			if (!bFound)
				throw exFull;
		}
	}

	/**
	 * Return a list of all the handlers.
	 * @return a jfArray object containing all of the handlers for this
	 *         messenger.
	 */
	List<LogMessageHandler> enumerateHandlers() {
		return mMsgHandlerArray;
	}

	/**
	 * As the messenger sends messages we keep track of the highest severity
	 * message sent.
	 * 
	 * @return a int enum, representing the highest severity message sent by
	 *         this messenger.
	 *
	 * @exclude from published api.
	 */
	public int getSeverity() {
		return meSeverity;
	}

	/**
	 * Remove all handlers.
	 *
	 * @exclude from published api.
	 */
	void removeAll() {
		mMsgHandlerArray.clear();
	}

	/**
	 * Remove a message handler at a given index.
	 * @param nIndex
	 *            the index
	 *
	 * @exclude from published api.
	 */
	void removeHandler(int nIndex) {
		// Check for out-of-bounds index
		if (nIndex < mMsgHandlerArray.size()) {
			mMsgHandlerArray.remove(nIndex);
		}
	}
	
	/**
	 * @exclude from published api.
	 */
	public void removeStoredMessages() {
		mStoredMsgArray.clear();
	}

	/**
	 * Flush target.
	 *
	 * @exclude from published api.
	 */
	public void flush() {
		int size = mMsgHandlerArray.size();
		for (int i = 0; i < size; i++) {
			mMsgHandlerArray.get(i).flush();
		}
	}
	
	/**
	 * Send message to target.
	 * @param source
	 *            contains message Id and text
	 * @param eSeverity
	 *            the message severity
	 *
	 * @exclude from published api.
	 */
	void sendMessage(ExFull source, int eSeverity) {
		LogMessage oMessage = new LogMessage(source, eSeverity);
		sendMessage(oMessage);
	}

	/**
	 * Send message to target.
	 * @param nId
	 *            the message Id
	 * @param sText
	 *            the message text
	 * @param eSeverity
	 *            the message severity
	 *
	 * @exclude from published api.
	 */
	public void sendMessage(int nId, String sText, int eSeverity) {
		LogMessage oMessage = new LogMessage(nId, sText, eSeverity);
		sendMessage(oMessage);
	}

	/**
	 * Send message to target.
	 * @param message
	 *            the message
	 *
	 * @exclude from published api.
	 */
	public void sendMessage(LogMessage message) {
		// Each message contains one or more LogMessageData components.
		// Each of these components has an id and a severity. The component
		// with the highest severity becomes the severity of the overall message.
		// As each message is sent via the messenger, the messenger as well
		// keeps track of the highest severity of all messages it sends for
		// reporting purposes.

		// Get the message's current overall severity.
		int eMessageSeverity = message.getSeverity();
				
		// Check to see if we need to modify the severity.
		// The severity map is a user specified list of message
		// ids that they wish to modify. Either lowering/increasing
		// the severity level or ignoring the message altogether. These
		// messages are populated by specifying them in the configuration model.
		int nSize = mSeverityMap.size();
		if (nSize > 0) {
			// We will build up an error string that will be used if the
			// message is fatal.  If it is we will create and throw an
			// exception with this message.
			// JavaPort: This variable is never used in C++
			//String sExceptionMessage = "";

			// Get the current message and this messengers severity, used for
			// reporting purposes.
			//XFALogMessage::XFALogMessageSeverity eMessageSeverity = oMessage.getSeverity();
			//XFALogMessage::XFALogMessageSeverity eMessengerSeverity = getSeverity();

			int nUpdatedMessageSeverity = 0;
			
			// Search through all message data components
			int nCount = message.count();
			for (int i = 0; i < nCount; i++) {
				
				// Get the current LogMessageData
				LogMessageData msgLogData = message.get(i);

				// Get the id and severity
				int nMsgLogId = msgLogData.getId();
				int nMsgLogSeverity = msgLogData.getSeverity();

				// Add to the exception message.  Used if it's a fatal error.
				//sExceptionMessage += msgLogData.getText();

				// Does the LogMessageData severity need to be updated?
				if (!mSeverityMap.containsKey(nMsgLogId)) {
					int nMapSeverity = mSeverityMap.get(nMsgLogId);
					if (nMsgLogSeverity != nMapSeverity) {
						// We are in an update situation.  Change the current
						// LogMessageData severity appropriately
						nMsgLogSeverity = nMapSeverity;
						msgLogData.setSeverity(nMsgLogSeverity);
					}
				}
				
				// As we're looping through all of the LogMessageData
				// components, keep track of the highest severity
				if (nMsgLogSeverity > nUpdatedMessageSeverity) {
					nUpdatedMessageSeverity = nMsgLogSeverity;
				}
			}

			// Update the overall message severity
			eMessageSeverity = nUpdatedMessageSeverity;
			message.setSeverity(eMessageSeverity);
		}

		// Update the messenger's overall severity
		if (eMessageSeverity > getSeverity())
			setSeverity(eMessageSeverity);

		if (eMessageSeverity != LogMessage.MSG_IGNORE) {
			// Send the message to all specified handlers.
			int size = mMsgHandlerArray.size();
			for (int i = 0; i < size; i++) {
				mMsgHandlerArray.get(i).sendMessage(message);
			}
			
			// if no handlers, add to the stored messages queue
			if (size == 0)
				mStoredMsgArray.add(message);
		}

	}

	/**
	 * Send message to target.
	 * @param source
	 *            contains message Id and text
	 * @param eSeverity
	 *            the message severity
	 */
	void sendMessage(MsgFormatPos source, int eSeverity) {
		LogMessage oMessage = new LogMessage(source, eSeverity);
		sendMessage(oMessage);
	}

	private void setSeverity(int eSeverity) {
		meSeverity = eSeverity;
	}

	/**
	 * @exclude from published api.
	 */
	public List<LogMessage> storedMsgArray() {
		return mStoredMsgArray;
	}

	/**
	 * Check through the messaging elements in the configuration model and add
	 * them to a cache. When ever messages are sent, we will compare with the
	 * cache to see if the message severity needs to be modified.
	 * @param model
	 *            the configuration model containing the messaging information
	 * @exception InvalidOptionValueException
	 *                thrown if the user specified an invalid severity in the
	 *                configuration options.
	 *
	 * @exclude from published api.
	 */
	public void updateMessaging(Model model) {
		if (model != null) {
			Node context = model.getContext();
			if (model == context)
				context = model.resolveNode("present.common");
			Node messaging = context.resolveNode(XFA.MESSAGING);
			if (messaging != null && messaging.getFirstXFAChild() != null) {
				NodeList nodeList = messaging.getNodes();
				for (int i = 0; i < nodeList.length(); i++) {
					Node message = (Node)nodeList.item(i);
					Element msgId = (Element) message.resolveNode(XFA.MSGID);
					if (msgId == null)
						break;
					//
					// Got a msgId, so lets convert it to a int.
					//
					int nMessageId = 0;
					try {
						TextNode textNode = msgId.getText(true, false, false);
						if (textNode != null)
							nMessageId = Integer.parseInt(textNode.getValue());
					} catch (NumberFormatException e) {
					}
					// Now get the Severity
					Element severity = (Element) message.resolveNode(XFA.SEVERITY);
					int eSeverity = LogMessage.MSG_IGNORE;
					if (severity != null) {
						String sSeverity = "";
						TextNode textNode = severity.getText(true, false, false);
						if (textNode != null)
							sSeverity = textNode.getValue();
						if (sSeverity.equals("ignore"))
							eSeverity = LogMessage.MSG_IGNORE;
						else if (sSeverity.equals("warning"))
							eSeverity = LogMessage.MSG_WARNING;
						else if (sSeverity.equals("information"))
							eSeverity = LogMessage.MSG_INFORMATION;
						else if (sSeverity.equals("error"))
							eSeverity = LogMessage.MSG_FATAL_ERROR;
						else if (sSeverity.equals("trace"))
							eSeverity = LogMessage.MSG_TRACE;
						else {
							MsgFormatPos format
								= new MsgFormatPos(ResId.InvalidOptionValueException);
							format.format(XFA.SEVERITY).format(sSeverity);
							throw new ExFull(format);
						}
					}
					//
					// If the message is already in the list, then update it.
					// Last one wins.
					//
					mSeverityMap.put(Integer.valueOf(nMessageId), Integer.valueOf(eSeverity));
				}
			}
		}
	}
}
