/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.agent;

import java.io.*;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Document;
import com.adobe.xfa.DOMSaveOptions;
import com.adobe.xfa.Element;
import com.adobe.xfa.LogMessage;
import com.adobe.xfa.LogMessageData;
import com.adobe.xfa.LogMessageHandler;
import com.adobe.xfa.TextNode;
import com.adobe.xfa.ut.StringUtils;


/**
 * Class MessageHandlerXML formats messages to a XML file.
 * Class MessageHandlerXML will eventually support multiple targets(files)
 * and be able to select targets(files) depending on the severity of a message.
 *
 * @author		Ian Benedict Carreon
 *
 * @exclude from published api.
 */
public class MessageHandlerXML extends LogMessageHandler {

	private OutputStream moStream = null;

	private Document moDomDoc = null;

	private Element moRootDomElem = null;

	/**
	 * Constructor.
	 * Will write out to oSource but does not close the stream file.
	 * Note: OutputStream could be any platform-specific implementation.
	 *
	 * @param oSource - stream file
	 */
	public MessageHandlerXML(OutputStream oSource) {
		moStream = oSource;
	}

	/**
	 * Flushes any buffered messages to be written out
	 * to this object's output stream.
	 */
	public void flush() {
		if (moDomDoc != null) {
			DOMSaveOptions oOptions = new DOMSaveOptions();
			oOptions.setDisplayFormat(DOMSaveOptions.PRETTY_OUTPUT);
			oOptions.setIncludeDTD(false);
			moDomDoc.saveAs(moStream, moRootDomElem, oOptions);
		}
	}

	/**
	 * Send message to target using XML file format.
	 *
	 * @param oMessage - the message
	 */
	public void sendMessage(LogMessage oMessage) {
		if (moDomDoc == null) {
			AppModel app = new AppModel(null);
			moDomDoc = app.getDocument();
			app.newDOM();
			moRootDomElem = moDomDoc.createElementNS("", "log", null);
			moDomDoc.appendChild(moRootDomElem);
		}
		int nSize = oMessage.count();
		for (int i = 0; i < nSize; i++) {
			sendMessageData(oMessage.get(i));
		}
	}

	private void sendMessageData(LogMessageData oMessageData) {
		if (oMessageData.getSeverity() < getThreshold())
			return;
		Document doc = moRootDomElem.getOwnerDocument();
		Element message  = doc.createElementNS("", "m", null);
		//
		// Root:
		//
		// <log>
		//    <message>...</message>[0..n]
		// </log>
		//
		//
		// Message node schema:
		//
		// <m mid="xmlid" tid="cdata" sev="i | w | f | t | vw | ve" d="iso8601-date-time">...message-text...</m>
		//
		String sMsgId = "mid";
		String sThreadId = "tid";
		String sSeverity = "sev";
		String sDateTime = "d";
		String sRef = "ref";
		String sTidValue = Thread.currentThread().getName();
		Integer nMsgId = Integer.valueOf(oMessageData.getId());
		String sSeverityValue = getSeverity(oMessageData.getSeverity());
		String sDateTimeValue = oMessageData.getDateTime();
		if (oMessageData.getSeverity() == LogMessage.MSG_VALIDATION_ERROR || 
			oMessageData.getSeverity() == LogMessage.MSG_VALIDATION_WARNING)
			message.setAttribute("", sRef, sRef, oMessageData.getSOMExpression());
		else
			message.setAttribute("", sMsgId, sMsgId, nMsgId.toString());
		message.setAttribute("", sThreadId, sThreadId, sTidValue);
		message.setAttribute("", sSeverity, sSeverity, sSeverityValue);
		message.setAttribute("", sDateTime, sDateTime, sDateTimeValue);
		//
		// Add appContext tag if non-empty
		//
		if (!StringUtils.isEmpty(getAppContext())) {
			String sAppContext = "app";
			message.setAttribute("", sAppContext, sAppContext, getAppContext());
		}
		
		TextNode textVal = doc.getAppModel().createTextNode(null, null, oMessageData.getText());
		message.appendChild(textVal, false);
		moRootDomElem.appendChild(message, false);
	}

	private String getSeverity(int eSeverity) {
		switch (eSeverity) {
		case LogMessage.MSG_FATAL_ERROR:
			return "f";
		case LogMessage.MSG_WARNING:
			return "w";
		case LogMessage.MSG_INFORMATION:
			return "i";
		case LogMessage.MSG_TRACE:
			return "t";
		case LogMessage.MSG_VALIDATION_ERROR:
			return "ve";
		case LogMessage.MSG_VALIDATION_WARNING:
			return "vw"; 
		}
		return "";
	}

}
