/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.template;

import java.util.List;

import com.adobe.xfa.Attribute;
import com.adobe.xfa.Document;
import com.adobe.xfa.Element;
import com.adobe.xfa.EnumAttr;
import com.adobe.xfa.Node;
import com.adobe.xfa.ProcessingInstruction;
import com.adobe.xfa.STRS;
import com.adobe.xfa.StringAttr;
import com.adobe.xfa.TextNode;
import com.adobe.xfa.XFA;
import com.adobe.xfa.Manifest;
import com.adobe.xfa.Generator;
import com.adobe.xfa.content.DateTimeValue;
import com.adobe.xfa.content.DateValue;
import com.adobe.xfa.content.DecimalValue;
import com.adobe.xfa.content.FloatValue;
import com.adobe.xfa.content.IntegerValue;
import com.adobe.xfa.content.TimeValue;
import com.adobe.xfa.template.containers.Field;
import com.adobe.xfa.template.containers.Subform;
import com.adobe.xfa.template.formatting.Button;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.PictureFmt;
import com.adobe.xfa.ut.StringUtils;


/**
 * Fixup errors found during the template DOM load.
 * NOTE! this implementation does not attempt to do nearly the amount of
 * fixing up that we do on the C++ side.  For now we're mainly interested 
 * in fixing problems that show up as syntax errors and clog our log file.
 *
 * @exclude from published api.
 */
public class TemplateModelFixup {
	private final List<ExFull> mErrorList;
	private final List<Element> mErrorContextList;
	private final TemplateModel mModel;
	private final int meGenerator;
	private final int mnBuildNumber;
	private boolean mbFixupRenderCache;
	private boolean mbCheckedInteractive;
	private boolean mbIsInteractive;
	private boolean mbCheckedCache;


	/**
	 * Constructor.  Pass in the model that we want to patch up 
	 * immediately after it's been loaded and before its error list has been cleared.
	 * @param model The model to fixup.
	 */
	public TemplateModelFixup(TemplateModel model, boolean bFixupRenderCache) {
		mModel = model;		
		mErrorList = model.getErrorList();
		mErrorContextList = model.getErrorContextList();
		mbFixupRenderCache = bFixupRenderCache;
		
		Generator generator = mModel.getGenerator();
		if (generator != null) {
			meGenerator = generator.generator();
			mnBuildNumber = generator.getBuildNumber();
		}
		else { 
			meGenerator = Generator.XFAGen_Unknown;
			mnBuildNumber = 0;
		}
	}
	
	/**
	 * Process whatever fixups we can based on the context nodes associated with
	 * the errors found during load.  Note that for any successful fixups we'll 
	 * remove the errors from the error list accordingly.
	 *
	 */
	public void processFixups() {
		for (int i = mErrorContextList.size(); i > 0; i--) {
			Element e = mErrorContextList.get(i - 1);
			boolean bHandled = false;
			
			if (e.getClassName() == XFA.LAYOUT) {
				// Layout no longer valid in template 2.0
				e.remove();
				bHandled = true;
			}
			
			if (meGenerator == Generator.XFAGen_FF99V250_01 ||
			    meGenerator == Generator.XFAGen_FF99V310 	||
			    meGenerator == Generator.XFAGen_FF99V40 	||
			    meGenerator == Generator.XFAGen_FF99V50 	||
			    meGenerator == Generator.XFAGen_JDV530_01     ) {
			
				// This fixup applies to all versions of FF99 (only)
				if (meGenerator != Generator.XFAGen_JDV530_01) {
					if (e.isSameClass(XFA.EXTRASTAG)) {
						bHandled = handleExtrasTag(e);
					}
				}
				
				if (e.getName() == XFA.DESC && !(e.getXFAParent() instanceof TemplateModel)) {
					e.privateSetName(XFA.EXTRAS);
				}
				
				// XFA 1.0 -> XFA 2.0 ref fixup
				// FIXUP: If this is a ref child of an value,traverse,
				// then remove it. For traverse, make ref an attribute.
				if ((e.getName() == XFA.REF) && !(e.getXFAParent() instanceof Manifest)) {
					bHandled = upgradeXFARef(e);
				}
				
				// trim out all <extras> tags.
				if (e.getName() == XFA.EXTRAS) {
					if (e.getXFAParent() instanceof TemplateModel) {
						bHandled = doDescFixup(e);
					}
					
					e.remove();
				}
			}
			
			if (bHandled) {
				mErrorList.remove(i - 1);
				mErrorContextList.remove(i - 1);
			}
		}
	}

	/**
	 * @exclude from published api.
	 */

	public void applyFixups(Node node) {

		if (node == null)
			return;
		
		if (stripRenderCache()) {
			if ((node instanceof ProcessingInstruction) &&
				(node.getName().equals(STRS.BOUNDS) ||
				 node.getName().equals(STRS.TEXTRUN) ||
	             node.getName().equals(STRS.LINE) ||
	             node.getName().equals(STRS.SUBSET) ||
	             node.getName().equals(STRS.MACRO))) {
				node.remove();
				return;
			}
		}

		boolean bNodeRemoved = false;
		if (node instanceof Element &&
			meGenerator != Generator.XFAGen_Unknown)
			bNodeRemoved = fixupElement((Element)node);

		if (!bNodeRemoved) {
			Node child = node.getFirstXMLChild();
			while (child != null) {
				Node nextChild = child.getNextXMLSibling();
				applyFixups(child);
				child = nextChild;
			}
		}
	}

	private boolean fixupElement(Element node) {
		
		boolean bRemoved = false;
		
		//
		// Fixups for AdobeDesigner_V6.0_SAP
		//
		if (meGenerator == Generator.XFAGen_AdobeDesignerV60_SAP &&
			node instanceof Subform)
			// Fix for ECO#4 Table Accessibility
			fixupTableAccessibility(node);
		//
		// Fixups for AdobeDesiger_V6.0
		//
		if (meGenerator == Generator.XFAGen_AdobeDesignerV60 ||
			meGenerator == Generator.XFAGen_TemplateDesignerV60d) {
			// Fix for Vantive 656048.
			// Make sure the namespace for the body elements
			// children are set to XFASTRS::getString(aXFASTRS_XHTMLNS)
			if (node.getName() == XFA.EXDATA)
				fixupNameSpace(node);

			// Fix for Vantive 663566... (fix anything prior to build 4029 = Jan. 29th 2004)
			if ((mnBuildNumber < 4029) && (node.getClassTag() == XFA.BORDERTAG))
				bRemoved = fixup4SAPBorders(node);
			
			if (bRemoved)
				return bRemoved;

			// Fix format vs. validation pictures - anything prior to build 3346 = Dec 12th, 2003
			if ((mnBuildNumber < 3346) && (node.getClassTag() == XFA.VALIDATETAG))
				fixupPictures(node);
		}

		//
		// Fix any default values that are not in canonical format in
		// any Designer released prior to build 5048 (Feb 18, 2005).
		//
		if (((meGenerator == Generator.XFAGen_AdobeDesignerV60) ||
			 (meGenerator == Generator.XFAGen_AdobeDesignerV60_SAP) ||
			 (meGenerator == Generator.XFAGen_AdobeDesignerV70) &&
		     (mnBuildNumber < 5048)) &&
		    (node instanceof Field))
			fixupDefaults((Field)node);

		//
		// Fixups for AdobeDesiger_V7.0
		//
		if (meGenerator == Generator.XFAGen_AdobeDesignerV70 ||
			meGenerator == Generator.XFAGen_AdobeDesignerV70_SAP ||
			meGenerator == Generator.XFAGen_AdobeDesignerV71 ||
			meGenerator == Generator.XFAGen_AdobeDesignerV71_SAP)
			fixupButtonPresence(node);
		
		return bRemoved;
	}

	private void fixupButtonPresence(Element node) {
		boolean bButtonFound = false;
		
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child.getClassTag() == XFA.UITAG) {
				// figure out if its UI is a button type
				for (Node child2 = child.getFirstXMLChild(); child2 != null; child2 = child2.getNextXFASibling()) {
					if (child2 instanceof Button) {
						bButtonFound = true;
						break;
					}
				}
				break;
			}
		}

		if (!bButtonFound)
			return;

		if (!isFormInteractive()) {
			Attribute oPresenceAttr = node.getAttribute(XFA.PRESENCETAG, true, false);
			Attribute oRelevantAttr = node.getAttribute(XFA.RELEVANTTAG, true, false);

			if ((oPresenceAttr != null) && (oRelevantAttr != null)) {
				String sRelevant = "+print";
			
				if (oPresenceAttr.getAttrValue().equals(EnumAttr.getString(EnumAttr.PRESENCE_INVISIBLE))
					&& oRelevantAttr.getAttrValue().equals(sRelevant)) {

					// Add a presence=visible value to the container.
					// create an attribute for presence="visible"
					node.setAttribute(new StringAttr(XFA.PRESENCE, EnumAttr.getString(EnumAttr.PRESENCE_VISIBLE)), XFA.PRESENCETAG);
				}
			}
		}
	}


	private boolean isFormInteractive() {
		
		if (!mbCheckedInteractive) {
			String sDestination = null;
			Node oConfigNode = mModel.resolveNode("$config.present.destination");
			if (oConfigNode != null) {
				Node textNode = oConfigNode.getFirstXFAChild();
				if (textNode instanceof TextNode)
					sDestination = ((TextNode)textNode).getValue();
			}

			if (!StringUtils.isEmpty(sDestination)) {
				// Check is we are dealing with an interactive form.
				// Used for a FF99 fixup with Presentation Agent.  This
				// will not affect other agents, and will only be done once.
				String sSOM = "$config.present.";
				sSOM += sDestination;
				sSOM += ".interactive";

				oConfigNode = mModel.resolveNode(sSOM);
				if (oConfigNode != null) {
					Node textNode = oConfigNode.getFirstXFAChild();
					if (textNode instanceof TextNode) {
						String sInteractive = ((TextNode)textNode).getValue();

						if (sInteractive.equals("1"))
							mbIsInteractive = true;
					}
				}
			}

			mbCheckedInteractive = true;
		}

		return mbIsInteractive;
	}


	private void fixupDefaults(Field field) {
		//
		// with a bind picture and a default value...
		//
		Element oBind = null;
		Element oValue = null;
		for (Node child = field.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child.getClassTag() == XFA.BINDTAG) {
				oBind = (Element)child;
			}
			else if (child.getClassTag() == XFA.VALUETAG) {
				oValue = (Element)child;
			}
		}
		if ((oBind == null) || (oValue == null))
			return;

		Element oPicture = null;
		for (Node child = oBind.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child.getClassTag() == XFA.PICTURETAG) {
				oPicture = (Element)child;
				break;
			}
		}
		if (oPicture == null)
			return;

		//
		// Found one, so get the field's bind picture.
		//
		Node oPictChild = oPicture.getFirstXFAChild();
		if (oPictChild == null)
			return;
		String sPicture = oPictChild.getData();
		//
		// And get the field' locale.
		//
		String sLocale;
		Attribute oLocale = field.getAttribute(XFA.LOCALETAG, true, false);
		if (oLocale != null)
			sLocale = oLocale.getAttrValue();
		else
			sLocale = field.getXMLParent().getInstalledLocale();

		for (Node child = oValue.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			PictureFmt oPict = new PictureFmt(sLocale);
			//
			// If the value matches its bind picture Then store
			// its canonical representation Otherwise store nothing.
			//
			if (child.getClassTag() == XFA.FLOATTAG) {
				double d = ((FloatValue)child).getValue();
				String sCanon = oPict.parse(Double.toString(d), sPicture, null);
				((FloatValue)child).setValue(sCanon, false, true, false);
			}
			else if (child.getClassTag() == XFA.DECIMALTAG) {
				double d = ((DecimalValue)child).getValue();
				String sCanon = oPict.parse(Double.toString(d), sPicture, null);
				((DecimalValue)child).setValue(sCanon, false, true, false);
			}
			else if (child.getClassTag() == XFA.INTEGERTAG) {
				int i = ((IntegerValue)child).getValue();
				String sCanon = oPict.parse(Integer.toString(i), sPicture, null);
				((IntegerValue)child).setValue(sCanon, false);
			}
			else if (child.getClassTag() == XFA.DATETAG) {
				String date = ((DateValue)child).getValue();
				String sCanon = oPict.parse(date, sPicture, null);
				((DateValue)child).setValue(sCanon);
			}
			else if (child.getClassTag() == XFA.TIMETAG) {
				String time = ((TimeValue)child).getValue();
				String sCanon = oPict.parse(time, sPicture, null);
				((TimeValue)child).setValue(sCanon);
			}
			else if (child.getClassTag() == XFA.DATETIMETAG) {
				String dateTime = ((DateTimeValue)child).getValue();
				String sCanon = oPict.parse(dateTime, sPicture, null);
				((DateTimeValue)child).setValue(sCanon);
			}
		}
	}

	private void fixupPictures(Element node) {
		// Copy the validate/picture node from the format/picture,
		// if the formatTest attribute is set to warning or error,
		// and if the picture node is not already there.

		boolean bPictureFound = false;
		// Check to make sure the validate/picture node is not already there.
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child.getClassTag() == XFA.PICTURETAG) {
				bPictureFound = true;
				break;
			}
		}
		
		if (bPictureFound == false) {
			// Check if the formatTest attribute is set to warning (the default) or error.
			Attribute oFormatTestAttr = node.getAttribute(XFA.FORMATTESTTAG, true, false);
			boolean bAdjust = true;
			if (oFormatTestAttr != null) {
				String sFormatTestAttrValue = oFormatTestAttr.getAttrValue();
				bAdjust = sFormatTestAttrValue.equals("error") || sFormatTestAttrValue.equals("warning");
			}
			if (bAdjust) {
				// Copy the validate/picture node from the format/picture,
				for (Node child = node.getXFAParent().getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
					if (child.getClassTag() == XFA.FORMATTAG) {
						for (Node picture = child.getFirstXFAChild(); picture != null; picture = picture.getNextXFASibling()) {
							if (picture.getClassTag() == XFA.PICTURETAG) {
								picture.clone(node);
								break;
							}
						}
						break;
					}
				}
			}
		}
	}

	private boolean fixup4SAPBorders(Element node) {
		// This is a fixup for SAP...
		// if we have a <border> with no <edge> or <corner>
		// add a <edge presence="hidden">
		// also
		// look at the <fill> element... if there is no attribute then 
		// remove the fill... if border has no <fill> no <edge> and no <corner>
		// delete the <border>
		boolean bRemoved = false;
		boolean bHasFill = false;
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if ((child.getClassTag() == XFA.EDGETAG) || (child.getClassTag() == XFA.CORNERTAG))
				return bRemoved;

			if (child.getClassTag() == XFA.FILLTAG) {
				for (Node oFillChild = child.getFirstXMLChild(); oFillChild != null; oFillChild = oFillChild.getNextXMLSibling()) {
					if (oFillChild instanceof Element)
						bHasFill = true;
				}

				if (!bHasFill)
					child.remove();
			}
		}

		if (!bHasFill) {
			// remove the whole border
			node.remove();
			bRemoved = true;
		}
		else {
			// add an hidden edge
			Element oEdge = mModel.createElement(XFA.EDGE, null, (Element)node);
			oEdge.setAttribute(new StringAttr(XFA.PRESENCE, "hidden"), XFA.PRESENCETAG);
		}
		
		return bRemoved;
	}

	private void fixupNameSpace(Element node) {
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child.getClassName() == STRS.BODY) {
				((Element)child).setNS(STRS.XHTMLNS);
				break;
			}
		}
	}


	private void fixupTableAccessibility(Element node) {

		// Get it's layout attribute and check to see it's a row.
		Attribute oAttr = ((Element)node).getAttribute(XFA.LAYOUTTAG); 

		// If it's a row then check for it's overflowLeader
		if (oAttr != null &&
			oAttr.getAttrValue().equals(EnumAttr.getString(EnumAttr.ROW))) {
			
			// If it has an overflowLeader, then hold on to this id.
			Element oBreak = null;
			for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
				if (child.getClassTag() == XFA.BREAKTAG) {
					oBreak = (Element)child;
					break;
				}
			}

			if (oBreak != null) {
				Attribute oOverflowLeaderAttr = oBreak.getAttribute(XFA.OVERFLOWLEADERTAG, true, false);
				if (oOverflowLeaderAttr != null) {
					// Make sure the parent is a table subform.
					Element oParent = (Element)node.getXFAParent();

					if (oParent instanceof Subform) 	{
						// Get it's layout attribute and check to see it's a row.
						oAttr = oParent.getAttribute(XFA.LAYOUTTAG, true, false);

						// If it's a row then check for it's overflowLeader
						if (oAttr != null &&
							oAttr.getAttrValue().equals(EnumAttr.getString(EnumAttr.TABLE))) {
							// Get the id and eat the leading #
							String sOverFlowLeader = oOverflowLeaderAttr.getAttrValue();
							sOverFlowLeader = sOverFlowLeader.substring(1);
							
							Document doc = node.getOwnerDocument();
							
							// Go find the header row
							String aModelNamespace = ((Element)mModel.getXmlPeer()).getNS();
							String aOverFlowLeader = sOverFlowLeader.intern();
							Element oHeader = doc.getElementByXFAId(aModelNamespace, aOverFlowLeader);
							
							if (oHeader instanceof Subform) {
								
								Element oAssist = oHeader.peekElement(XFA.ASSISTTAG, false, 0);
								if (oAssist == null)
									oAssist = mModel.createElement(XFA.ASSIST, XFA.ASSIST, oHeader);
								assert(oAssist != null);	
								if (oAssist != null)
									oAssist.setAttribute(new StringAttr(XFA.ROLE, STRS.TH), XFA.ROLETAG);
							}
						}
					}
				}
			}
		}
	}

	/**
	 * This implementation only handles a small subset of the fixups from the C++ implementation.
	 *
	 * @param e - The extras element
	 * @return true if we performed a fixup.  false otherwise.
	 */
	private boolean handleExtrasTag(Element e) {
		for (int i = 0; i < e.getNumAttrs(); i++) {
			Attribute a = e.getAttr(i);
			
			// June 2003. Template scrub. Remove type attribute from extras.			
			if (a.getQName() == XFA.TYPE) {
				e.removeAttr(i);
				return true;
			}
		}
		return false;
	}
	
	/**
	 * This is a fixup for FF99 forms only.
	 */
	private boolean upgradeXFARef(Element e) {

		boolean bHandled = false;
		Element parent = e.getXFAParent();
		if (parent.getName() == XFA.TRAVERSE) {

			// get the text value for <ref> element
			TextNode oText = e.getText(false, false, false);
			if (oText != null) {
				
				// now we get the value and create a ref attribute and set its value
				String sRefValue = oText.getValue();

				parent.setAttribute(new StringAttr(XFA.REF, sRefValue), XFA.REFTAG);
			}
			
			bHandled = true;
		}
		
		if (parent.getName() == XFA.TRAVERSE || parent.getName() == XFA.VALUE) {
			// Remove the ref child
			parent.removeChild(e);
			bHandled = true;
		}
		return bHandled;
	}
	
	/**
	 * This is a fixup for FF99 forms only.
	 */
	private boolean doDescFixup(Element e) {
		//
		// Move template-level <desc> and <extras> children to a top-level-subform
		// <desc>, with a few modifications
		//
		Element template = e.getXFAParent();
		assert template instanceof TemplateModel;
		
		Element oTopLevelSubform = (Element)template.resolveNode("#subform");
		if (oTopLevelSubform == null)
			return false;

		Element oDesc = (Element)oTopLevelSubform.resolveNode("#desc");
		if (oDesc == null) {
			oDesc = template.getModel().createElement(XFA.DESC, XFA.DESC, oTopLevelSubform);
		}

		Node nextSibling = null;
		// Note that we do not use getFirstXFAChild()/getNextXFASibling here since the
		// nodes under <desc> might not have a valid class tag.
		for (Node child = e.getFirstXMLChild(); child != null; child = nextSibling) {
			// Need to remember nextSibling here since moving node will change it
			nextSibling = child.getNextXMLSibling();			
			
			if (!(child instanceof Element))
				continue;
			
			Element childElement = (Element)child;
			
			if (childElement.isPropertyValid(XFA.NAMETAG)) {

				Attribute attribute = childElement.peekAttribute(XFA.NAMETAG);
				if (attribute != null) {
					String sName = attribute.toString();
					if (sName.equals("DC.Date"))			// convert to DublinCore "refined" name
						childElement.setAttribute(new StringAttr(XFA.NAME, "issued"), XFA.NAMETAG);
					else if (sName.startsWith("DC."))		// remove DC. prefix
						childElement.setAttribute(new StringAttr(XFA.NAME, sName.substring(3)), XFA.NAMETAG);
					else if (sName.equals("CreationDate"))	// convert to DublinCore name
						childElement.setAttribute(new StringAttr(XFA.NAME, "created"), XFA.NAMETAG);
				}
			}

			oDesc.appendChild(childElement, false);
		}
		
		return true;
	}
	
	private boolean stripRenderCache() {
		if (!mbCheckedCache) {
			if (!mbFixupRenderCache) {
				// mbFixupRenderCache is set by the application, config will NOT override it.
				// If this is not set, then use the value in the config.
				Node oConfigNode = mModel.resolveNode("$config.present.cache.renderCache",
													  true, false, false);
				if (oConfigNode != null) {
					if (((Element)oConfigNode).getEnum(XFA.ENABLETAG) == EnumAttr.BOOL_FALSE)
						mbFixupRenderCache = true;
				}
			}

			mbCheckedCache = true;
		}

		return mbFixupRenderCache;
	}

}
