/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.ut;


import java.util.HashMap;
import java.util.Map;

import com.adobe.xfa.protocol.Protocol;
import com.adobe.xfa.protocol.ProtocolUtils;


/**
 * A class to provide some utility methods in support of some
 * protocol I/O capabilities.
 * 
 * @exclude from published api.
 */
public class Resolver {

	private final static ThreadLocal<Map<String, Protocol>> moProtocolRegistry = new ThreadLocal<Map<String,Protocol>>(){
		protected synchronized Map<String, Protocol> initialValue(){
			return new HashMap<String, Protocol>();
		}
	};

//	/** 
//	 * @exclude from published api.
//	 */
//	public static final String gsFileNoServer = "file:///";

//	/** 
//	 * @exclude from published api.
//	 */
//	public static final String gsFileServer = "file://";

//	/** 
//	 * @exclude from published api.
//	 */
//	public static final String gsFile = "file:/";

	/** 
	 * @exclude from published api.
	 */
	public static final String gsServerIndicator = "://";
	
    private Resolver() {
    }

    /**
     * Sets (adds) the given protocol to the list of protocol handlers.
     * @param oProtocol a protocol handler.
     * Subsequent to this operation, the I/O of any URL whose scheme matches
     * the given protocol will be routed to the given protocol handler.
     */
    public static void setProtocol(Protocol oProtocol) {
    	if (oProtocol != null) {
    		//
    		// Get the given protocol's scheme.
    		//
    		String sScheme = oProtocol.scheme();
    		//
    		// Search the registry looking for first available entry, or
    		// the first entry that matches the given protocol's scheme.
    		//
    		// Register the given protocol.
    		moProtocolRegistry.get().put(sScheme, oProtocol);
    	}
    }

    /**
     * Uninstall all protocol handlers.  It is normally not necessary to make this
     * call, as protocol handlers are automatically deleted on exit.  However, when
     * a protocol handler is in a DLL that is dynamically loaded, it is possible for
     * that DLL to be unloaded before the automatic deletion takes place, leading to
     * a crash.  In that case the DLL must call ResetProtocols just prior to unloading
     * (for example in the destructor of a global variable).
	 * 
	 * @exclude from published api.
     */
    public static void resetProtocols() {
    	moProtocolRegistry.get().clear();
    }

    /**
     * Get the currently installed protocol handler associated
     * with the given scheme.
     * @param sScheme a URL's scheme.
     * @return the matching protocol handler.
     */
    public static Protocol getProtocol(String sScheme) {
    	//
    	// Search the registry looking for the given scheme, and
    	// return the first matching entry, if present.
    	//
    	Protocol protocol;
    	
    	if (sScheme.endsWith(":"))
			sScheme = sScheme.substring(0, sScheme.length() - 1);
		
		protocol = moProtocolRegistry.get().get(sScheme);
		if (protocol == null)
			//
			// Search the registry looking for the default scheme.
			//
			protocol = moProtocolRegistry.get().get("");
    	
    	return protocol;
    }

    /**
     * Crack an url into its constituent parts.
     * The URL is cracked from the following rule:
     * <pre>
     *     scheme://[user[:password]@]host[:port][/path]
     * </pre>
     * @param sUrl the URL to crack.
     * @param sScheme the cracked URL's scheme.
     * @param sUser the cracked URL's user, if any.
     * @param sPwd the cracked URL's password, if any.
     * @param sHost the cracked URL's host.
     * @param sPort the cracked URL's port, if any.
     * @param sPath the cracked URL's path, if any.
     */
    public static void crackUrl(String sUrl, 
    							StringHolder sScheme, StringHolder sUser, 
    							StringHolder sPwd, StringHolder sHost, 
    							StringHolder sPort, StringHolder sPath) {
		String sTmp = sUrl;
		int nFoundAt = 0;
		String sHostValue = null;
		//
		// Crack the scheme part.
		//
		nFoundAt = sTmp.indexOf(gsServerIndicator);
		if (nFoundAt != -1) {
			if (sScheme != null)
				sScheme.value = sTmp.substring(0, nFoundAt);
			sHostValue = sTmp.substring(nFoundAt + 3);
			if (sPath != null) {
				//
				// Crack the path part.
				//
				nFoundAt = sHostValue.indexOf('/');
				if (nFoundAt != -1) {
					sPath.value = sHostValue.substring(nFoundAt + 1);
					sHostValue = sHostValue.substring(0, nFoundAt);
				}
			}
		}
		//
		// Crack the scheme part of any non IP-based schemes.
		// No special attention required for IPv6-style
		// hosts ( [a:b:c] ) since this is the non IP case.
		//
		else  {
			nFoundAt = sTmp.indexOf(':');
			if (nFoundAt != -1) {
				if (sScheme != null)
					sScheme.value = sTmp.substring(0, nFoundAt);
				sHostValue = sTmp.substring(nFoundAt + 1);
			}
		}
		//
		// Crack the host part.
		//
		if (sHostValue != null) {
			nFoundAt = sHostValue.indexOf('@');
			if ((sUser != null) && (nFoundAt != -1)) {
				sUser.value = sHostValue.substring(0, nFoundAt);
				sHostValue = sHostValue.substring(nFoundAt + 1);
				//
				// Crack the user and password part.
				//
				if (sPwd != null) {
					nFoundAt = sUser.value.indexOf(':');
					if (nFoundAt != -1) {
						sPwd.value = sUser.value.substring(nFoundAt + 1);
						sUser.value = sUser.value.substring(0, nFoundAt);
					}
				}
			}
			
			if (sPort != null) {
				//
				// Crack the port part.  For IPv6-style hosts ( [a:b:c] ) start
				// at the trailing ].
				//
				int nOffset = sHostValue.indexOf(']');
				if (nOffset == -1)
					nOffset = 0;
				nFoundAt = sHostValue.indexOf(':', nOffset);
				if (nFoundAt != -1) {
					sPort.value = sHostValue.substring(nFoundAt + 1);
					sHost.value = sHostValue.substring(0, nFoundAt);
				}
			}
		}
		
		if (sHost != null)
			sHost.value = sHostValue;
		
   }

    /**
     * Canon(ize) an url into its constituent parts.
     * The URL is canonized into the following rule:
     * <pre>
     *     scheme://host[:port][/path]
     * </pre>
     * @param sScheme the URL's scheme.
     * @param sHost the URL's host.
     * @param sPort the URL's port, if any.
     * @param sPath the URL's path, if any.
     * @return The canonized URL.
     */
    public static String canonUrl(String sScheme, String sHost,
			 			   		  String sPort, String sPath) {
		StringBuilder sUrl = new StringBuilder(sScheme);
		// IP-based schemes:
		//			"file",		"ftp",		"ftps",		"gopher",	"http",
		//			"https",	"ldap", 	"nntp", 	"telnet"
		if (sScheme.equals("file") || sScheme.equals("ftp") || sScheme.equals("ftps") ||
				sScheme.equals("gopher") || sScheme.equals("http") || sScheme.equals("https") ||
				sScheme.equals("ldap") || sScheme.equals("nntp") || sScheme.equals("telnet"))
			sUrl.append(gsServerIndicator);
		else
			sUrl.append(':');

		if (sHost != null)
			sUrl.append(sHost);
		if (sPort != null && sPort.length() > 0) {
			sUrl.append(':');
			sUrl.append(sPort);
		}
		if (sPath != null && sPath.length() > 0) {
			sUrl.append('/');
			sUrl.append(sPath);
		}
		return sUrl.toString();
    }

    /**
     * URL encode a string
     * @param sUrl string to be encoded
     * @return URL encoded string
     */
    public static String urlEncode(String sUrl) {
		StringHolder sScheme = new StringHolder();
		StringHolder sUser = new StringHolder();
		StringHolder sPwd = new StringHolder();
		StringHolder sHost = new StringHolder();
		StringHolder sPort = new StringHolder();
		StringHolder sPath = new StringHolder();
		Resolver.crackUrl(sUrl, sScheme, sUser, sPwd, sHost, sPort, sPath);
		if (sPath.value != null) {
			String sEncodedPath = ProtocolUtils.urlEncode(sPath.value);
			if (!sEncodedPath.equals(sPath.value))
				return Resolver.canonUrl(sScheme.value, sHost.value, sPort.value, sEncodedPath);
		}
		return sUrl;
    }

    /**
     * Decode a URL encoded string
     * @param sUrl string to be decoded
     * @return Decoded string
     */
    public static String urlDecode(String sUrl) {
		return ProtocolUtils.urlDecode(sUrl);
    }
}

