/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2021 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.day.cq.analytics.sitecatalyst;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;

import com.day.cq.analytics.sitecatalyst.util.AnalyticsConfig;
import com.day.cq.analytics.sitecatalyst.util.TokenProviderProxy;
import com.day.cq.analytics.sitecatalyst.util.WebService;
import com.day.cq.analytics.sitecatalyst.impl.servlets.SitecatalystServlet;
import com.google.gson.*;
import org.apache.http.HttpEntity;
import org.apache.http.client.entity.EntityBuilder;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.day.cq.wcm.webservicesupport.Configuration;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public class ImsApiAdapter extends AbstractApiAdapter {

    private static final Logger LOG = LoggerFactory.getLogger(ImsApiAdapter.class);
    private static final String DEFAULT_ENDPOINT_URI = "https://analytics.adobe.io";
    private static final String SUPPORT_ENGINE = "oberon";

    private final WebService client;
    private final Configuration configuration;
    private final AnalyticsConfig analyticsConfig;

    public ImsApiAdapter(Map configuration,
                          WebService client,
                          TokenProviderProxy tokenProviderProxy) {
        this.configuration = null;
        this.client = client;
        this.analyticsConfig = new AnalyticsConfig(configuration, tokenProviderProxy);
    }

    @Override
    protected ApiKind getApiKind() {
        return ApiKind.IMS;
    }

    @Override
    protected Logger getLogger() {
        return LOG;
    }

    private URI buildURI(@Nonnull String uriStr, @Nullable Map<String,String> parameters) throws Exception {
        try {
            URI uri = new URI(uriStr);
            URIBuilder uriBuilder = new URIBuilder(uri);

            if (parameters != null) {
                for (Map.Entry<String,String> parameter : parameters.entrySet()) {
                    uriBuilder.addParameter(parameter.getKey(), parameter.getValue());
                }
            }
            if(LOG.isDebugEnabled())
                LOG.debug(uriStr);

            return uriBuilder.build();
        } catch (URISyntaxException e) {
            LOG.error("Unable to build URI", e);
            throw new Exception("Unable to build URI", e);
        }
    }


    @Override
    public String getSegments(Map params) throws SitecatalystException {
        try {
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/segments?includeType=all", analyticsConfig.getCompanyId()), params);
            String response = client.request(new HttpGet(uri), analyticsConfig);
            JsonObject jsonResponse = (JsonObject) new JsonParser().parse(response);
            response = jsonResponse.get("content").toString();
            String error = getError(response);
            if (error != null) {
                throw new SitecatalystException(error);
            }
            if(LOG.isDebugEnabled())
                LOG.debug(response);
            return response;
        } catch (Exception e) {
            LOG.error("Unable to get segments", e);
            throw new SitecatalystException(e.getMessage(), e);
        }
    }

    @Override
    public String getReportSuites(Map params) throws SitecatalystException {
        try {
            params.put("limit", "1000");
            params.put("expansion", "enabledSolutions");
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/collections/suites", analyticsConfig.getCompanyId()), params);
            String response = client.request(new HttpGet(uri), analyticsConfig);
            JsonObject jsonResponse = (JsonObject) new JsonParser().parse(response);
            JsonObject finalResponse = new JsonObject();
            finalResponse.add("report_suites", jsonResponse.get("content"));
            if(LOG.isDebugEnabled())
                LOG.debug(finalResponse.getAsString());
            return finalResponse.toString();
        } catch (Exception e) {
            LOG.error("Unable to get report suites", e);
            throw new SitecatalystException(e.getMessage(), e);
        }
    }

    public JsonObject getGlobalCompanyID(Map params) throws SitecatalystException {
        try {
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + "/discovery/me", params);
            String response = client.request(new HttpGet(uri), analyticsConfig);
            JsonObject jsonResponse = (JsonObject) new JsonParser().parse(response);
            String globalCompanyID = ((JsonObject)((JsonArray)((JsonObject)((JsonArray)jsonResponse.get("imsOrgs")).get(0)).get("companies")).get(0)).get("globalCompanyId").getAsString();
            String CompanyID = ((JsonObject)((JsonArray)((JsonObject)((JsonArray)jsonResponse.get("imsOrgs")).get(0)).get("companies")).get(0)).get("companyName").getAsString();
            JsonObject finalResponse = new JsonObject();
            finalResponse.add("x-proxy-global-company-id", new JsonPrimitive(globalCompanyID));
            finalResponse.add("company", new JsonPrimitive(CompanyID));
            if(LOG.isDebugEnabled())
                LOG.debug(finalResponse.getAsString());
            return finalResponse;
        } catch (Exception e) {
            LOG.error("Unable to get global company ID", e);
            throw new SitecatalystException(e.getMessage(), e);
        }
    }

    @Override
    public String getTrackingServer(Configuration configuration, Map params) throws SitecatalystException {
        try {
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/companies/me/trackingserver", analyticsConfig.getCompanyId()), params);
            String response = client.request(new HttpGet(uri), analyticsConfig);
            JsonObject jsonResponse = (JsonObject) new JsonParser().parse(response);
            JsonObject finalResponse = new JsonObject();
            finalResponse.add("tracking_server", jsonResponse.get("trackingServer"));
            uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/companies/me", analyticsConfig.getCompanyId()), params);
            response = client.request(new HttpGet(uri), analyticsConfig);
            jsonResponse = (JsonObject) new JsonParser().parse(response);
            finalResponse.add("namespace", jsonResponse.get("namespace"));
            if(LOG.isDebugEnabled())
                LOG.debug(finalResponse.getAsString());
            return finalResponse.toString();
        } catch (Exception e) {
            LOG.error("Unable to get tracking server", e);
            throw new SitecatalystException(e.getMessage(), e);
        }
    }

    @Override
    public String getTrafficVars(Configuration configuration, String rsid) throws SitecatalystException {
        try {
            Map<String, String> params = new HashMap<>();
            params.put(SitecatalystServlet.PN_COMPANYID, configuration.getInherited(SitecatalystServlet.PN_COMPANYID,""));
            params.put("rsid", rsid);
            params.put("support", SUPPORT_ENGINE);
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/dimensions", analyticsConfig.getCompanyId()), params);
            String response = client.request(new HttpGet(uri), analyticsConfig);
            String error = getError(response);
            if (error != null) {
                throw new SitecatalystException(error);
            }
            JsonArray jsonResponse = (JsonArray) new JsonParser().parse(response);
            JsonArray finalResponse = new JsonArray();
            JsonObject finalResponseObject = new JsonObject();
            JsonArray jsonArray = new JsonArray();
            for (JsonElement item: jsonResponse){
                if( ((JsonObject)item).has("extraTitleInfo") && "Content".equals(((JsonObject)item).get("category").getAsString())){
                    JsonObject temp = new JsonObject();
                    String id = ((JsonObject)item).get("id").getAsString();
                    temp.add("id",new JsonPrimitive( id.substring(id.indexOf("/")+1)));
                    temp.add("name", ((JsonObject)item).get("name"));
                    temp.add("enabled",new JsonPrimitive(true));
                    jsonArray.add(temp);
                }
            }
            finalResponseObject.add("props", jsonArray);
            finalResponse.add(finalResponseObject);
            if(LOG.isDebugEnabled())
                LOG.debug(finalResponse.getAsString());
            return finalResponse.toString();
        } catch (Exception e) {
            LOG.error("Unable to get traffic vars", e);
            throw new SitecatalystException(e.getMessage(), e);
        }
    }

    @Override
    public String getEvars(Configuration configuration, String rsid) throws SitecatalystException {
        try {
            Map<String, String> params = new HashMap<>();
            params.put(SitecatalystServlet.PN_COMPANYID, configuration.getInherited(SitecatalystServlet.PN_COMPANYID,""));
            params.put("rsid", rsid);
            params.put("support", SUPPORT_ENGINE);
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/dimensions", analyticsConfig.getCompanyId()), params);
            String response = client.request(new HttpGet(uri), analyticsConfig);
            String error = getError(response);
            if (error != null) {
                throw new SitecatalystException(error);
            }
            JsonArray jsonResponse = (JsonArray) new JsonParser().parse(response);
            JsonArray finalResponse = new JsonArray();
            JsonObject finalResponseObject = new JsonObject();
            JsonArray jsonArray = new JsonArray();
            for (JsonElement item: jsonResponse){
                if( ((JsonObject)item).has("extraTitleInfo") && "Conversion".equals(((JsonObject)item).get("category").getAsString())){
                    JsonObject temp = new JsonObject();
                    String id = ((JsonObject)item).get("id").getAsString();
                    temp.add("id",new JsonPrimitive( id.substring(id.indexOf("/")+1) ));
                    temp.add("name", ((JsonObject)item).get("name"));
                    temp.add("enabled", new JsonPrimitive(true));
                    temp.add("type", ((JsonObject)item).get("type"));
                    jsonArray.add(temp);
                }
            }
            finalResponseObject.add("evars", jsonArray);
            finalResponse.add(finalResponseObject);
            if(LOG.isDebugEnabled())
                LOG.debug(finalResponse.getAsString());
            return finalResponse.toString();
        } catch (Exception e) {
            LOG.error("Unable to get evars", e);
            throw new SitecatalystException(e.getMessage(), e);
        }
    }

    @Override
    public String getSuccessEvents(Configuration configuration, String rsid) throws SitecatalystException {
        try {
            Map<String, String> params = new HashMap<>();
            params.put(SitecatalystServlet.PN_COMPANYID, configuration.getInherited(SitecatalystServlet.PN_COMPANYID,""));
            params.put("rsid", rsid);
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/metrics", analyticsConfig.getCompanyId()), params);
            String response = client.request(new HttpGet(uri), analyticsConfig);
            String error = getError(response);
            if (error != null) {
                throw new SitecatalystException(error);
            }
            JsonArray jsonResponse = (JsonArray) new JsonParser().parse(response);
            JsonArray finalResponse = new JsonArray();
            JsonObject finalResponseObject = new JsonObject();
            JsonArray jsonArray = new JsonArray();
            for (JsonElement item: jsonResponse){
                JsonObject temp = new JsonObject();
                String id = ((JsonObject)item).get("id").getAsString();
                temp.add("id",new JsonPrimitive( id.substring(id.indexOf("/")+1)));
                temp.add("name", ((JsonObject)item).get("name"));
                temp.add("type", ((JsonObject)item).get("type"));
                jsonArray.add(temp);
            }
            finalResponseObject.add("events", jsonArray);
            finalResponse.add(finalResponseObject);
            if(LOG.isDebugEnabled())
                LOG.debug(finalResponse.getAsString());
            return finalResponse.toString();
        } catch (Exception e) {
            LOG.error("Unable to get success events", e);
            throw new SitecatalystException(e.getMessage(), e);
        }
    }

    @Override
    public String getMetrics(Configuration configuration, String rsid) throws SitecatalystException {
        try {
            Map<String, String> params = new HashMap<>();
            params.put(SitecatalystServlet.PN_COMPANYID, configuration.getInherited(SitecatalystServlet.PN_COMPANYID,""));
            params.put("rsid", rsid);
            params.put("support", SUPPORT_ENGINE);
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/metrics", analyticsConfig.getCompanyId()), params);
            String response = client.request(new HttpGet(uri), analyticsConfig);
            String error = getError(response);
            if (error != null) {
                throw new SitecatalystException(error);
            }
            if(LOG.isDebugEnabled())
                LOG.debug(response);
            return response;
        } catch (Exception e) {
            LOG.error("Unable to get metrics", e);
            throw new SitecatalystException(e.getMessage(), e);
        }
    }

    @Override
    public String getReport(Configuration configuration, String data) throws Exception {
        try {
            Map<String, String> params = new HashMap<>();
            params.put("x-proxy-global-company-id", configuration.getInherited("x-proxy-global-company-id",""));
            String reportsuite = configuration.getInherited("reportsuite", null);
            params.put("rsid", reportsuite);
            URI uri = buildURI(DEFAULT_ENDPOINT_URI + String.format("/api/%s/reports", analyticsConfig.getCompanyId()), params);
            HttpEntity entity = EntityBuilder.create()
                    .setContentType(ContentType.APPLICATION_JSON)
                    .setContentEncoding("UTF-8")
                    .setText(data)
                    .build();
            HttpPost httpPost = new HttpPost(uri);
            httpPost.setEntity(entity);

            JsonObject jsonData = (JsonObject) new JsonParser().parse(data);
            String response = client.request(httpPost, analyticsConfig);

            String error = getError(response);
            if (error != null) {
                throw new SitecatalystException(error);
            }

            JsonObject finalResponse = getJsonObject(reportsuite, jsonData, response);
            if(LOG.isDebugEnabled())
                LOG.debug(finalResponse.toString());
            return finalResponse.toString();
        } catch (Exception e) {
            LOG.error("Unable to get report", e);
            throw new SitecatalystException(e.getMessage(), e);
        }

    }

    /**
     * This method converts the report response of analytics 2.0 into one required by AEM.
     * @param reportsuite
     * @param jsonData
     * @param response
     * @return converted json
     */
    private JsonObject getJsonObject(String reportsuite, JsonObject jsonData, String response) {
        JsonObject jsonResponse = (JsonObject) new JsonParser().parse(response);
        JsonObject finalResponse = new JsonObject();
        finalResponse.add("reportSuite",  new JsonParser().parse("{\n" +
                "            \"id\": "+ reportsuite +",\n" +
                "            \"name\": "+ reportsuite +"\n" +
                "        }"));
        finalResponse.add("elements",  new JsonParser().parse("[{\n" +
                "            \"id\": "+ jsonData.get("dimension").getAsString().substring(jsonData.get("dimension").getAsString().indexOf("/")+1) +",\n" +
                "            \"name\": "+ jsonData.get("dimension").getAsString().substring(jsonData.get("dimension").getAsString().indexOf("/")+1) +"\n" +
                "        }]"));
        finalResponse.add("metrics", jsonData.get("metricContainer").getAsJsonObject().get("metrics").getAsJsonArray());
        int averagetimespentonsiteIndex = -1;
        for(int i=0; i< finalResponse.get("metrics").getAsJsonArray().size();i++){
            String id = finalResponse.get("metrics").getAsJsonArray().get(i).getAsJsonObject().get("id").getAsString();
            id = id.substring(id.indexOf("/") + 1);
            if("averagetimespentonsite".equals(id)){
                id = "averagetimespentonpage";
                averagetimespentonsiteIndex = i;
            }
            finalResponse.get("metrics").getAsJsonArray().get(i).getAsJsonObject().add("id",new JsonPrimitive(id));
        }
        finalResponse.add("totals", jsonResponse.get("summaryData").getAsJsonObject().get("totals").getAsJsonArray());
        JsonArray rowData = new JsonArray();
        for(JsonElement o : jsonResponse.get("rows").getAsJsonArray())
        {
            JsonObject temp = o.getAsJsonObject();
            temp.add("name", temp.get("value"));
            JsonArray countData =  temp.getAsJsonArray("data");
            if(averagetimespentonsiteIndex != -1)
                countData.set(averagetimespentonsiteIndex, new JsonPrimitive(countData.get(averagetimespentonsiteIndex).getAsDouble()/60.0));
            temp.add("counts", countData);
            rowData.add(temp);
        }
        finalResponse.add("data", rowData);
        finalResponse.add( "type", new JsonPrimitive("ranked"));
        return finalResponse;
    }
}
