/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2018 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.commons.util;

import com.day.cq.dam.api.Asset;
import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;

/**
 * This class provides various utility methods pertaining to Adobe Stock integration required
 * in presentation/user interface.
 */
public class StockUtil {

    public static final String STOCK_ROOT_PATH = "/libs/dam/stock/asset";

    public static final String STOCK_STATE_UNLICENSED = "Unlicensed";

    public static final String PN_STOCK_ID = "stock:id";

    public static final String PN_STOCK_LICENSE = "stock:license";

    public static final String PN_STOCK_IS_EDITORIAL = "stock:isEditorial";

    /**
     * Returns true if resource is an asset imported from Adobe Stock
     *
     * @param res a resource
     * @return returns true if resource is identified as Stock asset
     */
    public static boolean isStockAsset(Resource res) {
        if (res == null) {
            return false;
        }
        Asset asset = res.adaptTo(Asset.class);
        return asset != null && StringUtils.isNotEmpty(asset.getMetadataValue(PN_STOCK_ID));
    }

    /**
     * Returns true if resource is a licensed asset imported from Adobe Stock
     *
     * @param res a resource
     * @return returns true if resource is identified as licensed Stock asset
     */
    public static boolean isStockAssetLicensed(Resource res) {
        if (isStockAsset(res)) {
            String license = res.adaptTo(Asset.class).getMetadataValue(PN_STOCK_LICENSE);
            return StringUtils.isNotEmpty(license) && !license.equals(STOCK_STATE_UNLICENSED);
        }
        return false;
    }

    /**
     * Returns true if resource is a Adobe Stock Asset and isEditorial property is true
     *
     * @param res a resource
     * @return returns true if resource is identified as editorial Stock asset
     */
    public static boolean isStockAssetEditorial(Resource res) {
        if (isStockAsset(res)) {
            String isEditorialValue = res.adaptTo(Asset.class).getMetadataValue(PN_STOCK_IS_EDITORIAL);
            return Boolean.valueOf(isEditorialValue);
        }
        return false;
    }    

    /**
     * Returns true if Stock API is accessible by the current context
     *
     * @param resourceResolver resource resolver
     * @return returns true if stock API is accessible
     */
    public static boolean isStockAccessible(ResourceResolver resourceResolver) {
        return !ResourceUtil.isNonExistingResource(resourceResolver.resolve(STOCK_ROOT_PATH));
    }

}
