/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.search.eval;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.query.Row;

import org.apache.jackrabbit.JcrConstants;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.resource.collection.ResourceCollection;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.propertytypes.ServiceVendor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.granite.toggle.api.ToggleRouter;
import com.day.cq.search.Predicate;

/**
 * Finds items that are member of a specific
 * <a href="https://docs.adobe.com/docs/en/aem/6-2/develop/ref/javadoc/org/apache/sling/resource/collection/ResourceCollection.html">
 * sling resource collection</a>.
 *
 * <p>
 * This is a filtering-only predicate and cannot leverage a search index.
 * Does not support facet extraction.
 *
 * <h3>Name:</h3>
 * memberOf
 *
 * <h3>Properties:</h3>
 * <dl>
 * <dt>memberOf</dt>
 * <dd>path of sling resource collection</dd>
 * </dl>
 *
 * @since 6.0
 */
@ServiceVendor("Adobe Systems Incorporated")
@Component(factory = "com.day.cq.search.eval.PredicateEvaluator/memberOf")
public class CollectionPredicateEvaluator extends AbstractPredicateEvaluator {
    private static final Logger LOG = LoggerFactory.getLogger(CollectionPredicateEvaluator.class);

    public static final String COLLECTION = "memberOf";

    private static final String COLLECTION_PATH_SET_KEY = "collectionPathSet";
    private static final String COLLECTION_ID_SET_KEY = "collectionIdSet";
    private static final String COLLECTION_ID_SET_FAILED_KEY = "collectionIdSetFailed";
    private static final String COLLECTION_FOLDER_SET_KEY = "collectionFolderSet";

    static final String TOGGLE_USE_QUERY = "CT_ASSETS-32899";

    @Reference
    private ToggleRouter toggleRouter;

    @Override
    public boolean includes(Predicate p, Row row, EvaluationContext context) {
        if (!p.hasNonEmptyValue(COLLECTION)) {
            return true;
        }
        // Set of paths obtained from sling:members property
        Set<String> collectionPathSet = (Set<String>)context.get(COLLECTION_PATH_SET_KEY);
        // Set of path of members which are folders
        Set<String> collectionFolderSet = (Set<String>)context.get(COLLECTION_FOLDER_SET_KEY);
        if (collectionPathSet == null) {
            collectionPathSet = Collections.<String>emptySet();
            collectionFolderSet = Collections.<String>emptySet();

            ResourceCollection collection = getCollection(p, context);
            if (collection != null) {
                Iterator<Resource> collectionItr = collection.getResources();
                collectionPathSet = new HashSet<String>();
                collectionFolderSet = new HashSet<String>();
                while(collectionItr.hasNext()) {
                    Resource collectionMember = collectionItr.next();
                    collectionPathSet.add(collectionMember.getPath());
                    if (isFolder(collectionMember)) {
                        collectionFolderSet.add(collectionMember.getPath() + "/");
                    }
                }
            }
            // Put the set of paths in context so that it can be reused subsequently.
            context.put(COLLECTION_PATH_SET_KEY, collectionPathSet);
            context.put(COLLECTION_FOLDER_SET_KEY, collectionFolderSet);
        }

        // returns true if either it is a direct member or is member of a folder which is a direct member
        return collectionPathSet.contains(context.getPath(row)) || isFolderMember(collectionFolderSet, context.getPath(row));
    }

    public String getXPathExpression(Predicate predicate, EvaluationContext context) {

        if(!canXpath(predicate, context)) {
            return null;
        }

        StringBuffer answer = new StringBuffer("(");

        Set<String> itemIds = getCollectionItemIds(getCollection(predicate, context), context);
        if(itemIds.isEmpty()) {
            answer.append("false");
        } else {
            Iterator<String> idIter = itemIds.iterator();
            while (idIter.hasNext()) {
                answer.append(JcrConstants.JCR_UUID).append("='").append(idIter.next()).append("'");
                if(idIter.hasNext()) {
                    answer.append(" or ");
                }
            }
        }

        answer.append(")");
        LOG.debug("getXPathExpression: returning expression '{}'", answer.toString());

        return answer.toString();
    }


    private ResourceCollection getCollection(Predicate p, EvaluationContext context) {
        Resource colRes = context.getResourceResolver().getResource(p.get(COLLECTION));
        if(colRes != null) {
            return colRes.adaptTo(ResourceCollection.class);
        }
        return null;
    }


    @Override
    public boolean canXpath(Predicate predicate, EvaluationContext context) {
        boolean answer = false;

        if(useIndexedQuery()) {
            ResourceCollection collection = getCollection(predicate, context);
            if (collection!=null) {
                // If we are able to resolve a set of item uuids, we can use
                // the query-based optimisation.
                Set<String> itemIds = getCollectionItemIds(collection, context);
                answer = itemIds!=null;
            }
        }

        LOG.trace("canXpath: returning {}", answer);
        return answer;
    }

    @SuppressWarnings("unchecked")
    private Set<String> getCollectionItemIds(ResourceCollection collection, EvaluationContext context){

        // Check for previously loaded ID set in the context and if found, use those
        Set<String> itemIds = (Set<String>) context.get(COLLECTION_ID_SET_KEY);
        if (itemIds != null) {
            return itemIds;
        } else if (context.get(COLLECTION_ID_SET_FAILED_KEY) != null) {
            return null;
        }

        // Populate the item ID set by reading collection items and store in the context
        // for use in filtering the query.
        //
        // If...
        // * there are >1000 Assets in the collection or
        // * any Asset is without UUID
        //
        // ...we abort this (and store marker), as we cannot take advantage of the query-based optimisation.
        itemIds = new HashSet<>();
        Iterator<Resource> collectionItr = collection.getResources();
        LOG.debug("getCollectionItemIds: reading item ids for collection '{}' ({})", collection.getName(), collection.getPath());

        if(readItemIds(collectionItr, itemIds, context)){
            context.put(COLLECTION_ID_SET_KEY, itemIds);
            LOG.debug("getCollectionItemIds: identified {} item ids for collection", itemIds.size());
            return itemIds;
        } else {
            context.put(COLLECTION_ID_SET_FAILED_KEY, Boolean.TRUE);
            return null; // Return null if we have failed to extract collection items
        }
    }

    private boolean readItemIds(Iterator<Resource> items, Set<String> itemIds, EvaluationContext context) {

        try {
            while(items.hasNext()) {
                if(itemIds.size() > 1000) {
                    LOG.warn("readItemIds: Collection has >1000 members - unable to include in query");
                    return false;
                }

                Resource itemResource = items.next();
                if(itemResource.isResourceType("dam:Asset")) {
                    String uuid = getUUID(itemResource);
                    if(uuid != null) {
                        itemIds.add(uuid);
                    } else {
                        if(LOG.isDebugEnabled()) {
                            LOG.debug("readItemIds: Seen asset member without uuid at '{}'", itemResource.getPath());
                        }
                        LOG.warn("readItemIds: Collection contains asset member without uuid - unable to include in query");

                        return false;
                    }
                } else {
                    Iterator<Resource> nestedItems = null;

                    // If we encounter a nested folder or collection, recurse into those
                    ResourceCollection nestedCollection = itemResource.adaptTo(ResourceCollection.class);
                    if(nestedCollection != null) {
                        nestedItems = nestedCollection.getResources();
                    } else if (isFolder(itemResource)) {
                        nestedItems = itemResource.listChildren();
                    }
                    if(nestedItems != null && !readItemIds(nestedItems, itemIds, context)) {
                        return false;
                    }
                }
            }
        } catch (Exception e) {
            // Ensure an issue reading collection items does not break the search
            LOG.warn("readItemIds: unable to read item ids - " + e.getMessage(), e);
            return false;
        }

        return true;
    }

    @Override
    public boolean canFilter(Predicate predicate, EvaluationContext context) {
        return true;
    }

    private boolean isFolder(Resource resource){

        // Ensure we ignore the jcr:content node
        if(JcrConstants.JCR_CONTENT.equals(resource.getName())) {
            return false;
        }

        Node n = resource.adaptTo(Node.class);
        try {
            return n.isNodeType(JcrConstants.NT_FOLDER);
        } catch (RepositoryException e) {
            return false;
        }
    }

    private boolean isFolderMember(Set<String> collectionFolderSet, String path) {
        for (String folderPath : collectionFolderSet) {
            if(path.indexOf(folderPath) == 0) {
                return true;
            }
        }
        return false;
    }

    private boolean useIndexedQuery() {
        return toggleRouter != null && toggleRouter.isEnabled(TOGGLE_USE_QUERY);
    }

    private String getUUID(Resource assetResource) {
        ValueMap vm = assetResource.adaptTo(ValueMap.class);
        if(vm.containsKey(JcrConstants.JCR_UUID)) {
            return vm.get(JcrConstants.JCR_UUID, String.class);
        } else {
            return null;
        }
    }
}
