/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.tags;

import javax.servlet.jsp.tagext.TagSupport;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;
import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.net.URLEncoder;
import java.io.IOException;

/**
 * <code>RequestURLTag</code> writes the current request URL to the current
 * JspWriter. The body of this tag allows multiple addParam and removeParam
 * tags to modify the current request URL before it is written.
 */
public class RequestURLTag extends TagSupport {

    private static final long serialVersionUID = -5674803762870190034L;

    /**
     * Maps query parameter name to a list of values.
     */
    private Map<String, List<String>> parameters = new HashMap<String, List<String>>();

    /**
     * The request URI. Contains the web app context and the path, but no query
     * parameters. See {@link HttpServletRequest#getRequestURI()}.
     */
    private String uri;

    /**
     * Default constructor.
     */
    public RequestURLTag() {
        super();
        init();
    }

    /**
     * Removes the parameter and all its values with the given <code>name</code>.
     *
     * @param name the name of the parameter.
     */
    public void removeParam(String name) {
        parameters.remove(name);
    }

    /**
     * Removes a value from the parameter with the given <code>name</code>.
     *
     * @param name name of the parameter.
     * @param value the value to remove.
     */
    public void removeParam(String name, String value) {
        List<String> values = parameters.get(name);
        if (values != null) {
            values.remove(value);
            if (values.size() == 0) {
                parameters.remove(name);
            }
        }
    }

    /**
     * Adds a parameter with the given <code>name</code> and <code>value</code>.
     *
     * @param name the name of the parameter.
     * @param value the value for the parameter.
     */
    public void addParam(String name, String value) {
        List<String> values = parameters.get(name);
        if (values == null) {
            values = new ArrayList<String>();
            parameters.put(name, values);
        }
        values.add(value);
    }

    /**
     * {@inheritDoc}
     */
    public void release() {
        init();
        super.release();
    }

    /**
     * {@inheritDoc}
     */
    public int doStartTag() throws JspException {
        HttpServletRequest request = (HttpServletRequest) pageContext.getRequest();
        uri = request.getRequestURI();
        for (Object o : request.getParameterMap().entrySet()) {
            Map.Entry entry = (Map.Entry) o;
            List<String> values = new ArrayList<String>();
            values.addAll(Arrays.asList((String[]) entry.getValue()));
            parameters.put((String) entry.getKey(), values);
        }
        return EVAL_BODY_INCLUDE;
    }

    /**
     * {@inheritDoc}
     */
    public int doEndTag() throws JspException {
        JspWriter out = pageContext.getOut();
        try {
            out.print(uri);
            String separator = "?";
            for (Map.Entry<String, List<String>> entry : parameters.entrySet()) {
                String name = entry.getKey();
                List<String> values = entry.getValue();
                for (String value : values) {
                    out.print(separator);
                    separator = "&";
                    out.print(URLEncoder.encode(name, "UTF-8"));
                    out.print("=");
                    out.print(URLEncoder.encode(value, "UTF-8"));
                }
            }
            return super.doEndTag();
        } catch (IOException e) {
            throw new JspException(e);
        } finally {
            // reset
            init();
        }
    }

    private void init() {
        parameters.clear();
        uri = null;
    }
}
