/*************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.util;

import java.util.Date;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The <code>Timing</code> class supports timing code fragments. A timer has a
 * name, which may be used for informational purposes, and a start time against
 * which the elapsed time is calculated.
 * <p>
 * The start time is first set to the time of instance creation and may later
 * be set to the current time using the {@link #reset} method. The name of the
 * timer is also set at instance creation time and may be changed using the
 * {@link #setName} method.
 * <p>
 * The time elapsed since the last timer reset (either instance creation or
 * call to the {@link #reset} method) is accessible through the
 * {@link #getElapsed} and may be logged using the {@link #stamp} method.
 *
 * @version $Revision: 1.2 $
 * @author fmeschbe
 * @since gumbaer
 */
public class Timing {

    /** The defualt name for instances */
    public static final String DEFAULT_NAME = "Timing";

    /** The name of the logger used to log timing information */
    private static final String TIMING_LOGGER = "misc.timing";

    /** logger for timing information */
    private static final Logger timeLog = LoggerFactory.getLogger(TIMING_LOGGER);
    
    /** The name of this instance. This is for informational purpposes. */
    private String name;

    /** The time at which this instances was started */
    private long startTime;

    /**
     * Creates an instance with the {@link #DEFAULT_NAME default name}. The
     * start time of the new instance is set to the time of instance creation.
     * <p>
     * Calling this method is equivalent to calling
     * {@link #Timing(java.lang.String) new Timing(null)}
     *
     * @see #DEFAULT_NAME
     */
    public Timing() {
        this(null);
    }

    /**
     * Creates an instance with the given name. The start time of the new
     * instance is set to the time of instance creation.
     *
     * @param name The name of the new instance. If the name is empty or
     *      <code>null</code>, the name will be set to the
     *      {@link #DEFAULT_NAME default name}.
     */
    public Timing(String name) {
        reset();
        setName(name);
    }

    //---------- Property access -----------------------------------------------

    /**
     * Returns the name of this instance.
     *
     * @return The name of this instance.
     */
    public String getName() {
        return name;
    }

    /**
     * Sets the name of this instance.
     *
     * @param name The new name to set on this instance. If the name is empty or
     *      <code>null</code>, the name will be set to the
     *      {@link #DEFAULT_NAME default name}.
     */
    public void setName(String name) {
        this.name = (name == null || name.length() == 0) ? DEFAULT_NAME : name;
    }

    /**
     * Returns the starting time of this instance.
     * @return the start time
     * @see #reset
     */
    public long getStartTime() {
        return startTime;
    }

    //---------- Utility methods -----------------------------------------------

    /**
     * Resets the starting time of this instance to the current system time.
     * @see #getStartTime
     */
    public void reset() {
        startTime = System.currentTimeMillis();
    }

    /**
     * Returns the time in milliseconds elapsed since the starting time.
     * @return the elapsed time
     * @see #reset
     * @see #getStartTime
     */
    public long getElapsed() {
        return System.currentTimeMillis() - startTime;
    }

    /**
     * Writes the elapsed time to the timer log. The message contains the
     * timer name, the supplied message and the elapsed time in milliseconds.
     * <p>
     * The log message is logged as an informational message at level
     * <code>INFO</code> to the logger <code>misc.timing</code>.
     *
     * @param message The message to be added to the log message.
     */
    public void stamp(String message) {
        timeLog.info("{}: {} - {}ms", new String[] { name, message,
            String.valueOf(getElapsed()) });
    }

    //---------- Object overwrites ---------------------------------------------

    /**
     * Returns a string representation of this instance, which consists of
     * the name and human readable form of the start time.
     *
     * @return A string represenation of this instance.
     */
    public String toString() {
        return "Timing: name=" + name + ", startTime=" + new Date(startTime);
    }
}
