/*
 * File: TTInterpreter.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

/*
 * Based on code with the following copyright notices:
 * 
 *  Copyright (c) 1987-1990, 1992 by Apple Computer, Inc., all rights reserved.
 *  Copyright (c) 1989-2002. Microsoft Corporation, all rights reserved.
 */

package com.adobe.fontengine.font.opentype;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.Matrix;
import com.adobe.fontengine.font.ScalerDebugger;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.math.F16Dot16;
import com.adobe.fontengine.math.F16Dot16Vector;
import com.adobe.fontengine.math.F26Dot6;
import com.adobe.fontengine.math.F26Dot6Vector;
import com.adobe.fontengine.math.F2Dot14;
import com.adobe.fontengine.math.F2Dot14Vector;

/** Interprets TT programs.
 */

public class TTInterpreter {

  //----------------------------------------------------------------------------  
  
  private final class LocalGraphicState implements Cloneable {
    private boolean autoFlip;
    private int angleWeight;
    private int /*26.6*/ minimumDistance;
    private int /*26.6*/ controlValueCutIn;
    private int singleWidthCutIn;
    private int singleWidth;
    private int deltaBase;
    private int deltaShift;
    private int scanControl;
    private int instrControl;
    private int roundMode;
    private int roundPeriod;
    private int roundPhase;
    private int roundThreshold;
    
    public Object clone () throws CloneNotSupportedException {
      return super.clone ();
    }
  }
  
  private final class GraphicState {

    public GraphicState (int cvtSize, OTByteArray cvt) throws InvalidFontException {
      this.cvtStretch = new F16Dot16Vector ();
      this.cvtStretch.x = F16Dot16.ONE;
      this.cvtStretch.y = F16Dot16.ONE;
      this.controlValueTable = new ControlValueTable (cvtSize, cvt); 
      
      this.localGraphicState = new LocalGraphicState ();
    }
    
    LocalGraphicState defaultGraphicState;
    LocalGraphicState localGraphicState;

    //--------------------------------------------------------------------------

    boolean getAutoFlip () {
      return localGraphicState.autoFlip;
    }   
    void setAutoFlip (boolean v) {
      localGraphicState.autoFlip = v;
    }
    
    int getAngleWeight () {
      return localGraphicState.angleWeight; 
    }
    void setAngleWeight (int v) {
      localGraphicState.angleWeight = v;
    }
    
    int /*26.6*/ getMinimumDistance () {
      return localGraphicState.minimumDistance; 
    }
    void setMinimumDistance (int /*26.6*/ v) {
      localGraphicState.minimumDistance = v;
    }
    
    int /*26.6*/ getControlValueCutIn () {
      return localGraphicState.controlValueCutIn;
    }
    void setControlValueCutIn (int /*26.6*/ v) {
      localGraphicState.controlValueCutIn = v;
    }

    int snapToSingleWidth (int distance) {
      if (   localGraphicState.singleWidth - localGraphicState.singleWidthCutIn < distance 
          && distance < localGraphicState.singleWidth + localGraphicState.singleWidthCutIn) {
        return localGraphicState.singleWidth; }
      else if (   - (localGraphicState.singleWidth + localGraphicState.singleWidthCutIn) < distance 
               && distance < - (localGraphicState.singleWidth - localGraphicState.singleWidthCutIn)) {
        return -localGraphicState.singleWidth; }
      else {
        return distance; }
    }
    void setSingleWidthCutIn (int v) {
      localGraphicState.singleWidthCutIn = v;
    }
    void setSingleWidth (int v) {
      localGraphicState.singleWidth = v;
    }

    int getDeltaBase () {
      return localGraphicState.deltaBase;
    }
    void setDeltaBase (int v) {
      localGraphicState.deltaBase = v;
    }
    
    int getDeltaShift () {
      return localGraphicState.deltaShift; 
    }
    void setDeltaShift (int v) {
      localGraphicState.deltaShift = v;
    }
    
    void setScanControl (int v) {
      localGraphicState.scanControl = v;
    }
    int getScanControl () {
      return localGraphicState.scanControl;
    }
    
    void setInstrControl (int v) {
      localGraphicState.instrControl = v;
    }
    int getInstrControl () {
      return localGraphicState.instrControl; 
    }
    
    //------------------------------------------------------------- rounding ---
    void setRoundMode (int v) {
      localGraphicState.roundMode = v; 
    }
    
    void setRounding (int period, int phase, int threshold, int mode) {
      localGraphicState.roundPeriod = period;
      localGraphicState.roundPhase = phase;
      localGraphicState.roundThreshold = threshold;
      localGraphicState.roundMode = mode;
    }
     
    
    int round (int vIn, int engineCompensation) {
      return round (vIn, engineCompensation, localGraphicState.roundMode);
    }

    int roundOff (int vIn, int engineCompenstation) {
      return round (vIn, engineCompenstation, ROUND_OFF);
    }
    
    int roundToGrid (int vIn, int engineCompensation) {
      return round (vIn, engineCompensation, ROUND_TO_GRID);
    }
    
    
    final int ROUND_TO_DOUBLE_GRID = 0;
    final int ROUND_TO_GRID = 1;
    final int ROUND_TO_HALF_GRID = 2;
    final int ROUND_OFF = 3;
    final int ROUND_DOWN_TO_GRID = 4;
    final int ROUND_UP_TO_GRID = 5;
    final int SUPER_ROUND = 6;
    final int SUPER_45_ROUND = 7;      
    
    int round (int vIn, int engineCompensation, int roundMode) {
      int v;
      boolean negate;
      
      if (vIn < 0) {
        v = -vIn;
        negate = true; }
      else {
        v = vIn;
        negate = false; }
      
      v += engineCompensation;
            
      switch (roundMode) {
        case ROUND_TO_DOUBLE_GRID: {
          v += F26Dot6.ONE / 4;
          v &=  ~ (F26Dot6.ONE / 2 - 1);
          break; }
          
        case ROUND_TO_GRID: {
          v += F26Dot6.ONE / 2;
          v &= ~ (F26Dot6.ONE - 1);
          break; }
        
        case ROUND_TO_HALF_GRID: {
          v &= ~ (F26Dot6.ONE - 1);
          v += F26Dot6.ONE / 2;
          break;}

        case ROUND_OFF: {
          break; }
        
        case ROUND_DOWN_TO_GRID: {
          v &= ~ (F26Dot6.ONE - 1);
          break; }
        
        case ROUND_UP_TO_GRID: {
          v += F26Dot6.ONE - 1;
          v &= ~ (F26Dot6.ONE - 1);
          break; }
        
        case SUPER_ROUND: {
          v += localGraphicState.roundThreshold - localGraphicState.roundPhase;
          v &= ~ (localGraphicState.roundPeriod - 1);
          v += localGraphicState.roundPhase;
          break; }
        
        case SUPER_45_ROUND: {
          v = F26Dot6.divideByF2Dot14 (v, localGraphicState.roundPeriod);
          v = F26Dot6.truncate (v);
          v = F26Dot6.multiplyByF2Dot14 (v, localGraphicState.roundPeriod);
          v += localGraphicState.roundPhase;
          break; }}
      
      if (negate) {
        v = -v; }
      
      if (! F26Dot6.sameSign (v, vIn) && vIn != 0) {
        v = 0; }
      
      return v;
    }
        
    //-------------------------------------------------------------- vectors ---
   
    private F2Dot14Vector projectionVector = new F2Dot14Vector (0, 0);
    private F2Dot14Vector freedomVector = new F2Dot14Vector (0, 0);
    private F2Dot14Vector originalProjectionVector = new F2Dot14Vector (0, 0);
    private int /*2.14*/ fDotP;
       
    void setVector (F2Dot14Vector v, F26Dot6Vector p1, F26Dot6Vector p2, boolean rotate) {
      int /*26.6*/ x = p2.x - p1.x;
      int /*26.6*/ y = p2.y - p1.y;
      
      if (x == 0 && y == 0) {
        setVector (v, F2Dot14.ONE, F2Dot14.ZERO, rotate); }

      else {
        double xx = F26Dot6.toDouble (x);
        double yy = F26Dot6.toDouble (y);
        double length = Math.sqrt (xx * xx + yy * yy);
        setVector (v, F2Dot14.fromDouble (xx / length),
                      F2Dot14.fromDouble (yy / length),
                   rotate); }
    }
      
    void setVector (F2Dot14Vector v, int /*2.14*/ x, int /*2.14*/ y, boolean rotate) {
      if (rotate) {
        v.x = -y;
        v.y = x; }
      else {
        v.x = x;
        v.y = y; }
      
      if (v == projectionVector || v == freedomVector) {
        fDotP = F2Dot14.multiply (projectionVector.x, freedomVector.x)
              + F2Dot14.multiply (projectionVector.y, freedomVector.y);
        if (-F2Dot14.ONE_SIXTEENTH < fDotP && fDotP < F2Dot14.ONE_SIXTEENTH) {
          fDotP = (fDotP < 0) ? - F2Dot14.ONE : F2Dot14.ONE; }}
    }
    
    // Return the projection along a vector 
    int /*26.6*/ project (F2Dot14Vector v, int /*26.6*/ x, int /*26.6*/ y) {
      return F26Dot6.multiplyByF2Dot14 (x, v.x) 
           + F26Dot6.multiplyByF2Dot14 (y, v.y);
    }
    
    int /*f26.6*/ project (F2Dot14Vector v, F26Dot6Vector p) {
      return project (v, p.x, p.y);
    }
    
    int /*f26.6*/ project (F2Dot14Vector v, F26Dot6Vector p1, F26Dot6Vector p2) {
      return project (v, p2.x - p1.x, p2.y - p1.y);
    }
    
    //------------------------------------------------------------------ CVT ---
    private F16Dot16Vector cvtStretch;
    private ControlValueTable controlValueTable;
    private int /*f16.16*/ cvtxN, cvtyN;
    private int /*f16.16*/ cvtD;
    
    void resetAndScaleCVT () {
      controlValueTable.resetAndScale (cvtxN, cvtD);
    }
    
    int /*f16.6*/ getCVTscale () {
      int /*f16.16*/ z
        = F16Dot16.multiplyByF2Dot14 (F16Dot16.square (cvtStretch.x),
                                      F2Dot14.square (projectionVector.x))
        + F16Dot16.multiplyByF2Dot14 (F16Dot16.square (cvtStretch.y),
                                      F2Dot14.square (projectionVector.y));
      
      if (z >= F16Dot16.ONE) {
        return F16Dot16.ONE; }
      else {
        return F16Dot16.fromDouble (Math.sqrt (F16Dot16.toDouble (z))); }
    }

    int /*f26.6*/ getCVT (int cvtIndex) throws InvalidGlyphException {
      return F26Dot6.multiplyByF16Dot16 (controlValueTable.get (cvtIndex), getCVTscale ());
    }
    
    void putCVT (int cvtIndex, int /*f26.6*/ value)  throws InvalidGlyphException {
      controlValueTable.put (cvtIndex, F26Dot6.divideByF16Dot16 (value, getCVTscale ()));
    }
    
    void putCVTPixels (int cvtIndex, int value) throws InvalidGlyphException {
      controlValueTable.put (cvtIndex, F16Dot16.multiplyDivide (F26Dot6.fromDouble (value), cvtxN, cvtD));
    }
    
    void incrementCVT (int cvtIndex, int /*f26.6*/ value) throws InvalidGlyphException {
      controlValueTable.put (cvtIndex, 
          controlValueTable.get (cvtIndex) 
          + F26Dot6.divideByF16Dot16 (value, getCVTscale ()));
    }
    
    //--------------------------------------------------------------------------
    private double pixelsPerEm;
    int getPixelsPerEm () {
      return F26Dot6.toInt (F26Dot6.multiplyByF16Dot16 (F26Dot6.fromDouble (pixelsPerEm), 
          getCVTscale ()));
    }
    void setPixelsPerEm (double pixelsPerEm) {
      this.pixelsPerEm = pixelsPerEm;
    }
    
    private double pointSize;
    int /*f26.6*/ getPointSize () {
      return F26Dot6.fromDouble (pointSize);
    }
    void setPointSize (double pointSize) {
      this.pointSize = pointSize;
    }
    
    //------------------------------------------------- operations on points ---
    void move (TTPoint p, int /*26.6*/ delta) {
      if (ScalerDebugger.debugOn && debugger != null) {
        logDetailNoLn ("--- moving by " + delta + " (" + F26Dot6.toDouble (delta) + ") from " + p.toString () + " to "); }
      
      if (fDotP != F2Dot14.ONE) {
          if (freedomVector.x != 0) {
          if (fDotP == freedomVector.x) {
            p.hinted.x += delta; }
          else {
            p.hinted.x += F26Dot6.multiplyByF2Dot14DivideByF2Dot14 (delta, freedomVector.x, fDotP); }
          p.touchedX = true; }
        
        if (freedomVector.y != 0) {
          if (fDotP == freedomVector.y) {
            p.hinted.y += delta; }
          else {
            p.hinted.y += F26Dot6.multiplyByF2Dot14DivideByF2Dot14 (delta, freedomVector.y, fDotP); }
          p.touchedY = true; }}
      
      else {
        if (freedomVector.x == F2Dot14.ONE) {
          p.hinted.x += delta;
          p.touchedX = true; }
        else if (freedomVector.y == F2Dot14.ONE) {
          p.hinted.y += delta;
          p.touchedY = true; }
        else {
          if (freedomVector.x != 0) {
            p.hinted.x += F26Dot6.multiplyByF2Dot14 (delta, freedomVector.x);
            p.touchedX = true; }
          if (freedomVector.y != 0) {
            p.hinted.y += F26Dot6.multiplyByF2Dot14 (delta, freedomVector.y);
            p.touchedY = true; }}}
      
      if (ScalerDebugger.debugOn && debugger != null) {
        logDetail (p.hinted.toString()); }
    }
      
    void move (TTPoint p, int deltaX, int deltaY) {    
      if (ScalerDebugger.debugOn && debugger != null) {
        logDetailNoLn ("--- moving by " + deltaX + ", " + deltaY 
            + " (" + F26Dot6.toDouble (deltaX) + ", " + F26Dot6.toDouble (deltaY) 
            + ") from " + p.toString () + " to "); }
      
      if (freedomVector.x != 0) {
        p.hinted.x += deltaX;
        p.touchedX = true; }
      if (freedomVector.y != 0) {
        p.hinted.y += deltaY;
        p.touchedY = true; }
      
      if (ScalerDebugger.debugOn && debugger != null) {
        logDetail (p.hinted.toString()); }
    }
    
    void set (TTPoint p, int x, int y) {
      p.hinted.x = x;
      p.touchedX = true;
      p.hinted.y = y;
      p.touchedY = true;
    }
    
    void untouch (TTPoint p) {
      if (freedomVector.x != 0) {
        p.touchedX = false; }
      if (freedomVector.x != 0) {
        p.touchedY = false; }
    }
    
    //----------------------------------------------------- reference points ---
    private int rp0;
    void setRp0 (int p) {
      rp0 = p;
    }
    int getRp0 () {
      return rp0;
    }
    private int rp1;
    void setRp1 (int p) {
      rp1 = p;
    }
    int getRp1 () {
      return rp1;
    }
    private int rp2;
    void setRp2 (int p) {
      rp2 = p;
    }
    int getRp2 () {
      return rp2;
    }

    private int zp0;
    void setZp0 (int zone) {
      zp0 = zone;
    }
    int getZp0 () {
      return zp0;
    }
    private int zp1;
    void setZp1 (int zone) {
      zp1 = zone;
    }
    int getZp1 () {
      return zp1;
    }
    private int zp2;
    void setZp2 (int zone) {
      zp2 = zone;
    }
    int getZp2 () {
      return zp2;
    }
    
    //----------------------------------------------------------------------
    private int loopVariable;
    void setLoopVariable (int v) {
      loopVariable = v;
    }
    int getAndResetLoopVariable () {
      int result = loopVariable;
      loopVariable = 1;
      return result;
    }    
  }
  
  //----------------------------------------------------------------------------  
  private final class GraphicParameters {
    int getEngineCompensation (int distanceType) {
      return 0; // TODO
    }
  }
    
  //----------------------------------------------------------------------------  
  private final class StorageArea {
    private final int[] data;
    
    public StorageArea (int size) {
      data = new int [size];
    }
    
    public int getuint32 (int location) throws InvalidGlyphException {
      if (location < 0 || data.length <= location) {
        throw new InvalidGlyphException ("attempting to access TT storage area location " + location + " (must be in [" + 0 + ", " + data.length + "[)"); }
      return data [location]; 
    }
    
    public void setuint32 (int location, int value) throws InvalidGlyphException {
      if (location < 0 || data.length <= location) {
        throw new InvalidGlyphException ("attempting to access TT storage area location " + location + " (must be in [" + 0 + ", " + data.length + "[)"); }
      data [location] = value;
    }
  }
  
  //----------------------------------------------------------------------------  
  private final class Stack {
    private final int [] data;
    private int currentDepth;
    
    public Stack (int maxSize) {
      this.data = new int [maxSize];
      this.currentDepth = 0;
    }
    
    public int pop () throws InvalidGlyphException {
      if (currentDepth == 0) {
        throw new InvalidGlyphException ("pop on empty TT stack"); }
      currentDepth--;
      return data [currentDepth];
    }
    
    public void push (int value) throws InvalidGlyphException {
      if (currentDepth > data.length) {
        throw new InvalidGlyphException ("push on full TT stack"); }
      data [currentDepth] = value;
      currentDepth++;
    }
    
    public int peek (int depth) {
      if (currentDepth - depth < 0) {
        System.err.println ("*** peek below stack bottom"); }
      return data [currentDepth - depth - 1];
    }
    
    public void clear () {
      currentDepth = 0;
    }
    
    public void copy (int n) throws InvalidGlyphException {
      if (n > currentDepth) {
        throw new InvalidGlyphException ("index of non-existing TT stack element (" + n + ", currentDepth=" + currentDepth + ")"); }
      push (data [currentDepth - n]);

    }
    
    public void pushDepth () throws InvalidGlyphException {
      push (currentDepth);
    }
    
    public void bubbleUp (int n) throws InvalidGlyphException {
      if (n > currentDepth) {
        throw new InvalidGlyphException ("index of non-existing TT stack element (" + n + ", currentDepth=" + currentDepth + ")"); }
      int v = data [currentDepth - n];
      for (int i = currentDepth - n; i < currentDepth - 1; i++) {
        data [i] = data [i+1]; }
      data [currentDepth - 1] = v;
    }
  }
  
  //----------------------------------------------------------------------------  
  private final class ControlValueTable {
    private int[] originalData;
    private int/*26.6*/[] data;
    
    ControlValueTable (int nbEntries, OTByteArray cvt) throws InvalidFontException {
      if (nbEntries < 0 || 200000 <= nbEntries) {
        throw new InvalidFontException ("attempt to CVT table with "
            + nbEntries + " entries"); }
      originalData = new int [nbEntries]; 
      for (int i = 0; i < nbEntries; i++) {
        originalData [i] = cvt.getint16 (2*i); }
      data = new int [nbEntries];
    }
    
    void resetAndScale (int /*16.16*/ N, int /*16.16*/ D) {
      for (int i = 0; i < originalData.length; i++) {
//        data [i] = F16Dot16.multiplyDivide (F26Dot6.fromInt (originalData [i]), N, D); }
        data [i] = scale (originalData [i], N, D); }
    }
    
    int countLowZeros (int n) {
      for (int i = 0; i < 32; i++) {
        if ((n & 0x1) != 0) {
          return i; }
        n >>= 1; }
      return 32;
    }
    
    int getShift (int v) {
      if ((v & (v-1)) != 0 || v == 0) {
        return -1; }
      else {
        return countLowZeros (v); }
    }


    int/*26.6*/ scale (int v, int /*16.16*/ n, int /*16.16*/ d) {
      int shift = countLowZeros (n | d) - 1;

      if (shift > 0) {
        n >>= shift;
        d >>= shift; }

      if (n < 0x02000000) {
        n <<= 6; }
      else{
        d >>= 6; }

      if (n <= 0x7fff) { /* Check to see if N fits in a short    */
        shift = getShift (d);

        if ( shift >= 0 ) { /* FAST SCALE */
          return ((v * n) + (d >> 1)) >> shift; }
        else if (v < 0) {
          return - ((-v * n) + (d >> 1)) / d; }
        else {
          return (  (v * n) + (d >> 1)) / d; }}
      else {
        return F16Dot16.multiply (v, F16Dot16.divide (n, d)); }
    }


    void put (int entry, int value) throws InvalidGlyphException {
      if (entry < 0 || data.length <= entry) {
        throw new InvalidGlyphException ("attempt to write CVT entry " + entry 
            + " (" + data.length + " actual entries)"); }
      data [entry] = value;
    }
    
    int get (int entry) throws InvalidGlyphException {
      if (entry < 0 || data.length <= entry) {
        return 0; }
      return data [entry];
    }
  }
  
  //----------------------------------------------------------------------------  
  private final class FunctionDefinitions {
    private OTByteArray[] instructionStreams;
    private int[] start;
    private int[] limit;
    
    public FunctionDefinitions (int max) {
      instructionStreams = new OTByteArray [max];
      start = new int [max];
      limit = new int [max];
    }
    
    public void setFunction (int index, OTByteArray instructionStream, 
                             int start, int limit) {
      instructionStreams [index] = instructionStream;
      this.start [index] = start;
      this.limit [index] = limit;
    }
  }
  
  //----------------------------------------------------------------------------
  private GraphicState gs;
  private GraphicParameters gp;
  private StorageArea storageArea;
  private Stack stack;
  private FunctionDefinitions functionDefs;
  private FunctionDefinitions instructionDefs = new FunctionDefinitions (0x100);
  private int unitsPerEm; 
  private Matrix em2px;
  private TTPoint[] twilightZone;
  
  public TTInterpreter (int storageAreaSize, int maxStackDepth, 
                        int cvtSize, OTByteArray cvt, 
                        int nbFunctionDefs, int nbTwilightPoints) 
  throws InvalidFontException, UnsupportedFontException {
    this.gs = new GraphicState (cvtSize, cvt);
    this.gp = new GraphicParameters ();
    this.storageArea = new StorageArea (storageAreaSize);
    this.stack = new Stack (maxStackDepth);
    this.functionDefs = new FunctionDefinitions (nbFunctionDefs);
    this.instructionDefs = new FunctionDefinitions (0x100);
    this.twilightZone = new TTPoint [nbTwilightPoints];
    for (int p = 0; p < twilightZone.length; p++) {
      this.twilightZone [p] = new TTPoint (0, 0, false); }
  }
  
  public void setUnitsPerEm (int unitsPerEm) {
    this.unitsPerEm = unitsPerEm;
  }
  
  public void setScaling (double pointSize, double ppemX, double ppemY,
                          double dX, double dY)
  throws InvalidFontException, UnsupportedFontException {
   
    gs.setPointSize (pointSize);
    
    boolean hintAtEmSquare = false; //TODO_ERIC
    boolean integerScaling = false; // TODO_ERIC
    
    em2px = new Matrix (ppemX, 0, 0, ppemY, dX, dY);
      
    { int /*f16.16*/ ax, ay;
    
      if (hintAtEmSquare) {
        ax = F16Dot16.fromInt (unitsPerEm);
        ay = F16Dot16.fromInt (unitsPerEm); }
      else {
        ax = F16Dot16.fromDouble (Math.max (em2px.a, em2px.c));
        ay = F16Dot16.fromDouble (Math.max (em2px.b, em2px.d));
      
        if (integerScaling) {
          ax = F16Dot16.round (ax);
          ay = F16Dot16.round (ay); }}

      gs.cvtxN = ax;
      gs.cvtD = F16Dot16.fromInt (unitsPerEm);
      gs.cvtyN = ay;
      
      if (ax >= ay) {
        gs.cvtStretch.x = F16Dot16.ONE;
        gs.cvtStretch.y = F16Dot16.divide (ay, ax);
        gs.setPixelsPerEm (ppemX); }
      else {
        gs.cvtStretch.x = F16Dot16.divide (ax, ay);
        gs.cvtStretch.y = F16Dot16.ONE; 
        gs.setPixelsPerEm (ppemY); }}
    
    gs.resetAndScaleCVT ();    
  }  

  private TTPoint getPoint (TTOutline outline, int zone, int index) throws InvalidGlyphException {
    if (zone == 0) {
      return twilightZone [index]; }
    else {
      return outline.getPoint (index); }
  }
  
  private int getNumOutlinePoints (TTOutline outline, int zone) throws InvalidGlyphException {
    if (zone == 0) {
      return twilightZone.length; }
    else {
      return outline.getNumOutlinePoints (); }
  }
   
  public int getScanType () {
    int scanControl = gs.getScanControl ();
    int ppemThreshold = scanControl & 0xFF;
    boolean doScanControl;
    
    boolean imageIsRotated = false; // TODO
    boolean imageIsStretched = false; // TODO
    
    if ((scanControl & 0x100) != 0 && gs.getPixelsPerEm () <= ppemThreshold) {
      doScanControl = true; }
    else if ((scanControl & 0x100) != 0 && ppemThreshold == 0xFF) {
      doScanControl = true; }
    else if ((scanControl & 0x200) != 0 && imageIsRotated) {
      doScanControl = true; }
    else if ((scanControl & 0x400) != 0 && imageIsStretched) {
      doScanControl = true; }
    else if ((scanControl & 0x800) != 0 && gs.getPixelsPerEm () > ppemThreshold) {
      doScanControl = false; }
    else if ((scanControl & 0x1000) != 0 && ! imageIsRotated) {
      doScanControl = false; }
    else if ((scanControl & 0x2000) != 0 && ! imageIsStretched) {
      doScanControl = false; }
    else {
      doScanControl = false; }

    if (doScanControl) {
      return scanControl >> 16; }
    else {
      return 2; }
  }
  
  
  public void runFpgm (OTByteArray instructionStream, int offset, int limit)  
  throws InvalidFontException, UnsupportedFontException {
    run (null, instructionStream, offset, limit);
  }
  
  public void runPrep (OTByteArray instructionStream, int offset, int limit)  
  throws InvalidFontException, UnsupportedFontException {
    if (gs.getPixelsPerEm() <= 1) {
      return; }
    
    gs.setAutoFlip (true);
    gs.setDeltaBase (9);
    gs.setDeltaShift (3);
    gs.setRoundMode (gs.ROUND_TO_GRID);
    gs.setMinimumDistance (F26Dot6.ONE);
    gs.setControlValueCutIn (F26Dot6.fromDouble (17.0d / 16.0d));
    gs.setSingleWidth (F26Dot6.ZERO);
    gs.setSingleWidthCutIn (F26Dot6.ZERO);
    gs.setAngleWeight(128);
    gs.setScanControl(0);
    gs.setInstrControl (0);

    try {
      gs.defaultGraphicState = (LocalGraphicState) gs.localGraphicState.clone (); }
    catch (CloneNotSupportedException e) {
      throw new RuntimeException ("LocalGraphicState cannot be cloned!"); }

    run (null, instructionStream, offset, limit);
    
    if ((gs.getInstrControl () & 0x02) == 0) {
      try {
        gs.defaultGraphicState = (LocalGraphicState) gs.localGraphicState.clone (); }
      catch (CloneNotSupportedException e) {
        throw new RuntimeException ("LocalGraphicState cannot be cloned!"); }}
  }
  
  public void runGlyf (TTSimpleOutline outline, 
                       OTByteArray instructionStream, int offset, int limit) 
  throws InvalidGlyphException, UnsupportedFontException {
    if (gs.getPixelsPerEm() <= 1) {
      return; }

    if ((gs.getInstrControl () & 0x01) != 0) {
      return; }
           
    try {
      gs.localGraphicState = (LocalGraphicState) gs.defaultGraphicState.clone (); }
    catch (CloneNotSupportedException e) {
      throw new RuntimeException ("LocalGraphicState cannot be cloned!"); }

    run (outline, instructionStream, offset, limit);
  }

  private void run (TTSimpleOutline outline, OTByteArray instructionStream, int offset, int limit) 
      throws InvalidGlyphException, UnsupportedFontException {

    gs.setRp0 (0);
    gs.setRp1 (0);
    gs.setRp2 (0);
    gs.setZp0 (1);
    gs.setZp1 (1);
    gs.setZp2 (1);
    gs.setVector (gs.projectionVector, F2Dot14.ONE, F2Dot14.ZERO, false);
    gs.setVector (gs.freedomVector, F2Dot14.ONE, F2Dot14.ZERO, false);
    gs.setVector (gs.originalProjectionVector, F2Dot14.ONE, F2Dot14.ZERO, false);
    gs.setLoopVariable (1);
    
    runWithoutInit (outline, instructionStream, offset, limit);
  }

  private int scaleAndRound(int value, int numerator, int denominator)
  {
	int d2 = denominator>>1;  
	if (value < 0) 
		value = -(-value * numerator + d2)/denominator;
  	else
  		value = (value * numerator + d2)/denominator;
	
	return value;
  }


  private void runWithoutInit (TTSimpleOutline outline, OTByteArray instructionStream, int offset, int limit) 
      throws InvalidGlyphException, UnsupportedFontException {

    int start = offset;
    

    try {
    while (offset < limit) {
      int instruction = instructionStream.getuint8 (offset); /* some intruction */
      if (ScalerDebugger.debugOn && debugger != null) {
        logInstruction (offset, instruction); }
      offset++;

      switch (instruction) {
        // for each instruction, we give the mnemonic, full name, and page
        // number in the OpenType spec 

        case 0x40: { // NPUSHB push n bytes [189]
          int n = instructionStream.getuint8 (offset);
          offset++;
          for (int i = 0; i < n; i++) {
            stack.push (instructionStream.getuint8 (offset));
            offset++; }
          break; }

        case 0x41: { // NPUSHW push n words [190]
          int n = instructionStream.getuint8 (offset);
          offset++;
          for (int i = 0; i < n; i++) {
            stack.push (instructionStream.getint16 (offset));
            offset += 2; }
          break; }
        
        case 0xB0:
        case 0xB1:
        case 0xB2:
        case 0xB3:
        case 0xB4:
        case 0xB5:
        case 0xB6:
        case 0xB7: { // PUSHB push bytes [191]
          int n = (instruction & 0x7) + 1;
          for (int i = 0; i < n; i++) {
            stack.push (instructionStream.getuint8 (offset));
            offset++; }
          break; }

        case 0xB8:
        case 0xB9:
        case 0xBA:
        case 0xBB:
        case 0xBC:
        case 0xBD:
        case 0xBE:
        case 0xBF: { // PUSHW push words [192]
          int n = (instruction & 0x7) + 1;
          for (int i = 0; i < n; i++) {
            stack.push (instructionStream.getint16 (offset));
            offset += 2; }
          break; }
        
        case 0x43: { // RS read store [194]
          stack.push (storageArea.getuint32 (stack.pop ()));
          break; }
        
        case 0x42: { // WS write store [195]
          int value = stack.pop ();
          int location = stack.pop ();
          storageArea.setuint32 (location, value);
          break; }
          
        case 0x44: { // WCVTP write control value table in pixel units [197]
          int /*f26.6*/ value = stack.pop ();
          int location = stack.pop ();
          gs.putCVT (location, value);
          break; }
        
        case 0x70: { // WCVTF write control value table in funits [198]
          int value = stack.pop ();
          int location = stack.pop ();
          gs.putCVTPixels (location, value);
          
          break; }
        
        case 0x45: { // RCVT read control value table [199]
          stack.push (gs.getCVT (stack.pop ()));
          break; }
        
        case 0x00:
        case 0x01: {  // SVTCA set vectors to coordinate axis [202]
          boolean rotate = (instruction & 0x1) == 0;

          gs.setVector (gs.projectionVector, 
              F2Dot14.ONE, F2Dot14.ZERO, rotate);
          gs.setVector (gs.originalProjectionVector, 
              F2Dot14.ONE, F2Dot14.ZERO, rotate);         
          gs.setVector (gs.freedomVector, 
              F2Dot14.ONE, F2Dot14.ZERO, rotate);
          break; }
        
        case 0x02:
        case 0x03: {  // SPVTCA set projection vector to coordinate axis [203]
          boolean rotate = (instruction & 0x1) == 0;
          gs.setVector (gs.projectionVector, 
              F2Dot14.ONE, F2Dot14.ZERO, rotate);
          gs.setVector (gs.originalProjectionVector, 
              F2Dot14.ONE, F2Dot14.ZERO, rotate); 
          break; }
        
        case 0x04:
        case 0x05: { // SFVTCA set freedom vector to coordinate axis [204]
          boolean rotate = (instruction & 0x1) == 0;
          gs.setVector (gs.freedomVector, 
              F2Dot14.ONE, F2Dot14.ZERO, rotate);
          break; }
        
        case 0x06:
        case 0x07: { // SPVTL set projection vector to line [205]
          boolean rotate = (instruction & 0x01) == 1;
          TTPoint p1 = getPoint (outline, gs.getZp2 (), stack.pop ());
          TTPoint p2 = getPoint (outline, gs.getZp1 (), stack.pop ());
          gs.setVector (gs.projectionVector, p1.hinted, p2.hinted, rotate);
          break; }

        case 0x08:
        case 0x09: { // SFVTL set freedom vector to line [208]
          boolean rotate = (instruction & 0x01) == 1;
          TTPoint p1 = getPoint (outline, gs.getZp2 (), stack.pop ());
          TTPoint p2 = getPoint (outline, gs.getZp1 (), stack.pop ());
          gs.setVector (gs.freedomVector, p1.hinted, p2.hinted, rotate); 
          break; }
        
        case 0x86:
        case 0x87: { // SDPVTL, set dual projection vector to line [211]
          boolean rotate = (instruction & 0x01) == 1;
          TTPoint p1 = getPoint (outline, gs.getZp2 (), stack.pop ());
          TTPoint p2 = getPoint (outline, gs.getZp1 (), stack.pop ());
          gs.setVector (gs.originalProjectionVector, p1.unhinted, p2.unhinted, rotate);
          gs.setVector (gs.projectionVector, p1.hinted, p2.hinted, rotate);
          break; }
        
        case 0x0e: { // SFVTPV set freedom vector to projection vector [210]
          gs.setVector (gs.freedomVector, 
                        gs.projectionVector.x, gs.projectionVector.y, false);
          break; }
        
        case 0x0a: { // SPVFS set projection vector from stack [212]
          int /*2.14*/ y = stack.pop ();
          int /*2.14*/ x = stack.pop ();
          gs.setVector (gs.projectionVector, x, y, false);
          gs.setVector (gs.originalProjectionVector, x,y,false); 
          break; }
        
        case 0x0b: { // SFVFS set freedom vector from stack [214]
          int /*2.14*/ y = stack.pop ();
          int /*2.14*/ x = stack.pop ();
          gs.setVector (gs.freedomVector, x, y, false);
          break; }
        
        case 0x0c: { // GPV get projection vector [216]
          stack.push (gs.projectionVector.x);
          stack.push (gs.projectionVector.y);
          break; }

        case 0x0d: { // GFV get freemdom vector [218]
          stack.push (gs.freedomVector.x);
          stack.push (gs.freedomVector.y);
          break; }
        
        case 0x10: { // SRP0 set reference point 0 [220]
          gs.setRp0 (stack.pop ());
          break; }
        
        case 0x11: { // SRP1 set reference point 1 [221]
          gs.setRp1 (stack.pop ());
          break; }
        
        case 0x12: { // SRP2 set reference point 2 [222]
          gs.setRp2 (stack.pop ());
          break; }
        
        case 0x13: { // SZP0 set zone pointer 0 [223]
          gs.setZp0 (stack.pop ());
          break; }
        
        case 0x14: { // SZP1 set zone pointer 1 [224]
          gs.setZp1 (stack.pop ());
          break; }
        
        case 0x15: { // SZP2 set zone pointer 2 [225]
          gs.setZp2 (stack.pop ());
          break; }
        
        case 0x16: { // SZPS set zone pointers [226]
          int zone = stack.pop ();
          gs.setZp0 (zone);
          gs.setZp1 (zone);
          gs.setZp2 (zone);
          break; }
        
        case 0x19: { // RTHG round to half grid [227]
          gs.setRoundMode (gs.ROUND_TO_HALF_GRID);
          break; }
        
        case 0x18: { // RTG round to  grid [228]
          gs.setRoundMode (gs.ROUND_TO_GRID);
          break; }
        
        case 0x3d: { // RTDG round to double grid [229]
          gs.setRoundMode (gs.ROUND_TO_DOUBLE_GRID);
          break; }

        case 0x7d: { // RDTG round down to grid [230]
          gs.setRoundMode (gs.ROUND_DOWN_TO_GRID);
          break; }
        
        case 0x7c: { // RUTG round up to grid [231]
          gs.setRoundMode (gs.ROUND_UP_TO_GRID);
          break; }
        
        case 0x7a: { // ROFF round off [232]
          gs.setRoundMode (gs.ROUND_OFF);
          break; }
        
        case 0x76:   // SROUND super round [233]
        case 0x77: { // S45ROUND super round 45 degrees [238]
          int n = stack.pop ();
          int period, period45, phase = 0, threshold;
          
          if ((instruction & 0x1) == 0x0) {
            switch (n & 0xc0) {
              case 0x00: {
                period =  0x20; // 1/2 in F26.6
                break; }
              case 0x40: {
                period = 0x40; // 1 in F26.6
                break; }
              case 0x80: {
                period = 0x80; // 2 in F26.6
                break; }
              default: {
                period = 999; /* Illegal */
                break; }}}
          
          else {
            period45 = 11591;
            switch (n & 0xc0) {
              case 0x00: {
                period45 >>= 1;
                break; }
              case 0x40: {
                break; }
              case 0x80: {
                period45 <<= 1;
                break; }
              default: {
                period45 = 999; /* Illegal */
                break; }}

            int tmp = (16 /*bits*/ - 2 - 6);
            period = (period45 + (1 << (tmp - 1))) >> tmp; /*convert from 2.30 to 26.6 */ }

          switch (n & 0x30) {
            case 0x00: {
              phase = 0;
              break; }
            case 0x10: {
              phase = ((period + 2) >> 2);
              break; }
            case 0x20: {
              phase = ((period + 1) >> 1);
              break; }
            case 0x30: {
              phase = ((period + period + period + 2) >> 2);
              break; }}

          if ((n & 0x0f) == 0) {
            threshold = period - 1; }
          else {
            threshold = (((n & 0x0f) - 4) * period + 4) >> 3; }

          gs.setRounding (period, phase, threshold, ((instruction & 0x1) == 0x0) ? gs.SUPER_ROUND : gs.SUPER_45_ROUND);

          break; }
                
        case 0x17: { // SLOOP set loop variable [239]
          gs.setLoopVariable (stack.pop ());
          break; }
        
        case 0x1a: { // SMD set minimum distance [240]
          gs.setMinimumDistance (stack.pop ());
          break; }
        
        case 0x8e: { // INSTCRL instruction execution control [241]
          int selectorFlag = stack.pop ();
          int value = stack.pop ();
          
          int instrControl = gs.getInstrControl();
          if (selectorFlag == 0x01) {
            instrControl &= ~ 0x01; }
          else if (selectorFlag == 0x02) {
            instrControl &= ~ 0x02; }
          instrControl |= value;
          gs.setInstrControl (instrControl);
          break; }
        
        case 0x85: { // SCANCTRL scan conversion control [243]
          gs.setScanControl ((gs.getScanControl () & 0xffff0000) | (stack.pop () & 0xffff));
//          int flags = stack.pop ();
//          int ppemThreshold = flags & 0xFF;
//          boolean scanControl = false;
//          if ((flags & 0x100) != 0) {
//            scanControl |= gs.getPixelsPerEm() <= ppemThreshold; }
//          if ((flags & 0x200) != 0) {
//            scanControl |= false; /* glyph is rotated; TODO */ }
//          if ((flags & 0x400) != 0) {
//            scanControl |= false; /* glyph is stretched; TODO */ }
//          if ((flags & 0x800) != 0) {
//            scanControl &= gs.getPixelsPerEm () <= ppemThreshold; }
//          if ((flags & 0x1000) != 0) {
//            scanControl &= false; /* glyph is rotated TODO */ }
//          if ((flags & 0x2000) != 0) {
//            scanControl &= false; /* glyph is stretched TODO */ }
//          gs.setDropoutControl (scanControl);
          break; }
        
        case 0x8d: { // SCANTYPE scantype [245]
          gs.setScanControl ((stack.pop () << 16) | (gs.getScanControl () & 0xffff));
//          gs.setScanControl (stack.pop ());
          break; }
        
        case 0x1d: { // SCVTCI set control value table cut in [247]
          gs.setControlValueCutIn (stack.pop ());
          break; }
        
        case 0x1e: { // SSWCI set single width cut in [248]
          gs.setSingleWidthCutIn (stack.pop ());
          break; }
        
        case 0x1f: { // SSW set single width [249]
          gs.setSingleWidth (stack.pop ());
          break; }
        
        case 0x4d: { // FLIPON set the auto flip boolean to on [250]
          gs.setAutoFlip (true);
          break; }
        
        case 0x4e: { // FLIPOFF set the auto flip boolean to off [251]
          gs.setAutoFlip (false);
          break; }
        
        case 0x7e: { // SANGW set angle weight [252]
          gs.setAngleWeight (stack.pop ());
          break; }
        
        case 0x5e: { // SDB set delta base [253]
          gs.setDeltaBase (stack.pop ());
          break; }
        
        case 0x5f: { // SDS set delta shift [254]
          gs.setDeltaShift (stack.pop ());
          break; }
        
        case 0x46:
        case 0x47: { // GC get coordinate projected onto projection vector [256]
           TTPoint pt = getPoint (outline, gs.getZp2 (), stack.pop ());
          
          if ((instruction & 0x1) == 0) {
            stack.push (gs.project (gs.projectionVector, pt.hinted)); }
          else {
            stack.push (gs.project (gs.originalProjectionVector, pt.unhinted)); }
 
          break; }

        case 0x48: { // SCFS set coordinate from stack using projection
                     // vector and freedom vector [258]
          int targetCoordinate = stack.pop ();
          TTPoint p = getPoint (outline, gs.getZp2 (), stack.pop ());
          int actualCoordinate = gs.project (gs.projectionVector, p.hinted);
          gs.move (p, targetCoordinate - actualCoordinate);
          if (gs.getZp2 () == 0) { // undocumented
        	  p.unhinted.x = p.hinted.x;
        	  p.unhinted.y = p.hinted.y;
        	  
          }
          
          break; }
        
        case 0x49:
        case 0x4a: { // MD measure distance [259]
          TTPoint p1 = getPoint (outline, gs.getZp1 (), stack.pop ());
          TTPoint p2 = getPoint (outline, gs.getZp0 (), stack.pop ());
          int /*f26.6*/ v;
          if ((instruction & 0x1) == 0) {
            v = gs.project (gs.originalProjectionVector, p1.unhinted, p2.unhinted); }
          else {
            v = gs.project (gs.projectionVector, p1.hinted, p2.hinted); }
          stack.push (v);
          break; }
        
        case 0x4b: { // MPPEM measure pixels per em [261]
          stack.push (gs.getPixelsPerEm ()); 
          break; }
        
        case 0x4c: { // MPS measure point size [262]
          stack.push (gs.getPointSize ());
          break; }
        
        case 0x80: { // FLIPPT flip point [263]
          int zone = gs.getZp0 ();
          for (int count = gs.getAndResetLoopVariable (); count > 0; count--) {
            TTPoint p = getPoint (outline, zone, stack.pop ());
            p.onCurve = ! p.onCurve; }
          break; }
        
        case 0x81:   // FLIPRGON flip range on [264]
        case 0x82: { // FLIPRGOFF flip range off [265]
          int zone = gs.getZp0 ();
          int hp = stack.pop ();
          int lp = stack.pop ();
          boolean onCurve = (instruction & 0x1) == 1;
          for (int i = lp; i <= hp; i++) {
            getPoint (outline, zone, i).onCurve = onCurve; }
          break; }

        case 0x32:
        case 0x33: { // SHP shift point by the last point [266]
          TTPoint lastPoint = (instruction & 0x1) == 0 ?
              getPoint (outline, gs.getZp1 (), gs.getRp2 ()) 
            : getPoint (outline, gs.getZp0 (), gs.getRp1 ());
            
         int delta = gs.project (gs.projectionVector, lastPoint.unhinted, lastPoint.hinted);
         int dx = F26Dot6.multiplyByF2Dot14DivideByF2Dot14 (delta, gs.freedomVector.x, gs.fDotP);
         int dy = F26Dot6.multiplyByF2Dot14DivideByF2Dot14 (delta, gs.freedomVector.y, gs.fDotP);
          
          for (int count= gs.getAndResetLoopVariable (); count > 0; count--) {
            TTPoint p = getPoint (outline, gs.getZp2 (), stack.pop ());
            if (p != lastPoint) {
              gs.move (p, dx, dy); }}
            
          break; }

        case 0x34:
        case 0x35: { // SHC shift contour by last point [267]
          int contour = stack.pop ();
          TTPoint lastPoint = (instruction & 0x1) == 0 ?
              getPoint (outline, gs.getZp1 (), gs.getRp2 ()) 
            : getPoint (outline, gs.getZp0 (), gs.getRp1 ());
            
         int delta = gs.project (gs.projectionVector, lastPoint.unhinted, lastPoint.hinted);
         int dx = F26Dot6.multiplyByF2Dot14DivideByF2Dot14 (delta, gs.freedomVector.x, gs.fDotP);
         int dy = F26Dot6.multiplyByF2Dot14DivideByF2Dot14 (delta, gs.freedomVector.y, gs.fDotP);

         int first = outline.getContourFirstPoint (contour);
         int last = outline.getContourLastPoint (contour);
         for (int i = first; i <= last; i++) {
           TTPoint p = getPoint (outline, gs.getZp2 (), i);
           if (p != lastPoint) {
             gs.move (p, dx, dy); }}

          break; }
            
        case 0x36:
        case 0x37: { // SHZ shift zone by last point [268]
          int zone = stack.pop ();
          TTPoint lastPoint = (instruction & 0x1) == 0 ?
              getPoint (outline, gs.getZp1 (), gs.getRp2 ()) 
            : getPoint (outline, gs.getZp0 (), gs.getRp1 ());
            
          int delta = gs.project (gs.projectionVector, lastPoint.unhinted, lastPoint.hinted);
          int dx = F26Dot6.multiplyByF2Dot14DivideByF2Dot14 (delta, gs.freedomVector.x, gs.fDotP);
          int dy = F26Dot6.multiplyByF2Dot14DivideByF2Dot14 (delta, gs.freedomVector.y, gs.fDotP);

          int first = 0;
          int last = getNumOutlinePoints (outline, zone)-1;
          for (int i = first; i <= last; i++) {
            TTPoint p = getPoint (outline, zone, i);
            if (p != lastPoint) {
              gs.move (p, dx, dy); }}

          break; }
            
        case 0x38: { // SHPIX shift point by pixel amount [269]
          int /*f26.6*/ shiftAmount = stack.pop ();
          int dx = F26Dot6.multiplyByF2Dot14 (shiftAmount, gs.freedomVector.x);
          int dy = F26Dot6.multiplyByF2Dot14 (shiftAmount, gs.freedomVector.y);
          
          for (int count = gs.getAndResetLoopVariable (); count > 0; count--) {
            TTPoint p = getPoint (outline, gs.getZp2 (), stack.pop ());
            gs.move (p, dx, dy); }

          break; }
        
        case 0x3a:
        case 0x3b: { // MSIRP move stack indirect relative point [270]
          int desiredDistance = stack.pop ();
          int pIndex = stack.pop ();
          TTPoint p = getPoint (outline, gs.getZp1 (), pIndex);
          TTPoint rp0 = getPoint (outline, gs.getZp0 (), gs.getRp0 ());
          
          if (gs.getZp1 () == 0) { // undocumented
            p.unhinted.x = F26Dot6.multiplyByF2Dot14 (desiredDistance, 
                                     gs.projectionVector.x);
            p.unhinted.y = F26Dot6.multiplyByF2Dot14 (desiredDistance,
                                     gs.projectionVector.y);
            p.hinted.x = p.unhinted.x;
            p.hinted.y = p.unhinted.y; }
          
          int actualDistance = gs.project (gs.projectionVector, 
                                   rp0.hinted, p.hinted);
          
          gs.move (p, desiredDistance - actualDistance);
          
          gs.setRp1 (gs.getRp0 ());
          gs.setRp2 (pIndex);
          if ((instruction & 0x1) != 0) {
            gs.setRp0 (pIndex); }

          break; }
        
        case 0x2e:
        case 0x2f: { // MDAP move direct absolute point [271]
          int pIndex = stack.pop ();
          TTPoint p = getPoint (outline, gs.getZp0 (), pIndex);

          gs.setRp0 (pIndex);
          gs.setRp1 (pIndex);
                          
          int delta = 0;
          if ((instruction & 0x1) != 0){
            int projection = gs.project (gs.projectionVector, p.hinted);
            delta = gs.round (projection, 
                              gp.getEngineCompensation (0)) - projection; }
          gs.move (p, delta);

          break; }
        
        case 0x3e:
        case 0x3f: { // MIAP move indirect absolute point [272]
          int /*F26.6*/ desiredDistance = gs.getCVT (stack.pop ());         
          int pIndex = stack.pop ();
          TTPoint p = getPoint (outline, gs.getZp0 (), pIndex);

          gs.setRp0 (pIndex);
          gs.setRp1 (pIndex);
         
          if (gs.getZp0 () == 0) { // undocumented
            p.unhinted.x = F26Dot6.multiplyByF2Dot14 (desiredDistance, gs.projectionVector.x);
            p.unhinted.y = F26Dot6.multiplyByF2Dot14 (desiredDistance, gs.projectionVector.y);
            p.hinted.x = p.unhinted.x;
            p.hinted.y = p.unhinted.y; }
          
          int currentDistance = gs.project (gs.projectionVector, p.hinted);
          
          if ((instruction & 0x1) != 0) {
            if (Math.abs (desiredDistance - currentDistance) > gs.getControlValueCutIn ()) {
              desiredDistance = currentDistance; }
            desiredDistance = gs.round (desiredDistance, gp.getEngineCompensation (0)); }        

          gs.move (p, desiredDistance - currentDistance);
          
          break; }

        case 0xc0:
        case 0xc1:
        case 0xc2:
        case 0xc3:
        case 0xc4:
        case 0xc5:
        case 0xc6:
        case 0xc7:
        case 0xc8:
        case 0xc9:
        case 0xca:
        case 0xcb:
        case 0xcc:
        case 0xcd:
        case 0xce:
        case 0xcf:
        case 0xd0:
        case 0xd1:
        case 0xd2:
        case 0xd3:
        case 0xd4:
        case 0xd5:
        case 0xd6:
        case 0xd7:
        case 0xd8:
        case 0xd9:
        case 0xda:
        case 0xdb:
        case 0xdc:
        case 0xdd:
        case 0xde:
        case 0xdf: { // MDRP move direct relative point [276]
          int pIndex = stack.pop ();
          TTPoint p = getPoint (outline, gs.getZp1 (), pIndex);
          TTPoint rp0 = getPoint (outline, gs.getZp0 (), gs.getRp0 ());
          
          int desiredDistance;
          if (gs.getZp0 () == 0 || gs.getZp1() == 0 || outline.unscaledCoordinatesAreInvalid) {
            desiredDistance = gs.project (gs.originalProjectionVector, rp0.unhinted, p.unhinted); }
          else {
        	int xDiff = scaleAndRound(p.unscaled.x - rp0.unscaled.x, gs.cvtxN >> 10, gs.cvtD >> 16);
        	int yDiff = scaleAndRound(p.unscaled.y - rp0.unscaled.y, gs.cvtyN >> 10, gs.cvtD >> 16);
        		
            desiredDistance = gs.project (gs.originalProjectionVector, xDiff, yDiff); }
//
          
//          MDRP 16
//          XProject ffffffab
//          RoundToGrid ffffffc0
//          XProject ffffffcd
//          XMovePoint 1c0 16
//          MDRP fffffff3
          
//          int desiredDistance = gs.project (gs.originalProjectionVector, rp0.unhinted, p.unhinted);
          
          desiredDistance = gs.snapToSingleWidth (desiredDistance);
          int unroundedDesiredDistance = desiredDistance; // to keep the sign

          int engineCompensation = gp.getEngineCompensation ((instruction & 0x3));
          if ((instruction & 0x4) != 0) { // round
            desiredDistance = gs.round (desiredDistance, engineCompensation); }
          else {
            desiredDistance = gs.roundOff (desiredDistance, engineCompensation); }
          
          if ((instruction & 0x8) != 0) { // keepDistance
            int min = gs.getMinimumDistance ();
            if (unroundedDesiredDistance >= 0) {
              if (desiredDistance < min) {
                desiredDistance = min; }}
            else {
              if (desiredDistance > - min) {
                desiredDistance = - min; }}}
          int currentDistance = gs.project (gs.projectionVector, rp0.hinted, p.hinted);
          gs.move (p, desiredDistance - currentDistance); 
          
          gs.setRp1 (gs.getRp0());
          gs.setRp2 (pIndex);
          if ((instruction & 0x10) != 0) {
            gs.setRp0 (pIndex); }
          
          break; }

        case 0xe0:
        case 0xe1:
        case 0xe2:
        case 0xe3:
        case 0xe4:
        case 0xe5:
        case 0xe6:
        case 0xe7:
        case 0xe8:
        case 0xe9:
        case 0xea:
        case 0xeb:
        case 0xec:
        case 0xed:
        case 0xee:
        case 0xef:
        case 0xf0:
        case 0xf1:
        case 0xf2:
        case 0xf3:
        case 0xf4:
        case 0xf5:
        case 0xf6:
        case 0xf7:
        case 0xf8:
        case 0xf9:
        case 0xfa:
        case 0xfb:
        case 0xfc:
        case 0xfd:
        case 0xfe:
        case 0xff: { // MIRP move indirect relative point [279]         
          int desiredDistance = gs.getCVT (stack.pop ());
          int pIndex = stack.pop ();
          TTPoint p = getPoint (outline, gs.getZp1 (), pIndex);
          TTPoint rp0 = getPoint (outline, gs.getZp0 (), gs.getRp0 ());
          
          desiredDistance = gs.snapToSingleWidth (desiredDistance);
          
          if (gs.getZp1 () == 0) { // undocumented
            p.unhinted.x = rp0.unhinted.x + F26Dot6.multiplyByF2Dot14 (desiredDistance, gs.projectionVector.x);
            p.unhinted.y = rp0.unhinted.y + F26Dot6.multiplyByF2Dot14 (desiredDistance, gs.projectionVector.y);
            p.hinted.x = p.unhinted.x;
            p.hinted.y = p.unhinted.y; }
          
          int originalDistance = gs.project (gs.originalProjectionVector, rp0.unhinted, p.unhinted);
          int currentDistance = gs.project (gs.projectionVector, rp0.hinted, p.hinted);
          
          if (gs.getAutoFlip () && ! F26Dot6.sameSign (originalDistance, desiredDistance)) {
            desiredDistance = - desiredDistance; }            
          
          int engineCompensation = gp.getEngineCompensation (instruction & 0x3);
          
          if ((instruction & 0x4) != 0) { // round
            if (Math.abs (desiredDistance - originalDistance) > gs.getControlValueCutIn ()) {
              desiredDistance = originalDistance; }           
            desiredDistance = gs.round (desiredDistance, engineCompensation); }
          else {
            desiredDistance = gs.roundOff (desiredDistance, engineCompensation); }
          
          if ((instruction & 0x8) != 0) { // keepDistance
            int min = gs.getMinimumDistance ();
            if (originalDistance >= 0) {
              if (desiredDistance < min) {
                desiredDistance = min; }}
            else {
              if (desiredDistance > - min) {
                desiredDistance = - min; }}}

          gs.move (p, desiredDistance - currentDistance); 
          
          gs.setRp1 (gs.getRp0 ());
          gs.setRp2 (pIndex);
          if ((instruction & 0x10) != 0) {
            gs.setRp0 (pIndex); }
          
          break; }
        
        case 0x3c: { // ALIGNRP align relative point [284]
          TTPoint p0 = getPoint (outline, gs.getZp0 (), gs.getRp0 ());
          
          for (int count = gs.getAndResetLoopVariable (); count > 0; count--) {
            TTPoint p = getPoint (outline, gs.getZp1 (), stack.pop ());
            gs.move (p, - gs.project (gs.projectionVector, p0.hinted, p.hinted)); }

          break; }
        
        case 0x7f: { // AA adjust angle 
          // this instruction is marked "no longer supported"
          throw new InvalidGlyphException ("TT instruction AA is no longer supported"); }
        
        case 0x0f: { // ISECT move point to the intersection of two lines [286]
          TTPoint b1 = getPoint (outline, gs.getZp0 (), stack.pop ());
          TTPoint b0 = getPoint (outline, gs.getZp0 (), stack.pop ());
          TTPoint a1 = getPoint (outline, gs.getZp1 (), stack.pop ());
          TTPoint a0 = getPoint (outline, gs.getZp1 (), stack.pop ());
          TTPoint p  = getPoint (outline, gs.getZp2 (), stack.pop ());
          
          int bx = b1.hinted.x - b0.hinted.x;
          int by = b1.hinted.y - b0.hinted.y;

          int ax = a1.hinted.x - a0.hinted.x;
          int ay = a1.hinted.y - a0.hinted.y;

          int N;
          int D;
          
          if (by == 0) {
            if (ax == 0) {
              gs.set (p, a0.hinted.x, b0.hinted.y);
              break; }
            N = a0.hinted.y - b0.hinted.y;
            D = -ay; } 
          
          else if (bx == 0) {
            if (ay == 0) {
              gs.set (p, b0.hinted.x, a0.hinted.y);
              break; }
            N = a0.hinted.x - b0.hinted.x;
            D = -ax; } 
          
          else if (F26Dot6.abs (bx) >= F26Dot6.abs (by)) {
            /* To prevent out of range problems,
             *  divide both N and D with the max */
            N = (a0.hinted.y - b0.hinted.y) - F26Dot6.multiplyDivide (a0.hinted.x - b0.hinted.x, by, bx);
            D = F26Dot6.multiplyDivide (ax, by, bx) - ay; } 
          
          else {
            N = F26Dot6.multiplyDivide (a0.hinted.y - b0.hinted.y, bx, by) - (a0.hinted.x - b0.hinted.x);
            D = ax - F26Dot6.multiplyDivide (ay, bx, by); }

          if (D != 0) {
            gs.set (p,
                a0.hinted.x + F26Dot6.multiplyDivide (ax, N, D),
                a0.hinted.y + F26Dot6.multiplyDivide (ay, N, D)); } 
          else {
            /* degenerate case: parallel lines, put point in the middle */
            gs.set (p,
                (a0.hinted.x + (ax >> 1) + b0.hinted.x + (bx >> 1)) >> 1,
                (a0.hinted.y + (ay >> 1) + b0.hinted.y + (by >> 1)) >> 1); }

          break; }
        
        case 0x27: { // ALIGNPTS align points [288]
          TTPoint p1 = getPoint (outline, gs.getZp1 (), stack.pop ());
          TTPoint p2 = getPoint (outline, gs.getZp0 (), stack.pop ());

          int p1_p2 = gs.project (gs.projectionVector, p1.hinted, p2.hinted);
          gs.move (p1, p1_p2 / 2);
          gs.move (p2, p1_p2 - (p1_p2 / 2));

          break; }
        
        case 0x39: { // IP interpolate point by last relative stretch [289]
          TTPoint rp1 = getPoint (outline, gs.getZp0 (), gs.getRp1 ());
          TTPoint rp2 = getPoint (outline, gs.getZp1 (), gs.getRp2 ());
          
          boolean useScaledValues =    gs.getZp0 () == 0 
                                    || gs.getZp1 () == 0
                                    || gs.getZp2 () == 0
                                    || outline.unscaledCoordinatesAreInvalid;
          
          int rp1_rp2_original;
          if (useScaledValues) {
            rp1_rp2_original = gs.project (gs.originalProjectionVector, rp1.unhinted, rp2.unhinted); }
          else {
            rp1_rp2_original = gs.project (gs.originalProjectionVector, rp2.unscaled.x - rp1.unscaled.x, rp2.unscaled.y - rp1.unscaled.y); }
          
          int rp1_rp2_current = gs.project (gs.projectionVector, rp1.hinted, rp2.hinted);

          for (int count =  gs.getAndResetLoopVariable (); count > 0; count--) {
            TTPoint p = getPoint (outline, gs.getZp2 (), stack.pop ());
            int rp1_p_original;
            if (useScaledValues) {
              rp1_p_original = gs.project (gs.projectionVector, rp1.unhinted, p.unhinted); }
            else {
              rp1_p_original = gs.project (gs.projectionVector, p.unscaled.x - rp1.unscaled.x, p.unscaled.y - rp1.unscaled.y); }
            int rp1_p_current = gs.project (gs.projectionVector, rp1.hinted, p.hinted); 

            int desired;
            if (rp1_rp2_original == 0) {
              desired = rp1_p_original; }
            else {
              desired = F26Dot6.multiplyDivide (rp1_rp2_current, 
                                                rp1_p_original, 
                                                rp1_rp2_original); }
            gs.move (p, desired - rp1_p_current); }
          
          break; }
          

        case 0x29: { // UTF untouch point [290]
          TTPoint p = getPoint (outline, gs.getZp0 (), stack.pop ());
          gs.untouch (p);
          break; }
        
        case 0x30:
        case 0x31: { // IUP interpolate untouched points through the outline [291]
          /*
           * Think of a countour as closed (i.e. not just a sequence of points
           * with a first and a last).
           * 
           * If all the points are untouched, this instruction does nothing.
           * 
           * Otherwise, for each untouched point, there is a first touched point
           * after it on the contour, and there is a last touched point before
           * it on the contour. If i is the index of the untouched point, then
           * the index of the first touched point is the first index of a
           * touched point in the (i+1, i+2, ..., last[c], first[c], ..., i-1);
           * and the index of the last touched point is the first index of a
           * touched point in the list (i-1, i-2, ..., first[c], last[c], ...,
           * i+1). Of course, first[c] and last[c] are the indices of the
           * first and last point in the contour.
           * 
           * Now, for each untouched point, consider the relative positions (in
           * x/y, not in contour order) of this point and of the two touched
           * points around it.
           * 
           * If the original positions of the two touched points are the same
           * (on the axis considered by the instruction), then the untouched
           * points are moved by the same amount as the touched point with the
           * highest index was moved. The choice of the highest point seems
           * rather arbitrary, but that is what CoolType does.
           * 
           * If an untouched point was originally between the two touched
           * points, then its new position is interpolated from the positions of
           * the two touched points.
           * 
           * If an untouched point was originally to the left of the leftmost
           * touched point, then it is moved by the same amount as that touched
           * point was moved.
           * 
           * If an untouched point was originally to the right of the rightmost
           * touched point, then it is moved by the same amount as that touched
           * point was moved.
           */
          boolean xDirection = (instruction & 0x1) == 1;

          if (gs.getZp2 () != 1) {
            throw new InvalidGlyphException ("ZP2 should be 1 instead of " 
                + gs.getZp2 () + " in an IUP instruction"); }
          
          for (int contour = 0; contour < outline.getNumContours (); contour++) {
            int first = outline.getContourFirstPoint (contour);
            int last = outline.getContourLastPoint (contour);
            
            // find a touched point on the contour, to use as an anchor in processing
            int anchor = first;
            while (anchor <= last && ! getPoint (outline, 1, anchor).touched (xDirection)) {
              anchor ++; }
            
            // nothing touched on that contour, we are done with it
            if (anchor > last) {
              continue; }
            
            // find the next touched/untouched boundary, with
            // touchedBefore as the touched point, and 
            // p as the untouched point
            
            int touchedBefore = anchor;
            int untouched = outline.getContourNextPoint (contour, anchor);
           
            while (untouched != anchor) {
              
              if (getPoint (outline, 1, untouched).touched (xDirection)) {
                touchedBefore = untouched;
                untouched = outline.getContourNextPoint (contour, untouched); }

              else { // found an untouched point
                // find the next touched point
                int touchedAfter = outline.getContourNextPoint (contour, untouched);
                while (! getPoint (outline, 1, touchedAfter).touched (xDirection)) {
                  touchedAfter = outline.getContourNextPoint (contour, touchedAfter); }
                
                TTPoint p1 = getPoint (outline, 1, touchedBefore);
                TTPoint p2 = getPoint (outline, 1, touchedAfter);
                
                int orig1 = xDirection ? 
                                (outline.unscaledCoordinatesAreInvalid ? p1.unhinted.x : p1.unscaled.x)
                              : (outline.unscaledCoordinatesAreInvalid ? p1.unhinted.y : p1.unscaled.y);     
                int orig2 = xDirection ? 
                                (outline.unscaledCoordinatesAreInvalid ? p2.unhinted.x : p2.unscaled.x)
                              : (outline.unscaledCoordinatesAreInvalid ? p2.unhinted.y : p2.unscaled.y);  
                if (orig1 > orig2) {
                  { TTPoint temp = p1; p1 = p2; p2 = temp; }
                  { int temp = orig1; orig1 = orig2; orig2 = temp; }}
                               
                // touchedBefore and touchedAfter bracket at least one untouched points
                while (untouched != touchedAfter) {
                  TTPoint p = getPoint (outline, 1, untouched);
                  int orig = xDirection ? 
                                 (outline.unscaledCoordinatesAreInvalid ? p.unhinted.x : p.unscaled.x)
                               : (outline.unscaledCoordinatesAreInvalid ? p.unhinted.y : p.unscaled.y);
                  
                  /* If the original positions of the two touched points are the same
                  * (on the axis considered by the instruction), then the untouched
                  * points are moved by the same amount as the touched point with the
                  * highest index was moved. The choice of the highest point seems
                  * rather arbitrary, but that is what CoolType does. */
                  
                  int oldx = p.hinted.x;
                  int oldy = p.hinted.y;
                  
                  if (orig1 == orig2) {
                    if (xDirection) {
                      p.hinted.x += (p2.hinted.x - p2.unhinted.x); }
                    else {
                      p.hinted.y += (p2.hinted.y - p2.unhinted.y); }}
                   
                  /* If an untouched point was originally between the two touched
                  * points, then its new position is interpolated from the positions of
                  * the two touched points. */
                  
                  else if (orig1 < orig && orig < orig2) {
                    if (xDirection) {
                      p.hinted.x = p1.hinted.x 
                        + F26Dot6.multiplyDivide (orig - orig1, 
                            p2.hinted.x - p1.hinted.x, 
                            orig2 - orig1); }
                    else {
                      p.hinted.y = p1.hinted.y
                        + F26Dot6.multiplyDivide (orig - orig1, 
                            p2.hinted.y - p1.hinted.y, 
                            orig2 - orig1); }}
                     
                  /* If an untouched point was originally to the left of the leftmost
                  * touched point, then it is moved by the same amount as that touched
                  * point was moved. */
                  
                  else if (orig <= orig1) {
                    if (xDirection) {
                      p.hinted.x += (p1.hinted.x - p1.unhinted.x); }
                    else {
                      p.hinted.y += (p1.hinted.y - p1.unhinted.y); }}
                      
                  /* If an untouched point was originally to the right of the rightmost
                  * touched point, then it is moved by the same amount as that touched
                  * point was moved. */

                  else { /* orig2 <= unhinted) */
                    if (xDirection) {
                      p.hinted.x += p2.hinted.x - p2.unhinted.x; }
                    else {
                      p.hinted.y += p2.hinted.y - p2.unhinted.y; }}
                    
                  if (ScalerDebugger.debugOn && debugger != null) {
                    logDetail ("   moving " + untouched + " from (" + F26Dot6.toDouble(oldx) + ", " + F26Dot6.toDouble(oldy) + ") to (" + F26Dot6.toDouble(p.hinted.x) + ", " + F26Dot6.toDouble(p.hinted.y) + ")"); }
                  
                  untouched = outline.getContourNextPoint (contour, untouched); }}}}
          break; }
        
        case 0x5d:   // DELTAP1 delta exception P1 [295]
        case 0x71:   // DELTAP2 delta exception P2 [296]
        case 0x72:   // DELTAP3 delta exception P3 [297]
        case 0x73:   // DELTAC1 delta exception C1 [298]
        case 0x74:   // DELTAC2 delta exception C2 [299]
        case 0x75: { // DELTAC3 delta exception C3 [300]
          int base;
          if (instruction == 0x72 || instruction == 0x75) {
            base = gs.getDeltaBase () + 32; }
          else if (instruction == 0x71 || instruction == 0x74) {
            base = gs.getDeltaBase () + 16; }
          else {
            base = gs.getDeltaBase (); }
            
          int pixelsPerEm = gs.getPixelsPerEm ();

          int n = stack.pop ();
          for (int i = 0; i < n; i++) {
            int pointOrCvtEntry = stack.pop ();
            int exception = stack.pop ();
            int applyAtPpem = base + ((exception >> 4) & 0xf);
            if (applyAtPpem == pixelsPerEm) {
              int magnitude = exception & 0xf;
              int steps = magnitude - ((magnitude < 8) ? 8 : 7);
              int /*26.6*/ value = steps << (6 - gs.getDeltaShift ());
              if (instruction < 0x73) {
                gs.move (getPoint (outline, gs.getZp0 (), pointOrCvtEntry), value); }
              else {
                gs.incrementCVT (pointOrCvtEntry, value); }}}

          break; }

        case 0x20: { // DUP duplicate top stack element [303]
          stack.copy (1);
          break; }
        
        case 0x21: { // POP pop top stack element [304]
          stack.pop ();
          break; }
        
        case 0x22: { // CLEAR clear the entire stack [305]
          stack.clear ();
          break; }
        
        case 0x23: { // SWAP swap the top two elements on the stack [306]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e2);
          stack.push (e1);
          break; }
        
        case 0x24: { // DEPTH return the depth of the stack [307]
          stack.pushDepth ();
          break; }
        
        case 0x25: { // CINDEX copy the indexed element to the top of the stack [308]
          stack.copy (stack.pop ());
          break; }
        
        case 0x26: { // MINDEX move the indexed element to the stop of the stack [309]
          stack.bubbleUp (stack.pop ());
          break; }
        
        case 0x8a: { // ROLL roll the top three stack elements [310]
          int a = stack.pop ();
          int b = stack.pop ();
          int c = stack.pop ();
          stack.push (b);
          stack.push (a);
          stack.push (c);
          break; }
        
        case 0x58: { // IF if test [312]
          int n = stack.pop ();
          if (n == 0) {
            // find the else or endif
            int nestedIfs = 0;
            int targetOffset = offset;
            boolean isTarget = false;
            while (targetOffset < limit && ! isTarget) {
              switch (instructionStream.getuint8 (targetOffset)) {
                case 0x58: { // IF
                  nestedIfs++;
                  break; }
                case 0x1b: { // ELSE
                  if (nestedIfs == 0) { 
                    isTarget = true; }
                  break; }
                case 0x59: { // EIF
                  if (nestedIfs == 0) {
                    isTarget = true; }
                  else {
                    nestedIfs--; }
                  break; }}
                targetOffset = skipInstruction (instructionStream, targetOffset); }
            if (isTarget) {
              offset = targetOffset; }
            else {
              throw new InvalidFontException ("incorrect IF - no ELSE or EIF in TT outline"); }}
          break; }

        
        case 0x1b: { // ELSE else [314]
          // we have reached the end of an IF branch, jump to the EIF
          int nestedIfs = 0;
          int targetOffset = offset;
          boolean isTarget = false;
          while (targetOffset < limit && ! isTarget) {
            switch (instructionStream.getuint8 (targetOffset)) {
              case 0x58: { // IF
                nestedIfs++;
                break; }
              case 0x59: { // EIF
                if (nestedIfs == 0) {
                  isTarget = true; }
                else {
                  nestedIfs--; }
                break; }}
              targetOffset = skipInstruction (instructionStream, targetOffset); }
          if (isTarget) {
            offset = targetOffset; }
          else {
            throw new InvalidGlyphException ("incorrect IF - ELSE - no EIF in TT outline"); }
          break; }
       
        case 0x59: { // EIF end if [315]
          break; }
        
        case 0x78: { // JROT jump relative on true [316]
          int b = stack.pop ();
          int off = stack.pop ();
          if (b != 0) {
            offset += off - 1; 
            if (offset < start || limit < offset) {
              throw new InvalidGlyphException ("JROF outside of hinted instruction block"); }}
          break; }
        
        case 0x1c: { // JMPR jump [318]
          int off = stack.pop ();
          offset += off - 1; 
          if (offset < start || limit < offset) {
            throw new InvalidGlyphException ("JROF outside of hinted instruction block"); }
          break; }
        
        case 0x79: { // JROF jump relative on false [319]
          int b = stack.pop ();
          int off = stack.pop ();
          if (b == 0) {
            offset += off - 1;
            if (offset < start || limit < offset) {
              throw new InvalidGlyphException ("JROF outside of hinted instruction block"); }}
          break; }
        
        case 0x50: { // LT less than [322]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e1 < e2 ? 1 : 0);
          break; }
        
        case 0x51: { // LTEQ less than or equal [323]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e1 <= e2 ? 1 : 0);
          break; }
        
        case 0x52: { // GT greater than [324]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e1 > e2 ? 1 : 0);
          break; }
        
        case 0x53: { // GTEQ greater than or equal [325]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e1 >= e2 ? 1 : 0);
          break; }
        
        case 0x54: { // EQ equal [326]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e1 == e2 ? 1 : 0);
          break; }
        
        case 0x55: { // NEQ not equal [327]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e1 != e2 ? 1 : 0);
          break; }
        
        case 0x56: { // ODD odd [328]
          int e1 = stack.pop ();
          // XXX_ERIC: see below
          stack.push (F26Dot6.isEven (gs.roundToGrid (e1, 0)) ? 0 : 1);
          break; }
        
        case 0x57: { // EVEN even [329]
          int e1 = stack.pop ();
          // XXX_ERIC: the spec "round as specified by the round_state
          // before testing". That would imply that even is defined
          // on fractional numbers, but what does that means?
          // CT uses systematically RoundToGrid
          // FT does the rounding and then tests bit 6 (the 2^0 bit)
          
          stack.push (F26Dot6.isEven (gs.roundToGrid (e1, 0)) ? 1 : 0);
          break; }
         
        case 0x5a: { // AND logical and [330]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e1 != 0 && e2 != 0 ? 1 : 0);
          break; }
        
        case 0x5b: { // OR logical or [332]
          int e2 = stack.pop ();
          int e1 = stack.pop ();
          stack.push (e1 != 0 || e2 != 0 ? 1 : 0);
          break; }
        
        case 0x5c: { // NOT not [334]
          int e1 = stack.pop ();
          stack.push (e1 == 0 ? 1 : 0);
          break; }
         
        case 0x60: { // ADD add [336]
          stack.push (F26Dot6.add (stack.pop (), stack.pop ()));
          break; }
        
        case 0x61: { // SUB subtract [337]
          int /*26.6*/ e1 = stack.pop ();
          int /*26.6*/ e2 = stack.pop ();
          stack.push (F26Dot6.subtract (e2, e1));
	      break; }
        
        case 0x62: { // DIV div [338]
          int /*26.6*/ e1 = stack.pop ();
          int /*26.6*/ e2 = stack.pop ();
          stack.push (F26Dot6.divide (e2, e1));
          break; }
        
        case 0x63: { // MUL multiply [339]
          stack.push (F26Dot6.multiply (stack.pop (), stack.pop ()));
          break; }
  
        case 0x64: { // ABS abs [340]
          stack.push (F26Dot6.abs (stack.pop ()));
          break; }
         
        case 0x65: { // NEG neg [341]
          stack.push (F26Dot6.negate (stack.pop ()));
          break; }
         
        case 0x66: { // FLOOR floor [342]
          stack.push (F26Dot6.floor (stack.pop ()));
          break; }
         
        case 0x67: { // CEILING ceiling [343]
          stack.push (F26Dot6.ceiling (stack.pop ()));
          break; }
         
        case 0x8b: { // MAX maximum of top two stack elements [344]
          stack.push (Math.max (stack.pop (), stack.pop ()));
          break; }
  
        case 0x8c: { // MIN minumum of top two stack elements [345]
          stack.push (Math.min (stack.pop (), stack.pop ()));
          break; }
  
        case 0x68:
        case 0x69:
        case 0x6a:
        case 0x6b: { // ROUND round value [347]
          int v = stack.pop ();
          stack.push (gs.round (v, gp.getEngineCompensation (instruction & 0x3)));
          break; }
        
        case 0x6c:
        case 0x6d:
        case 0x6e:
        case 0x6f: { // NROUND no rounding of value [348]
          int v = stack.pop ();
          stack.push (gs.roundOff (v, gp.getEngineCompensation (instruction & 0x3)));
          break; }
        
        case 0x2c: { // FDEF function definition [350]
          int f = stack.pop ();

          if (f < 0 || f > functionDefs.instructionStreams.length) {
            throw new InvalidGlyphException ("Invalid function identifier in FDEF (" + f + ")"); }
            
          int targetOffset = offset;
          boolean isTarget = false;
          while (targetOffset < limit && ! isTarget) {
            switch (instructionStream.getuint8 (targetOffset)) {
              case 0x2c:   // FDEF
              case 0x89: { // IDEF
                throw new InvalidGlyphException ("Nested FDEF/IDEF definitions"); }
              case 0x2d: { // ENDF
                isTarget = true;
                break; }}
              targetOffset = skipInstruction (instructionStream, targetOffset); }

          if (! isTarget) {
            throw new InvalidGlyphException ("FDEF without IDEF in TT outline"); }

          functionDefs.setFunction (f, instructionStream, offset, targetOffset - 1);
          offset = targetOffset;
          break; }
        
        case 0x2d: { // ENDF end function definition [351]
          throw new InvalidGlyphException ("dangling ENDF"); }
        
        case 0x2b: { // CALL call function [352]
          int f = stack.pop ();

          if (f < 0 || f > functionDefs.instructionStreams.length) {
            throw new InvalidGlyphException ("Invalid function identifier in CALL (" + f + ")"); }
          if (functionDefs.limit [f] == -1) {
            throw new InvalidGlyphException ("Undefined function identifier in CALL (" + f + ")"); }
            
          runWithoutInit (outline,
               functionDefs.instructionStreams [f],
               functionDefs.start [f],
               functionDefs.limit [f]);
          break; }
        
        case 0x2a: { // LOOPCALL loop and call function [353]
          int f = stack.pop ();
          int count = stack.pop ();

          if (f < 0 || f > functionDefs.instructionStreams.length) {
            throw new InvalidGlyphException ("Invalid function identifier in LOOPCALL (" + f + ")"); }
          if (functionDefs.limit [f] == -1) {
            throw new InvalidGlyphException ("Undefined function identifier in LOOPCALL (" + f + ")"); }
            
          for (int i = 0; i < count; i++) {
            runWithoutInit (outline, 
                 functionDefs.instructionStreams [f],
                 functionDefs.start [f],
                 functionDefs.limit [f]); }
          break; }
        
        case 0x89: { // IDEF instruction definition [354]
          int opcode = stack.pop ();
          
          if (opcode < 0 || opcode > 0xff) {
            throw new InvalidGlyphException ("Illegal IDEF opcode (" + opcode + ")"); }
          
          int targetOffset = offset;
          boolean isTarget = false;
          while (targetOffset < limit && ! isTarget) {
            switch (instructionStream.getuint8 (targetOffset)) {
              case 0x2c:   // FDEF
              case 0x89: { // IDEF
                throw new InvalidGlyphException ("Nested FDEF/IDEF definitions"); }
              case 0x2d: { // ENDF
                isTarget = true;
                break; }}
              targetOffset = skipInstruction (instructionStream, targetOffset); }
          if (isTarget) {
            offset = targetOffset; }
          else {
            throw new InvalidGlyphException ("IDEF without ENDF TT outline"); }

          instructionDefs.setFunction (opcode, instructionStream, 
                                       offset, targetOffset - 1);
          offset = targetOffset;
          break; }
        
        case 0x4f: { // DEBUG debug call [355]
          stack.pop ();
          break; }
        
        case 0x88: { // GETINFO get information [356]
          int selector = stack.pop ();
          int answer = 0;
          boolean rotated = false; // TODO
          boolean stretched = false; // TODO
          boolean greyscale = false; // TODO
          if ((selector & 0x0001) != 0) {
            answer |= 37; }
          
          if ((selector & 0x0002) != 0) {
            answer |= rotated ? 0x100 : 0; }
          
          if ((selector & 0x0004) != 0) {
            answer |= stretched ? 0x200 : 0; }
          
          if ((selector & 0x0010) != 0) {
            answer |= greyscale ? 0x1000 : 0; }
          
          stack.push (answer);
          break; }
        
        default: { // unknown, check in instructionDefs
          if (instructionDefs.instructionStreams [instruction] != null) { 
            runWithoutInit (outline, 
                 instructionDefs.instructionStreams [instruction],
                 instructionDefs.start [instruction],
                 instructionDefs.limit [instruction]); }}}
    

	   if (ScalerDebugger.debugOn && debugger != null) {
	     logResult (); }}}
      
      catch (InvalidGlyphException f) {
    	  throw(f);
      } catch (InvalidFontException e) {
    	  throw new InvalidGlyphException(e);
      }
      
  }
  
  private int skipInstruction (OTByteArray instructionStream, int offset) 
  throws InvalidFontException {
    
    int instruction = instructionStream.getuint8 (offset);
    offset++;
    
    switch (instruction) {
      case 0x40: { // NPUSHB push n bytes [189]
        int nBytes = instructionStream.getuint8 (offset);
        offset++;
        offset += nBytes;
        break; }
      case 0x41: { // NPUSHW push n words [190]
        int nWords = instructionStream.getuint8 (offset);
        offset++;
        offset += 2 * nWords;
        break; }
      case 0xB0:
      case 0xB1:
      case 0xB2:
      case 0xB3:
      case 0xB4:
      case 0xB5:
      case 0xB6:
      case 0xB7: { // PUSHB push bytes [191]
        offset += (instruction & 0x7) + 1;
        break; }
      case 0xB8:
      case 0xB9:
      case 0xBA:
      case 0xBB:
      case 0xBC:
      case 0xBD:
      case 0xBE:
      case 0xBF: { // PUSHW push words [192]
        offset += 2 * ((instruction & 0x7) + 1);
        break; }}
    
    return offset;
  }
  
  //---------------------------------------------------------------- logging ---

  static class Instr {
    int opcode;
    String name;
    int stackPop;
    int stackPush;

    public Instr (int opcode, String name, int stackPop, int stackPush) {
      this.opcode = opcode;
      this.name = name;
      this.stackPop = stackPop;
      this.stackPush = stackPush;
    }
  }

  static Instr[] instructions = new Instr[] {
      new Instr (0x00, "SetVectorsToCoordAxis {x}", 0, 0),
      new Instr (0x01, "SetVectorsToCoordAxis {y}", 0, 0),
      new Instr (0x02, "SetPVToCoordAxis {x}", 0, 0),
      new Instr (0x03, "SetPVToCoordAxis {y}", 0, 0),
      new Instr (0x04, "SetFVToCoordAxis {x}", 0, 0),
      new Instr (0x05, "SetFVToCoordAxis {y}", 0, 0),
      new Instr (0x06, "SetPVToLine {para}", 2, 0),
      new Instr (0x07, "SetPVToLine {perp}", 2, 0),
      new Instr (0x08, "SetFVToLine {para}", 2, 0),
      new Instr (0x09, "SetFVToLine {para}", 2, 0),
      new Instr (0x0a, "SetPVFromStack", 2, 0),
      new Instr (0x0b, "SetFVFromStack", 2, 0),
      new Instr (0x0c, "GetPV", 0, 2),
      new Instr (0x0d, "GetFV", 0, 2),
      new Instr (0x0e, "SetFVtoPV", 0, 0),
      new Instr (0x0f, "movepointtoInterSECTion", 5, 0),
      new Instr (0x10, "SetRP0", 1, 0),
      new Instr (0x11, "SetRP1", 1, 0),
      new Instr (0x12, "SetRP2", 1, 0),
      new Instr (0x13, "SetZP0", 1, 0),
      new Instr (0x14, "SetZP1", 1, 0),
      new Instr (0x15, "SetZP2", 1, 0),
      new Instr (0x16, "SetZPS", 1, 0),
      new Instr (0x17, "SetLOOPvariable", 1, 0),
      new Instr (0x18, "RoundToGrid", 0, 0),
      new Instr (0x19, "RoundToHalfGrid", 0, 0),
      new Instr (0x1a, "SetMinimumDistance", 1, 0),
      new Instr (0x1b, "ELSE", 0, 0),
      new Instr (0x1c, "JuMPRelative", 1, 0),
      new Instr (0x1d, "SetCVTCutIn", 1, 0),
      new Instr (0x1e, "SetSingleWidthCutIn", 1, 0),
      new Instr (0x1f, "SetSingleWdith", 1, 0),
      new Instr (0x20, "DUP", 0, 1),
      new Instr (0x21, "POP", 0, 0),
      new Instr (0x22, "CLEAR", 0, 0),
      new Instr (0x23, "SWAP", 2, 2),
      new Instr (0x24, "DEPTH", 0, 1),
      new Instr (0x25, "CINDEX", 1, 1),
      new Instr (0x26, "MINDEX", 1, 1),
      new Instr (0x27, "ALIGNPoinTS", 2, 0),
      new Instr (0x28, "<idef>", 0, 0),
      new Instr (0x29, "UnTouchPoint", 1, 0),
      new Instr (0x2a, "LOOPCALL", 2, 0),
      new Instr (0x2b, "CALL", 1, 0),
      new Instr (0x2c, "FDEF", 1, 0),
      new Instr (0x2d, "ENDF", 0, 0),
      new Instr (0x2e, "MoveDirectAbsolutePoint {noround}", 1, 0),
      new Instr (0x2f, "MoveDirectAbsolutePoint {round}", 1, 0),
      new Instr (0x30, "InterpolateUntouchedPoints {y}", 0, 0),
      new Instr (0x31, "InterpolateUntouchedPoints {x}", 0, 0),
      new Instr (0x32, "ShiftPoint {zp1-rp2}", 0, 0),
      new Instr (0x33, "ShiftPoint {zp0-rp1}", 0, 0),
      new Instr (0x34, "SHiftContour {zp1-rp2}", 0, 0),
      new Instr (0x35, "SHiftContour {zp0-rp1}", 0, 0),
      new Instr (0x36, "SHiftZone {zp1-rp2}", 0, 0),
      new Instr (0x37, "SHiftZone {zp0-rp1}", 0, 0),
      new Instr (0x38, "SHiftpointbyPIXelamount", 0, 0),
      new Instr (0x39, "InterPolate", 0, 0),
      new Instr (0x3a, "MoveStackIndirectRelativePoint", 2, 0),
      new Instr (0x3b, "MoveStackIndirectRelativePoint {setRp0}", 2, 0),
      new Instr (0x3c, "ALIGNRelativePoint", 0, 0),
      new Instr (0x3d, "RoundToDoubleGrid", 0, 0),
      new Instr (0x3e, "MoveIndirectAbsolutePoint {noround}", 2, 0),
      new Instr (0x3f, "MoveIndirectAbsolutePoint {round}", 2, 0),
      new Instr (0x40, "NPUSHB", 0, 0),
      new Instr (0x41, "NPUSHW", 0, 0),
      new Instr (0x42, "WriteStore", 2, 0),
      new Instr (0x43, "ReadStore", 1, 1),
      new Instr (0x44, "WriteCVTPixel", 2, 0),
      new Instr (0x45, "ReadCVT", 1, 1),
      new Instr (0x46, "GetCoordinate (proj)", 1, 1),
      new Instr (0x47, "GetCoordinate (unhinted)", 1, 1),
      new Instr (0x48, "SetCoordinateFromStack", 2, 0),
      new Instr (0x49, "MeasureDistance (hinted)", 2, 1),
      new Instr (0x4a, "MeasureDistance (original)", 2, 1),
      new Instr (0x4b, "MeasurePPEM", 0, 1),
      new Instr (0x4c, "MeasurePointSize", 0, 1),
      new Instr (0x4d, "FLIPON", 0, 0),
      new Instr (0x4e, "FLIPOFF", 0, 0),
      new Instr (0x4f, "DEBUG", 1, 0),
      new Instr (0x50, "LessThan", 2, 1),
      new Instr (0x51, "LessThanEQ", 2, 1),
      new Instr (0x52, "GreaterThan", 2, 1),
      new Instr (0x53, "GreateThanEQ", 2, 1),
      new Instr (0x54, "EQual", 2, 1),
      new Instr (0x55, "NotEQual", 2, 1),
      new Instr (0x56, "ODD", 1, 1),
      new Instr (0x57, "EVEN", 1, 1),
      new Instr (0x58, "IF", 1, 0),
      new Instr (0x59, "EIF", 0, 0),
      new Instr (0x5a, "AND", 2, 1),
      new Instr (0x5b, "OR", 2, 1),
      new Instr (0x5c, "NOT", 1, 1),
      new Instr (0x5d, "DELTAP1", 1, 0),
      new Instr (0x5e, "SetDeltaBase", 1, 0),
      new Instr (0x5f, "SetDeltaShift", 1, 0),
      new Instr (0x60, "ADD", 2, 1),
      new Instr (0x61, "SUB", 2, 1),
      new Instr (0x62, "DIV", 2, 1),
      new Instr (0x63, "MUL", 2, 1),
      new Instr (0x64, "ABS", 1, 1),
      new Instr (0x65, "NEG", 1, 1),
      new Instr (0x66, "FLOOR", 1, 1),
      new Instr (0x67, "CEILING", 1, 1),
      new Instr (0x68, "ROUND {comp0}", 1, 1),
      new Instr (0x69, "ROUND {comp1}", 1, 1),
      new Instr (0x6a, "ROUND {comp2}", 1, 1),
      new Instr (0x6b, "ROUND {comp3}", 1, 1),
      new Instr (0x6c, "NROUND {comp0}", 1, 1),
      new Instr (0x6d, "NROUND {comp1}", 1, 1),
      new Instr (0x6e, "NROUND {comp2}", 1, 1),
      new Instr (0x6f, "NROUND {comp3}", 1, 1),
      new Instr (0x70, "WriteCVTFunits", 2, 0),
      new Instr (0x71, "DELTAP2", 1, 0),
      new Instr (0x72, "DELTAP3", 1, 0),
      new Instr (0x73, "DELTAC1", 1, 0),
      new Instr (0x74, "DELTAC2", 1, 0),
      new Instr (0x75, "DELTAC3", 1, 0),
      new Instr (0x76, "SuperROUND", 1, 0),
      new Instr (0x77, "S45ROUND", 1, 0),
      new Instr (0x78, "JumpRelativeOnTrue", 2, 0),
      new Instr (0x79, "JumpRelativeOnFalse", 2, 0),
      new Instr (0x7a, "RoundOFF", 0, 0),
      new Instr (0x7b, "<idef>", 0, 0),
      new Instr (0x7c, "RoundUpToGrid", 0, 0),
      new Instr (0x7d, "RoundDownToGrid", 0, 0),
      new Instr (0x7e, "SetANGleWeight", 1, 0),
      new Instr (0x7f, "AdjustAngle", 0, 0),
      new Instr (0x80, "FliPPoinT", 0, 0),
      new Instr (0x81, "FLIPRanGeON", 2, 0),
      new Instr (0x82, "FLIPRanGeOFF", 2, 0),
      new Instr (0x83, "<idef>", 0, 0),
      new Instr (0x84, "<idef>", 0, 0),
      new Instr (0x85, "SCANCTRL", 1, 0),
      new Instr (0x86, "SetDPVToLine {para}", 2, 0),
      new Instr (0x87, "SetDPVToLine {perp}", 2, 0),
      new Instr (0x88, "GETINFO", 1, 1),
      new Instr (0x89, "IDEF", 1, 0),
      new Instr (0x8a, "ROLL", 3, 3),
      new Instr (0x8b, "MAX", 2, 1),
      new Instr (0x8c, "MIN", 2, 1),
      new Instr (0x8d, "SCANTYPE", 1, 0),
      new Instr (0x8e, "INSTCRL", 2, 0),
      new Instr (0x8f, "<idef>", 0, 0),
      new Instr (0x90, "<idef>", 0, 0),
      new Instr (0x91, "<idef>", 0, 0),
      new Instr (0x92, "<idef>", 0, 0),
      new Instr (0x93, "<idef>", 0, 0),
      new Instr (0x94, "<idef>", 0, 0),
      new Instr (0x95, "<idef>", 0, 0),
      new Instr (0x96, "<idef>", 0, 0),
      new Instr (0x97, "<idef>", 0, 0),
      new Instr (0x98, "<idef>", 0, 0),
      new Instr (0x99, "<idef>", 0, 0),
      new Instr (0x9a, "<idef>", 0, 0),
      new Instr (0x9b, "<idef>", 0, 0),
      new Instr (0x9c, "<idef>", 0, 0),
      new Instr (0x9d, "<idef>", 0, 0),
      new Instr (0x9e, "<idef>", 0, 0),
      new Instr (0x9f, "<idef>", 0, 0),
      new Instr (0xa0, "<idef>", 0, 0),
      new Instr (0xa1, "<idef>", 0, 0),
      new Instr (0xa2, "<idef>", 0, 0),
      new Instr (0xa3, "<idef>", 0, 0),
      new Instr (0xa4, "<idef>", 0, 0),
      new Instr (0xa5, "<idef>", 0, 0),
      new Instr (0xa6, "<idef>", 0, 0),
      new Instr (0xa7, "<idef>", 0, 0),
      new Instr (0xa8, "<idef>", 0, 0),
      new Instr (0xa9, "<idef>", 0, 0),
      new Instr (0xaa, "<idef>", 0, 0),
      new Instr (0xab, "<idef>", 0, 0),
      new Instr (0xac, "<idef>", 0, 0),
      new Instr (0xad, "<idef>", 0, 0),
      new Instr (0xae, "<idef>", 0, 0),
      new Instr (0xaf, "<idef>", 0, 0),
      new Instr (0xb0, "PUSHB1", 0, 1),
      new Instr (0xb1, "PUSHB2", 0, 2),
      new Instr (0xb2, "PUSHB3", 0, 3),
      new Instr (0xb3, "PUSHB4", 0, 4),
      new Instr (0xb4, "PUSHB5", 0, 5),
      new Instr (0xb5, "PUSHB6", 0, 6),
      new Instr (0xb6, "PUSHB7", 0, 7),
      new Instr (0xb7, "PUSHB8", 0, 8),
      new Instr (0xb8, "PUSHW1", 0, 1),
      new Instr (0xb9, "PUSHW2", 0, 2),
      new Instr (0xba, "PUSHW3", 0, 3),
      new Instr (0xbb, "PUSHW4", 0, 4),
      new Instr (0xbc, "PUSHW5", 0, 5),
      new Instr (0xbd, "PUSHW6", 0, 6),
      new Instr (0xbe, "PUSHW7", 0, 7),
      new Instr (0xbf, "PUSHW8", 0, 8),
      new Instr (0xc0, "MoveDirectRelativePoint {comp0}", 1, 0),
      new Instr (0xc1, "MoveDirectRelativePoint {comp1}", 1, 0),
      new Instr (0xc2, "MoveDirectRelativePoint {comp2}", 1, 0),
      new Instr (0xc3, "MoveDirectRelativePoint {comp3}", 1, 0),
      new Instr (0xc4, "MoveDirectRelativePoint {round, comp0}", 1, 0),
      new Instr (0xc5, "MoveDirectRelativePoint {round, comp1}", 1, 0),
      new Instr (0xc6, "MoveDirectRelativePoint {round, comp2}", 1, 0),
      new Instr (0xc7, "MoveDirectRelativePoint {round, comp3}", 1, 0),
      new Instr (0xc8, "MoveDirectRelativePoint {keepMin, comp0}", 1, 0),
      new Instr (0xc9, "MoveDirectRelativePoint {keepMin, comp1}", 1, 0),
      new Instr (0xca, "MoveDirectRelativePoint {keepMin, comp2}", 1, 0),
      new Instr (0xcb, "MoveDirectRelativePoint {keepMin, comp3}", 1, 0),
      new Instr (0xcc, "MoveDirectRelativePoint {keepMin, round, comp0}", 1, 0),
      new Instr (0xcd, "MoveDirectRelativePoint {keepMin, round, comp1}", 1, 0),
      new Instr (0xce, "MoveDirectRelativePoint {keepMin, round, comp2}", 1, 0),
      new Instr (0xcf, "MoveDirectRelativePoint {keepMin, round, comp3}", 1, 0),
      new Instr (0xd0, "MoveDirectRelativePoint {setRp0, comp0}", 1, 0),
      new Instr (0xd1, "MoveDirectRelativePoint {setRp0, comp1}", 1, 0),
      new Instr (0xd2, "MoveDirectRelativePoint {setRp0, comp2}", 1, 0),
      new Instr (0xd3, "MoveDirectRelativePoint {setRp0, comp3}", 1, 0),
      new Instr (0xd4, "MoveDirectRelativePoint {setRp0, round, comp0}", 1, 0),
      new Instr (0xd5, "MoveDirectRelativePoint {setRp0, round, comp1}", 1, 0),
      new Instr (0xd6, "MoveDirectRelativePoint {setRp0, round, comp2}", 1, 0),
      new Instr (0xd7, "MoveDirectRelativePoint {setRp0, round, comp3}", 1, 0),
      new Instr (0xd8, "MoveDirectRelativePoint {setRp0, keepMin, comp0}", 1, 0),
      new Instr (0xd9, "MoveDirectRelativePoint {setRp0, keepMin, comp1}", 1, 0),
      new Instr (0xda, "MoveDirectRelativePoint {setRp0, keepMin, comp2}", 1, 0),
      new Instr (0xdb, "MoveDirectRelativePoint {setRp0, keepMin, comp3}", 1, 0),
      new Instr (0xdc, "MoveDirectRelativePoint {setRp0, keepMin, round, comp0}", 1, 0),
      new Instr (0xdd, "MoveDirectRelativePoint {setRp0, keepMin, round, comp1}", 1, 0),
      new Instr (0xde, "MoveDirectRelativePoint {setRp0, keepMin, round, comp2}", 1, 0),
      new Instr (0xdf, "MoveDirectRelativePoint {setRp0, keepMin, round, comp3}", 1, 0),
      new Instr (0xe0, "MoveIndirectRelativePoint {comp0}", 2, 0),
      new Instr (0xe1, "MoveIndirectRelativePoint {comp1}", 2, 0),
      new Instr (0xe2, "MoveIndirectRelativePoint {comp2}", 2, 0),
      new Instr (0xe3, "MoveIndirectRelativePoint {comp3}", 2, 0),
      new Instr (0xe4, "MoveIndirectRelativePoint {round, comp0}", 2, 0),
      new Instr (0xe5, "MoveIndirectRelativePoint {round, comp1}", 2, 0),
      new Instr (0xe6, "MoveIndirectRelativePoint {round, comp2}", 2, 0),
      new Instr (0xe7, "MoveIndirectRelativePoint {round, comp3}", 2, 0),
      new Instr (0xe8, "MoveIndirectRelativePoint {keepMin, comp0}", 2, 0),
      new Instr (0xe9, "MoveIndirectRelativePoint {keepMin, comp1}", 2, 0),
      new Instr (0xea, "MoveIndirectRelativePoint {keepMin, comp2}", 2, 0),
      new Instr (0xeb, "MoveIndirectRelativePoint {keepMin, comp3}", 2, 0 ),
      new Instr (0xec, "MoveIndirectRelativePoint {keepMin, round, comp0}", 2, 0),
      new Instr (0xed, "MoveIndirectRelativePoint {keepMin, round, comp1}", 2, 0),
      new Instr (0xee, "MoveIndirectRelativePoint {keepMin, round, comp2}", 2, 0),
      new Instr (0xef, "MoveIndirectRelativePoint {keepMin, round, comp3}", 2, 0),
      new Instr (0xf0, "MoveIndirectRelativePoint {setRp0, comp0}", 2, 0),
      new Instr (0xf1, "MoveIndirectRelativePoint {setRp0, comp1}", 2, 0),
      new Instr (0xf2, "MoveIndirectRelativePoint {setRp0, comp2}", 2, 0),
      new Instr (0xf3, "MoveIndirectRelativePoint {setRp0, comp3}", 2, 0),
      new Instr (0xf4, "MoveIndirectRelativePoint {setRp0, round, comp0}", 2, 0),
      new Instr (0xf5, "MoveIndirectRelativePoint {setRp0, round, comp1}", 2, 0),
      new Instr (0xf6, "MoveIndirectRelativePoint {setRp0, round, comp2}", 2, 0),
      new Instr (0xf7, "MoveIndirectRelativePoint {setRp0, round, comp3}", 2, 0),
      new Instr (0xf8, "MoveIndirectRelativePoint {setRp0, keepMin, comp0}", 2, 0),
      new Instr (0xf9, "MoveIndirectRelativePoint {setRp0, keepMin, comp1}", 2, 0),
      new Instr (0xfa, "MoveIndirectRelativePoint {setRp0, keepMin, comp2}", 2, 0),
      new Instr (0xfb, "MoveIndirectRelativePoint {setRp0, keepMin, comp3}", 2, 0),
      new Instr (0xfc, "MoveIndirectRelativePoint {setRp0, keepMin, round, comp0}", 2, 0),
      new Instr (0xfd, "MoveIndirectRelativePoint {setRp0, keepMin, round, comp1}", 2, 0),
      new Instr (0xfe, "MoveIndirectRelativePoint {setRp0, keepMin, round, comp2}", 2, 0),
      new Instr (0xff, "MoveIndirectRelativePoint {setRp0, keepMin, round, comp3}", 2, 0),
  };
  
  
  private ScalerDebugger debugger;
  public void setDebugger (ScalerDebugger debugger) {
    this.debugger = debugger; 
  }
  
  private Instr currentInstr;
  private StringBuffer sb = new StringBuffer ();
  private StringBuffer sbDetails = new StringBuffer ();

  private void logInstruction (int offset, int opcode) {
    sb.setLength (0);
    sbDetails.setLength (0);
    currentInstr = instructions [opcode];
    sb.append (offset);
    sb.append (" ");
    sb.append (currentInstr.name);
    sb.append (" (");
    logStack (sb, currentInstr.stackPop);
    sb.append (")"); 
  }

  private void logResult () {
    if (currentInstr.stackPush != 0) {
      sb.append (" -> ");
      logStack (sb, currentInstr.stackPush); }
    sb.append ("\n");
    sb.append (sbDetails); 
    debugger.ttInterpLog (sb.toString ());
  }   

  private void logDetail (String s) {
    sbDetails.append ("  ");
    sbDetails.append (s);
    sbDetails.append ("\n");
  }

  private void logDetailNoLn (String s) {
    sbDetails.append (s);
  }

  private void logStack (StringBuffer sb, int nbElems) {
    String prefix = "";
    for (int i = 0; i < nbElems; i++) {
      sb.append (prefix + Integer.toHexString (stack.peek (i))); 
      prefix = ", "; }
  }
}

