/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.httpcache.api;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.osgi.annotation.versioning.ProviderType;

@ProviderType
public interface CacheHandler {

    /**
     * Tri-State type.
     */
    public static enum TRI_STATE {
        /* Answer is yes, stop processing */
        YES,
        /* Answer is no, stop processing */
        NO,
        /* Ask next handler */
        NEXT_HANDLER,
    }

    /**
     * Used upon receiving a request to determine whether its response is suitable
     * to be cached.
     *
     * The default handler will try caching the response if the method is GET and
     * the request has no parameters.
     *
     * @param request request
     * @return {@link TRI_STATE#YES} to try caching the response;
     *         {@link TRI_STATE#NO} to skip this request;
     *         {@link TRI_STATE#NEXT_HANDLER} to ask the next handler
     */
    public TRI_STATE onReceive(HttpServletRequest request);

    /**
     * Used after a cache hit to give a cache handler the opportunity to modify
     * the response headers being delivered, or forbid delivering the cache file.
     *
     * The default handler will unconditionally deliver a cache hit.
     *
     * @param key cache key
     * @param headers response headers
     * @param response response
     *
     * @return {@link TRI_STATE#YES} to deliver the cache file;
     *         {@link TRI_STATE#NO} to forbid delivering;
     *         {@link TRI_STATE#NEXT_HANDLER} to ask the next handler
     */
    public TRI_STATE onDeliver(String key, Headers headers, HttpServletResponse response);

    /**
     * Used upon receiving the response to determine whether the response is suitable
     * for caching.
     *
     * The default handler will create a cache entry if the response status
     * is {@link HttpServletResponse#SC_OK} and if no cache control response
     * header denies caching.
     *
     * @param status response status
     * @param headers response headers that will be stored along with the response
     * @param response servlet response
     *
     * @return {@link TRI_STATE#YES} to create an entry in the cache
     *         {@link TRI_STATE#NO} to forbid creating;
     *         {@link TRI_STATE#NEXT_HANDLER} to ask the next handler
     */
    public TRI_STATE onFetch(int status, Headers headers, HttpServletResponse response);
}
