/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.rest.converter.siren;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.sling.api.resource.Resource;
import org.json.JSONException;
import org.json.JSONObject;

import com.adobe.granite.rest.Constants;
import com.adobe.granite.rest.converter.ResourceConverterContext;
import com.adobe.granite.rest.converter.ResourceConverterException;
import com.adobe.granite.rest.filter.Filter;
import com.adobe.granite.rest.filter.impl.FilteredIterator;
import com.adobe.granite.rest.utils.Resources;
import com.adobe.reef.siren.Entity;
import com.adobe.reef.siren.Link;
import com.adobe.reef.siren.builder.BuilderException;

/**
 * {@code AbstractCollectionConverter} is a base implementation of {@link AbstractSirenConverter} for pageable
 * resources. It provides functionality additional functionality for paging.
 * {@link com.adobe.granite.rest.converter.ResourceConverter} implementations are encouraged to extend from this
 * abstract base class.
 */
public abstract class AbstractPageableSirenConverter extends AbstractSirenConverter {

    public AbstractPageableSirenConverter(Resource resource) {
        super(resource);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected List<Link> getLinks(ResourceConverterContext context, boolean isChild) throws BuilderException,
        ResourceConverterException {
        List<Link> links = super.getLinks(context, isChild);
        if (!isChild) {
            Resource parent = resource.getParent();
            if (parent != null) {
                links.add(getLink(AbstractSirenConverter.REL_PARENT,
                    buildURL(context, parent.getPath(), Constants.EXT_JSON), null));
            } else {
                links.add(getLink(AbstractSirenConverter.REL_PARENT,
                    buildURL(context, context.getRootContextPath(), Constants.EXT_JSON), null));
            }
        }
        return links;
    }

    /**
     * Returns an Entity object for the provided {@link org.apache.sling.api.resource.Resource child}.
     * @param context Converter context
     * @param child Child resource to get entity for
     * @return Siren entity
     * @throws ResourceConverterException If an error occurs during conversion
     */
    protected abstract Entity getEntity(ResourceConverterContext context, Resource child)
        throws ResourceConverterException;

    @Override
    protected List<Entity> getEntities(ResourceConverterContext context, Iterator<Resource> children)
        throws BuilderException {
        List<Entity> entities = new LinkedList<Entity>();
        int offset = context.getOffset();
        int limit = context.getLimit();
        log.debug("offset = " + offset + ", limit = " + limit);
        int i = 0;
        int n = 0;

        Filter<Resource> filter = context.getFilter();
        if (filter != null) {
            children = new FilteredIterator<Resource>(children, filter);
        }

        for (Iterator<Resource> it = children; it.hasNext(); ++i) {
            Resource childResource = it.next();
            if (i < offset) {
                continue;
            }
            if (n >= limit) {
                break;
            }
            try {
                Entity entity = getEntity(context, childResource);
                if (entity != null) {
                    entities.add(entity);
                    ++n;
                }
            } catch (ResourceConverterException e) {
                log.error("A conversion error occurred for resource " + childResource, e);
            }
        }
        return entities;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Map<String, Object> getProperties(ResourceConverterContext context, boolean isChild) {
        Map<String, Object> props = super.getProperties(context, isChild);
        if (!isChild) {
            JSONObject pagingInfo = new JSONObject();
            try {
                pagingInfo.put("limit", context.getLimit());
                pagingInfo.put("offset", context.getOffset());
                if (context.getModelFilters().isEmpty()) {
                    pagingInfo.put("total", Resources.getSize(listChildren().iterator(), context.getFilter()));
                } else {
                    pagingInfo.put("total",
                        Resources.getSize(
                            filterChildren(context.getModelFilters(), context.getOffset(), context.getLimit())
                                .iterator(), context.getFilter()));

                }
                props.put(AbstractSirenConverter.PREFIX_SRN + "paging", pagingInfo);
            } catch (JSONException e) {
                log.error("Could not add paging info to properties", e);
            }
        }
        return props;
    }
}
