/* File: CertUtils.java
*  
* ****************************************************************************
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2006 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
* ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.credentials.impl.utils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.DSAPrivateKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateKeySpec;

import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;

import com.adobe.internal.pdftoolkit.core.credentials.PrivateKeyHolder;
import com.adobe.internal.pdftoolkit.core.credentials.PrivateKeyHolderFactory;

/**
 * A collection of utility methods related to cert creation and conversion
 * @author mdharan
 *
 */
public class CertUtils 
{
	private static final String RSA = "RSA" ;
	private static final String DSA = "DSA";
	private CertUtils() {
		super();
	}

	/**
	 * Creates a JCE X509 certificate from a DER encoded byte array representing the certificate.
	 * @param certData
	 * @return java.security.cert.Certificate
	 * @throws java.security.cert.CertificateException 
	 * @throws IOException 
	 */
	public static java.security.cert.Certificate importCertificate(byte[] certData) throws java.security.cert.CertificateException, IOException 
	{
	
		ByteArrayInputStream dataStream = new ByteArrayInputStream(certData);
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        java.security.cert.Certificate cert = cf.generateCertificate(dataStream);
        dataStream.close();
        return cert; 
       
    }
	
	
	/**
	 * Creates a JCE X509 certificate chain from a DER encoded byte array representing the cert chain
	 * @param certData
	 * @return certificate chain
	 * @throws java.security.cert.CertificateException
	 * @throws IOException
	 */
	public static java.security.cert.Certificate[] importCertificateChain(byte[][] certData) throws java.security.cert.CertificateException, IOException 
	{
		
		if (certData != null)
		{
			Certificate[] jceCertChain = new Certificate[certData.length];
			for (int i = 0; i <certData.length; i++)
			{
				jceCertChain[i] = importCertificate(certData[i]);
			}
			return jceCertChain;
		}
		return null;
	
       
    }
	
	/**
	 * Create a JCE PrivateKey given a DER encoded set of bytes representing the key.
	 * 
	 * @param encodedPriKey
	 * @param keyWrappingFormat
	 * @return PrivateKey
	 * @throws NoSuchAlgorithmException
	 * @throws InvalidKeySpecException
	 */
	public static PrivateKey createJCEPrivateKey(byte[] encodedPriKey, String keyWrappingFormat) throws NoSuchAlgorithmException, InvalidKeySpecException
	{
		
		PKCS8EncodedKeySpec pvtKeySpec = new PKCS8EncodedKeySpec(encodedPriKey);
		
		PrivateKey privateKey = null;
		
		if (keyWrappingFormat.equals(RSA))
		{
			KeyFactory keyFactory = KeyFactory.getInstance(RSA);
			return keyFactory.generatePrivate(pvtKeySpec);
		}
		else if (keyWrappingFormat.equals(DSA))
		{
			KeyFactory keyFactory = KeyFactory.getInstance(keyWrappingFormat);
			privateKey = keyFactory.generatePrivate(pvtKeySpec);
		}
		
		return privateKey;
	}
	  /**
	   * Create a JCE PrivateKey given a DER encoded set of bytes representing the key.
	   * 
	   * @param encodedPriKey
	   * @param keyWrappingFormat
	   * @return JCEPrivateKeyHolder
	   * @throws NoSuchAlgorithmException
	   * @throws InvalidKeySpecException
	   */
	  public static PrivateKeyHolder createJCEPrivateKeyHolder(byte[] encodedPriKey, String keyWrappingFormat) throws NoSuchAlgorithmException, InvalidKeySpecException
	  {
	    PKCS8EncodedKeySpec pvtKeySpec = new PKCS8EncodedKeySpec(encodedPriKey);

	    KeyFactory keyFactory = null;
	    PrivateKey privateKey = null;

	    if (keyWrappingFormat.equals(RSA))
	    {
	      keyFactory = KeyFactory.getInstance(RSA);
	      privateKey = keyFactory.generatePrivate(pvtKeySpec);
	    }
	    else if (keyWrappingFormat.equals(DSA))
	    {
	      keyFactory = KeyFactory.getInstance(keyWrappingFormat);
	      privateKey = keyFactory.generatePrivate(pvtKeySpec);
	    }

	    if (keyFactory != null && privateKey != null) {
	      PrivateKeyHolderFactory factory = PrivateKeyHolderFactory.newInstance();
	      return factory.createPrivateKey(privateKey, keyFactory.getProvider().getName());
	    }
	    else {
	      return null;
	    }
	  }

	/**
	 * Converts a java Certificate Chain to a der encoded byte array.
	 * @param certChain
	 * @return byte[][] representing the der cert chain
	 * @throws CertificateEncodingException
	 * @throws CertificateException
	 */
	public static byte[][] buildDERCertChain(Certificate[] certChain) throws CertificateEncodingException, CertificateException
	{

		if (certChain != null)
		{
			byte[][] derCertChain = new byte[certChain.length][];
			for (int i = 0; i < certChain.length; i++)
			{
				java.security.cert.X509Certificate javaCert = (java.security.cert.X509Certificate)certChain[i];
				derCertChain[i] = javaCert.getEncoded();
			}
			return derCertChain;
		}
		return null;
	}
	

	/**
	 * Computes the size of a JCE PrivateKey
	 * @param privateKey
	 * @return size of the JCE PrivateKey
	 * @throws NoSuchAlgorithmException
	 * @throws InvalidKeySpecException
	 */
	public static int computeJavaPrivateKeySize(PrivateKey privateKey) throws NoSuchAlgorithmException, InvalidKeySpecException 
	{
		BigInteger modulus = null;
		String algorithm = privateKey.getAlgorithm();
		KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
		if (algorithm.equals(RSA))
		{
				RSAPrivateKeySpec rsaKeySpec = keyFactory.getKeySpec(privateKey, RSAPrivateKeySpec.class);
				modulus = rsaKeySpec.getModulus();
		}
		if (algorithm.equals(DSA))
		{
				DSAPrivateKeySpec dsaKeySpec = keyFactory.getKeySpec(privateKey, DSAPrivateKeySpec.class);
				modulus = dsaKeySpec.getP();
		}
		
		return modulus.bitLength();
	}
	
	/**
	 * Returns value of "CN" attribute from subject name of this certificate.
	 * @param x509Certificate
	 * @return String
	 */
	public static String getCommonSubjectName(X509Certificate x509Certificate){
		String subjectCN = null;
		try{
			String subjectNameDN = x509Certificate.getSubjectX500Principal().getName();//javax.security.auth.x500.X500Principal.RFC2253
			if(subjectNameDN != null){
				LdapName ldapName = new LdapName(subjectNameDN);
				for(Rdn rdn:ldapName.getRdns()){
					if(rdn.getType().equals("CN")){
						subjectCN = (String) rdn.getValue();
						break;
					}
				}
			}
		}catch(InvalidNameException e){
			throw new IllegalArgumentException("Certificate DN is incorrectly formatted",e);
		}
		return subjectCN;
	}
}

