/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.font;


import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import com.adobe.fontengine.fontmanagement.FontLoader;
import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute;
import com.adobe.fontengine.inlineformatting.css20.CSS20FontDescription;

import com.adobe.xfa.ut.UnitSpan;


/**
 * A service class to handle all font related actions.
 * <p>
 * It provides a number of convenience methods for accessing fonts.
 * It provides operations to get a comprehensive list of fonts, metrics,
 * and other font related information.
 *
 * @exclude from published api.
 */
public class FontService {
	/*
	 * Encodings
	 */ 

	/**
	 * @exclude from published api.
	 */
	public static final int ISO_8859_1 = 1;
	/**
	 * @exclude from published api.
	 */
	public static final int ISO_8859_2 = 2;
	/**
	 * @exclude from published api.
	 */
	public static final int ISO_8859_5 = 3;
	/**
	 * @exclude from published api.
	 */
	public static final int ISO_8859_6 = 4;
	/**
	 * @exclude from published api.
	 */
	public static final int ISO_8859_7 = 5;
	/**
	 * @exclude from published api.
	 */
	public static final int ISO_8859_8 = 6;
	/**
	 * @exclude from published api.
	 */
	public static final int ISO_8859_9 = 7;
	/**
	 * @exclude from published api.
	 */
	public static final int IBM_874 = 8;
	/**
	 * @exclude from published api.
	 */
	public static final int IBM_1258 = 9;
	/**
	 * @exclude from published api.
	 */
	public static final int SHIFT_JIS = 10;
	/**
	 * @exclude from published api.
	 */
	public static final int SHIFT_JIS_83 = 11;
	/**
	 * @exclude from published api.
	 */
	public static final int KSC_5601 = 12;
	/**
	 * @exclude from published api.
	 */
	public static final int BIG_5 = 13;
	/**
	 * @exclude from published api.
	 */
	public static final int HKSCS_BIG5 = 14;
	/**
	 * @exclude from published api.
	 */
	public static final int GBK = 15;
	/**
	 * @exclude from published api.
	 */
	public static final int FONT_SPECIFIC = 16;
	/**
	 * @exclude from published api.
	 */
	public static final int UTF_8 = 17;
	/**
	 * @exclude from published api.
	 */
	public static final int UTF_16 = 18;
	/**
	 * @exclude from published api.
	 */
	public static final int UCS_2 = 19;
	/**
	 * @exclude from published api.
	 */
	public static final int IDENTITY = 20;
	/**
	 * @exclude from published api.
	 */
	public static final int MAC_ROMAN = 21;
	/**
	 * @exclude from published api.
	 */
	public static final int ENCODING_NONE = 22;

	private final SortedMap<FontInfo, FontItem> mFontList;

	/**
	 * @exclude from published api.
	 */
	public FontService (String fontPath) {		// TODO: how to construct a FontService?
		File fontDir = new File (fontPath);
		FontLoader fontLoader = new FontLoader();
		Font[] fonts = fontLoader.load (fontDir, true, null);
		mFontList = new TreeMap<FontInfo, FontItem> ();
		for (int fontIndex = 0; fontIndex < fonts.length; fontIndex++) {
			CSS20FontDescription[] descs = null;
			Font font = fonts[fontIndex];
			try {
				descs = font.getCSS20FontDescription();
			} catch (Exception e) {
				assert (false);
			}
			if (descs != null) {
//				System.out.println ("Font:");
				for (int descIndex = 0; descIndex < descs.length; descIndex++) {
					CSS20FontDescription desc = descs[descIndex];
					if (desc.getVariant() == CSS20Attribute.CSSVariantValue.NORMAL) {
//						System.out.print ("  \"" + desc.getFamilyName() + "\"");
//						System.out.print (" weight: " + desc.getWeight());
//						System.out.print (" style: " + ((desc.getStyle() == CSS20Attribute.CSSStyleValue.NORMAL) ? "normal" : "italic"));
//						System.out.print (" sizes: " + Double.toString (desc.getLowPointSize()) + " " + Double.toString (desc.getHighPointSize()));
//						System.out.println (" stretch: " + desc.getStretch().toString());
						FontInfo key = FontInfo.createFromAFEDescription (desc);
						boolean put = false;
						FontItem existingFont = mFontList.get (key);
						if (existingFont == null) {
							put = true;
						} else if (key.getStretchDiff() < existingFont.getStretchDiff()) {
							put = true;
						} else if ((key.getStretchDiff() == existingFont.getStretchDiff()) && (key.getLegacySizeDiff() < existingFont.getLegacySizeDiff())) {
							put = true;
						}
						if (put) {
							if (existingFont != null) {
//								System.out.println ("  Removing old definition...");
								mFontList.remove (key);
							}
							FontItem fontItem = new FontItem (key, font);
							if (fontItem.loadMetrics()) {
								mFontList.put (fontItem, fontItem);
							}
						}
					}
				}
			}
		}
	}

	/**
	 * @exclude from published api.
	 */
	public synchronized FontInstance reconcile (FontInfo info, UnitSpan size, double horizontalScale, double verticalScale) {
		FontItem fontItem = reconcile (info);
		if (fontItem == null) {
			return null;
		}
		return fontItem.reconcile (size, horizontalScale, verticalScale);
	}

	/**
	 * @exclude from published api.
	 */
	public synchronized FontItem reconcile (FontInfo key) {
		
		// JavaPort: Implementation modified to avoid using SortedArray
		
		FontItem fontItem = mFontList.get(key);
		if (fontItem != null)
			return fontItem;
		
		List<FontItem> fontItems = new ArrayList<FontItem>();
		
		for (FontItem item : mFontList.values()) {
			if (item.getTypeface().equals(key.getTypeface())) {
				fontItems.add(item);
			}
		}
		
		if (fontItems.size() == 0)
			return null;
		
		int i;
		int foundIndex = -1;
		int bestWeightDelta = Integer.MAX_VALUE;
		for (i = 0; i < fontItems.size(); i++) {
			FontItem testFont = fontItems.get(i);
			if (testFont.getItalic() == key.getItalic()) {
				int weightDelta = Math.abs(testFont.getWeight() - key.getWeight());

				if ((foundIndex < 0) || (weightDelta < bestWeightDelta)) {
					foundIndex = i;
					bestWeightDelta = weightDelta;
				}
			}
		}

		return fontItems.get(foundIndex);
	}

	/**
	 * @exclude from published api.
	 */
	public static int getEncoding(String sEncoding) {
	// Javaport: TODO
    	return FontService.ENCODING_NONE;
	}

	
	/**
	 * @exclude from published api.
	 */
	public FontInstance getDefaultFontInstance () {
		assert (false);								// TODO:
		return null;
	}

//	private FontItem lookup (Object key) {
//		return (FontItem) mFontList.get (key);
//	}

//	private FontItem getFont (int index) {
//		return mFontList.value (index);
//	}

	/**
	 * @exclude from published api.
	 */
	public synchronized String toString () {
		StringBuilder result = new StringBuilder ("Font Service:\n");
		for (FontItem fontItem : mFontList.values()) {
			result.append ("  (" +  fontItem.toString() + ")\n");
		}
		return result.toString();
	}

//	private static int scoreStretchValue (CSS20Attribute.CSSStretchValue stretch) {
//		if ((stretch == CSS20Attribute.CSSStretchValue.SEMICONDENSED) || (stretch == CSS20Attribute.CSSStretchValue.SEMIEXPANDED)) {
//			return 1;
//		} else if ((stretch == CSS20Attribute.CSSStretchValue.CONDENSED) || (stretch == CSS20Attribute.CSSStretchValue.EXPANDED)) {
//			return 2;
//		} else if ((stretch == CSS20Attribute.CSSStretchValue.EXTRACONDENSED) || (stretch == CSS20Attribute.CSSStretchValue.EXTRAEXPANDED)) {
//			return 3;
//		} else if ((stretch == CSS20Attribute.CSSStretchValue.ULTRACONDENSED) || (stretch == CSS20Attribute.CSSStretchValue.ULTRAEXPANDED)) {
//			return 4;
//		}
//		return 0;
//	}

}
