/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.template;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.xml.sax.Attributes;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Attribute;
import com.adobe.xfa.Chars;
import com.adobe.xfa.ChildReln;
import com.adobe.xfa.Element;
import com.adobe.xfa.EnumAttr;
import com.adobe.xfa.EnumValue;
import com.adobe.xfa.Generator;
import com.adobe.xfa.LogMessage;
import com.adobe.xfa.Measurement;
import com.adobe.xfa.Model;
import com.adobe.xfa.Node;
import com.adobe.xfa.NodeList;
import com.adobe.xfa.ProcessingInstruction;
import com.adobe.xfa.ProtoableNode;
import com.adobe.xfa.RichTextNode;
import com.adobe.xfa.STRS;
import com.adobe.xfa.Schema;
import com.adobe.xfa.ScriptHandler;
import com.adobe.xfa.ScriptInfo;
import com.adobe.xfa.ScriptTable;
import com.adobe.xfa.StringAttr;
import com.adobe.xfa.TextNode;
import com.adobe.xfa.XFA;
import com.adobe.xfa.XMLMultiSelectNode;
import com.adobe.xfa.AppModel.LegacyMask;
import com.adobe.xfa.content.ExDataValue;
import com.adobe.xfa.content.ImageValue;
import com.adobe.xfa.content.TextValue;
import com.adobe.xfa.template.automation.EventTag;
import com.adobe.xfa.template.automation.Script;
import com.adobe.xfa.template.containers.Container;
import com.adobe.xfa.template.containers.ExclGroup;
import com.adobe.xfa.template.containers.Field;
import com.adobe.xfa.template.containers.PageSet;
import com.adobe.xfa.template.containers.Subform;
import com.adobe.xfa.template.containers.Variables;
import com.adobe.xfa.template.formatting.Color;
import com.adobe.xfa.ut.Assertions;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.Numeric;
import com.adobe.xfa.ut.Peer;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.ResourceLoader;
import com.adobe.xfa.ut.Storage;
import com.adobe.xfa.ut.IntegerHolder;
import com.adobe.xfa.ut.StringUtils;
import com.adobe.xfa.xmp.XMPHelper;

/**
 * A class to model the collection of all the XFA nodes that make up form
 * template.
 */
public final class TemplateModel extends Model {

	private static final boolean ACROBAT_PLUGIN = ResourceLoader.loadProperty("ACROBAT_PLUGIN").equalsIgnoreCase("true");
	
	final static class ContainerIndex {
		public ContainerIndex(int nContainerIndex, Object container) {
			this.nContainerIndex = nContainerIndex;
			this.container = container;
		}
		
		public final int nContainerIndex;
		public final Object container;
	}

	final static class IndexTableElement {
		
		public IndexTableElement(Object parent) {
			this.parent = parent;
		}
		
		public final List<LikeNamedContainers> likeChildren = new ArrayList<LikeNamedContainers>();
		public final Object parent;
	}

	final static class LikeNamedContainers {
		
		public LikeNamedContainers(String likeName) {
			this.likeName = likeName;
		}
		
		public final List<ContainerIndex> containers = new ArrayList<ContainerIndex>();
		public final String likeName;
	}

	private static final TemplateSchema gTemplateSchema = new TemplateSchema();
	
	
	/**
	 * Static method to be shared by Form Model's version of enumerateScripts.  Recursive.
	 * @exclude from public api.
	 */
	public static void enumerateScripts(Model model,
										Node node,
										List<ScriptInfo> scripts,
										String sSingleLanguage) {
			
		if (node instanceof Field || 
			node instanceof Subform ||
			node instanceof ExclGroup) {
			
			Container element = (Container)node;

			// get calculations 
			// watson bug 1497977  calculates are valid on fields, subforms and exclgroups
			Element calcProp = element.getElement(XFA.CALCULATETAG, true, 0, false, false);
			if (calcProp != null) {
				ScriptInfo si = getScriptInfo(element, calcProp, sSingleLanguage, ScriptHandler.CALCULATE);
				if (si != null) {
					scripts.add(si);
				}
			}
			// get validations
			// watson bug 1497977  validates are valid on fields, subforms and exclgroups
			Element validateProp = element.getElement(XFA.VALIDATETAG, true, 0, false, false);
			if (validateProp != null) {
				ScriptInfo si = getScriptInfo(element, validateProp, sSingleLanguage, ScriptHandler.VALIDATE); 
				if (si != null) {
					scripts.add(si);
				}
			}
			
			// On fields or subforms, get other actions such as "click" or "initialize" event scripts,
			// as well as scripts in the <variables> tag.

			int nIndex = 0;
			NodeList children = element.getNodes();
			while (nIndex < children.length()){
				Node child = (Node)children.item(nIndex);
				if (child instanceof Variables) {
					int nVariablesIndex = 0;
					NodeList variablesChildren = child.getNodes();
					while (nVariablesIndex < variablesChildren.length()) {
						Node script = (Node)variablesChildren.item(nVariablesIndex);
						if (script instanceof Script) {
							Script scriptElement = (Script)script;
							String sLanguageName = scriptElement.getAttribute(XFA.CONTENTTYPETAG).toString();
							if (StringUtils.isEmpty(sSingleLanguage) || sSingleLanguage.equals(sLanguageName)) {
								TextNode text = scriptElement.getText(true, false, false);
								if (text != null) {
									String sScript = text.getValue();

									ScriptInfo si = new ScriptInfo(
											sScript,
											sLanguageName,
											script,
											"",	// <variables> scripts have no ref tags
											ScriptHandler.UNSPECIFIED);
									scripts.add(si);
								}
							}
						}

						nVariablesIndex++;
					}
				}
				else if (child instanceof EventTag) {
					EventTag eventElement = (EventTag) child;
					Node script = child.peekOneOfChild(false);
					if (script instanceof Script) {
						Script scriptElement = (Script)script;
						String sLanguageName = scriptElement.getAttribute(XFA.CONTENTTYPETAG).toString();
						if (StringUtils.isEmpty(sSingleLanguage) || sSingleLanguage.equals(sLanguageName)) {
							TextNode text = scriptElement.getText(true, false, false);
							if (text != null) {
								String sScript = text.getValue();

								String sReason = eventElement.getAttribute(XFA.ACTIVITYTAG).toString();
								String sRef = eventElement.getAttribute(XFA.REFTAG).toString();
								int eReason = ScriptHandler.stringToExecuteReason(sReason);
								ScriptInfo si = new ScriptInfo(
										sScript,
										sLanguageName,
										element,
										sRef,
										eReason);
								scripts.add(si);
							}
						}
					}
				}

				nIndex++;	
			}
		}

		if (node.isContainer()) {
			for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
				enumerateScripts(model, child, scripts, sSingleLanguage);
			}
		}
	}
	
	/**
	 * getScriptInfo is a static helper function to retrieve either calculate or validate info.
	 */
	private static ScriptInfo getScriptInfo(Node node, Element eventNode, String sSingleLanguage, int eReason) {
		Node scriptNode = eventNode.getElement(XFA.SCRIPTTAG, true, 0, false, false);

		if (! (scriptNode instanceof Script))
			return null;
		
		Script scriptElement = (Script)scriptNode;

		// See if the script is for us to execute -- accept either an
		// empty string or "XFA" in the "binding" tag.
		String sBinding = scriptElement.getAttribute(XFA.BINDINGTAG).toString();
		if (!StringUtils.isEmpty(sBinding) && !sBinding.equals("XFA"))
			return null;

		TextNode scriptText = scriptElement.getText(true, false, false);
		if (scriptText == null)
			return null;
		
		String sLanguageName = scriptElement.getAttribute(XFA.CONTENTTYPETAG).toString();
		if (!StringUtils.isEmpty(sSingleLanguage) && !sSingleLanguage.equals(sLanguageName))
			return null;
		
		return new ScriptInfo(
			scriptElement.getValue(),
			sLanguageName,
			node, 
			"",	// <calculate> and <validate> scripts have no ref tags
			eReason);
	}
	
	/**
	 * @exclude from public api.
	 */
	public static Schema getModelSchema() {
		return gTemplateSchema;
	}

	/**
	 * Gets the template model held within an XFA DOM hierarchy.
	 *
	 * @param app the application model.
	 *
	 * @param bCreateIfNotFound when true, create a template model if needed.
	 *
	 * @return the template model or null if none found.
	 */
	public static TemplateModel getTemplateModel(AppModel app,
								boolean bCreateIfNotFound /* = false */) {
		
		TemplateModel tm = (TemplateModel)Model.getNamedModel(app, XFA.TEMPLATE);		

		if (bCreateIfNotFound && tm == null) {
			TemplateModelFactory factory = new TemplateModelFactory();
			tm = (TemplateModel) factory.createDOM((Element)app.getXmlPeer());
			tm.setDocument(app.getDocument());
			
			app.notifyPeers(Peer.CHILD_ADDED, XFA.TEMPLATE, tm);
		}
		
		return tm;
	}

	/**
	 *
	 * @exclude from published api.
	 */
	@FindBugsSuppress(code="ES")
	static public String getValidationMessage(Element poValidateNode,
			String aType) {
		String sMessage = null;

		if (poValidateNode != null) {
			Element poMessageNode = poValidateNode.getElement(XFA.MESSAGETAG, true, 0, false, false);
			if (poMessageNode != null) {
				boolean bFirst = true;

				for (Node poChild = poMessageNode.getFirstXFAChild(); poChild != null; poChild = poChild.getNextXFASibling()) {
					if (poChild instanceof TextValue) {
						TextValue oText = (TextValue) poChild;
						String aName = poChild.getName();
						// if the name is equal to the type then stop looking
						if (aName == aType) {
							sMessage = oText.getValue();
							break;
						}

						// default to the first nameless text node;
						if (bFirst && aName == "") {
							sMessage = oText.getValue();
							bFirst = false;
						}
					}
				}
			}
		}

		return sMessage;
	}

	/**
	 * Override of Model.getOriginalVersion().
	 * This override is removed in the C++ version, but is required here because
	 * of the special handling for the model's namespace.
	 * 
	 * @exclude
	 */
	public String getOriginalVersion(boolean bDefault) {

		if (bDefault && (mnOriginalTemplateVersion != 0)) {
			String sVer = super.getOriginalVersion(false);
			if ((sVer.length() == 0) && (mnOriginalTemplateVersion <= getCurrentVersion())) 
				return getNS(mnOriginalTemplateVersion);
			else
				return super.getOriginalVersion(bDefault);
		}
		else
			return super.getOriginalVersion(bDefault);
	}

	/**
	 * Override of Model.getVersion().
	 * 
	 * @exclude
	 */
	public int getVersion(String sNS)
	{
		int nVersion = super.getVersion(sNS);
		if (nVersion == 0)
		{
			//this step is automatically done on C++ side while loading the dom 
			if(meInputGenerator == 0)
				initGenerator();
			// If we're here, it's because we couldn't find a namespace.
	
			// FF99 and output Designer Forms are not supported on the server
			// unless they've gone through the Adobe Designer first, however,
			// if we encounter them on the server (i.e. via our test suite),
			// we will bump them to XFA 2.1.
			// Similarly, unrecognized Generators will default to the lowest
			// common denominator.
			if (meInputGenerator == Generator.XFAGen_FF99V250_01 ||
				meInputGenerator == Generator.XFAGen_FF99V310 ||
				meInputGenerator == Generator.XFAGen_JDV530_01 ||
				meInputGenerator == Generator.XFAGen_FF99V40 ||
				meInputGenerator == Generator.XFAGen_FF99V50 ||
				meInputGenerator == Generator.XFAGen_AdobeDesignerV60 ||
				meInputGenerator == Generator.XFAGen_TemplateDesignerV60d ||
				meInputGenerator == Generator.XFAGen_AdobeDesignerV60_SAP ||
				meInputGenerator == Generator.XFAGen_2_0)
				return Schema.XFAVERSION_21;
			else if (meInputGenerator == Generator.XFAGen_AdobeDesignerV70 ||
					 meInputGenerator == Generator.XFAGen_AdobeDesignerV70_SAP)
					 return Schema.XFAVERSION_22;
			else if (meInputGenerator == Generator.XFAGen_AdobeDesignerV71 ||
					 meInputGenerator == Generator.XFAGen_AdobeDesignerV71_SAP ||
					 meInputGenerator == Generator.XFAGen_2_4)
					 return Schema.XFAVERSION_24;
			else if (meInputGenerator == Generator.XFAGen_AdobeDesignerV80 ||
					 meInputGenerator == Generator.XFAGen_AdobeDesignerV80_SAP)
					 return Schema.XFAVERSION_25;
			else
				return Schema.XFAVERSION_21;	// Default to lowest common denominator if no version is found
		}
		else
			return nVersion;
	}
	
	/**
	 * Common code, so that the implementation is in one spot.
	 * 
	 * @exclude
	 */
	protected void updateLegacyFlags(boolean bValue, LegacyMask nLegacyFlags, LegacyMask oLegacyInfo) {
		//Given a set of legacy flags that are to be set to a given value, determine which, if any, other legacy flags need to be set/cleared because they are dependent.
		LegacyMask oDependentFlagsToClear = new LegacyMask(0);
		LegacyMask oDependentFlagsToSet = new LegacyMask(0);

		// Layout legacy flags have dependencies. Find out what they are.
		LegacyMask DEPENDENT_LAYOUTLEGACY_FLAGS[] = new LegacyMask[] {AppModel.XFA_LEGACY_V27_LAYOUT, AppModel.XFA_LEGACY_V28_LAYOUT, AppModel.XFA_LEGACY_V29_LAYOUT, AppModel.XFA_LEGACY_V30_LAYOUT };		//must be in ascending order
		int DEPENDENT_LAYOUT_VERSIONS[]	   = new int[] {Schema.XFAVERSION_28,           Schema.XFAVERSION_29,           Schema.XFAVERSION_30,           Schema.XFAVERSION_31 };				//corresponding xfa rev each list entry above maps to
		int nSize = DEPENDENT_LAYOUTLEGACY_FLAGS.length;
		assert(DEPENDENT_LAYOUTLEGACY_FLAGS.length == DEPENDENT_LAYOUT_VERSIONS.length);
		getDependentFlags(bValue, nLegacyFlags, oDependentFlagsToSet, oDependentFlagsToClear, DEPENDENT_LAYOUTLEGACY_FLAGS, DEPENDENT_LAYOUT_VERSIONS, nSize);

		// Scripting legacy flags have dependencies. Find out what they are.
		LegacyMask DEPENDENT_SCRIPTINGLEGACY_FLAGS[] = new LegacyMask[] {AppModel.XFA_LEGACY_V27_SCRIPTING, AppModel.XFA_LEGACY_V28_SCRIPTING, AppModel.XFA_LEGACY_V29_SCRIPTING, AppModel.XFA_LEGACY_V30_SCRIPTING, AppModel.XFA_LEGACY_V32_SCRIPTING };	//must be in ascending order
		int DEPENDENT_SCRIPTING_VERSIONS[]	  = new int[] {Schema.XFAVERSION_28,              Schema.XFAVERSION_29,              Schema.XFAVERSION_30,				Schema.XFAVERSION_31, Schema.XFAVERSION_32 };				//corresponding xfa rev each list entry above maps to
		nSize = DEPENDENT_SCRIPTINGLEGACY_FLAGS.length;
		assert(DEPENDENT_SCRIPTINGLEGACY_FLAGS.length == DEPENDENT_SCRIPTING_VERSIONS.length);
		getDependentFlags(bValue, nLegacyFlags, oDependentFlagsToSet, oDependentFlagsToClear, DEPENDENT_SCRIPTINGLEGACY_FLAGS, DEPENDENT_SCRIPTING_VERSIONS, nSize);

		// Traversal legacy flags have dependencies. Find out what they are.
		LegacyMask DEPENDENT_TRAVERSALLEGACY_FLAGS[] = new LegacyMask[] {AppModel.XFA_LEGACY_V27_TRAVERSALORDER, AppModel.XFA_LEGACY_V29_TRAVERSALORDER };	//must be in ascending order
		int DEPENDENT_TRAVERSAL_VERSIONS[]	  = new int[] {Schema.XFAVERSION_28,                   Schema.XFAVERSION_30 };				//corresponding xfa rev each list entry above maps to
		nSize = DEPENDENT_TRAVERSALLEGACY_FLAGS.length;
		assert(DEPENDENT_TRAVERSALLEGACY_FLAGS.length == DEPENDENT_TRAVERSAL_VERSIONS.length);
		getDependentFlags(bValue, nLegacyFlags, oDependentFlagsToSet, oDependentFlagsToClear, DEPENDENT_TRAVERSALLEGACY_FLAGS, DEPENDENT_TRAVERSAL_VERSIONS, nSize);

		//Set/clear the original flags as requested 
		if (bValue)
			oLegacyInfo.set(oLegacyInfo.or(nLegacyFlags)); // turn on the necessary bits
		else
			oLegacyInfo.set(oLegacyInfo.and(nLegacyFlags.not())); // turn off the necessary bits

		//Now go and clear those that are required due to dependencies
		oLegacyInfo.set(oLegacyInfo.and(oDependentFlagsToClear.not()));

		//Finally, go and set those that are required due to dependencies
		oLegacyInfo.set(oLegacyInfo.or(oDependentFlagsToSet));
	}

	//Helper function that returns additional layout legacy flags to set/clear based on other flag settings
	private void getDependentFlags(boolean bValue, LegacyMask nLegacyFlags, LegacyMask oDependentFlagsToSet, 
								   LegacyMask oDependentFlagsToClear, LegacyMask DEPENDENT_LEGACY_FLAGS[],
								   int DEPENDENT_VERSIONS[], int nSize) {
		//LAYOUT FLAG DEPENDENCIES	
		//The dependency rules are as follows:
		//1) Setting v2.X-layout:0 --> use max(v2.(X+1) , originalVersion) layout. This is equivalent to setting the v2.?-layout and lower to 0 and all other flags as 1.
		//2) Setting v2.X-layout:1 --> harden as v2.X layout. This is equivalent to setting v2.X-layout flags and higher to 1 and all other layout legacy flags as 0.
		//3) Explicit layout flags trump any earlier layout flags values.
		//For instance, setting v2.8-layout:0 means v2.7-layout must also be set to zero.	
		//See https://zerowing.corp.adobe.com/display/xtg/OriginalXFAVersionClarificationProposal#OriginalXFAVersionClarificationProposal-(Amendment07August2008)
		

		for (int i = (nSize - 1), nCount = 0; nCount < nSize; i--, nCount++) {
			//Work backwards, looking for most recent v2.X-layout flag.
			if (DEPENDENT_LEGACY_FLAGS[i].and(nLegacyFlags).isNonZero()) {
				int j = 0;
				if (bValue == true) {
					//Setting v2.X-layout to 1. Clear all lower layout flags. Set v2.X-layout and higher flags to 1. 
					for (j = 0; j < i; j++)
						oDependentFlagsToClear.or_n_set(DEPENDENT_LEGACY_FLAGS[j]);
					for (j = (i + 1); j < nSize; j++)
						oDependentFlagsToSet.or_n_set(DEPENDENT_LEGACY_FLAGS[j]);
				}
				else {
					//Take the max of the originalversion and the corresponding XFA rev of the legacy flag
					while (mnOriginalVersion > DEPENDENT_VERSIONS[i] && ((i + 1) < nSize))
						i++;
						
					//Clear all max(v2.X, originalVersion)-layout to 0. Set all higher flags to 1. 
					for (j = 0; j < i; j++)
						oDependentFlagsToClear.or_n_set(DEPENDENT_LEGACY_FLAGS[j]);
					for (j = (i + 1); j < nSize; j++)
						oDependentFlagsToSet.or_n_set(DEPENDENT_LEGACY_FLAGS[j]);
				}

				break; //Done. Any other v2.X-layout bits set in 'nLegacyFlags' are trumped by the depencies of the one we just found.
			}
		}
	}

	/**
	 * @exclude
	 */
	void updateLegacyInfo(String sPIValue, String sOption, LegacyMask nLegacyFlags, LegacyMask oLegacyInfo, String sBehaviorOverride) {

		String sNum = null;
		int nStart = -1;
		
		// search for the flag option in the behaviorOverride value
		if (sBehaviorOverride != null) {
			nStart = sBehaviorOverride.indexOf(sOption);
			if (nStart != -1) {
				int nOffset = nStart + sOption.length() + 1; // need to trim of ":"
				sNum = sBehaviorOverride.substring(nOffset, nOffset+1);

				if (sNum.equals("0"))
					updateLegacyFlags(false, nLegacyFlags, oLegacyInfo);
				else
					updateLegacyFlags(true, nLegacyFlags, oLegacyInfo);
				
				return;
			}
		}
		
		// search for the flag option in the PI
		nStart = sPIValue.indexOf(sOption);
		if (nStart != -1) {
			int nOffset = nStart + sOption.length() + 1; // need to trim of ":"
			sNum = sPIValue.substring(nOffset,nOffset+1);
		}
		
		if (sNum != null) {
			if (sNum.equals("0"))
				updateLegacyFlags(false, nLegacyFlags, oLegacyInfo);
			else
				updateLegacyFlags(true, nLegacyFlags, oLegacyInfo);
		}
	}

	/**
	 *
	 * @exclude from published api.
	 */
	@FindBugsSuppress(code="ES")
	public static void setValidationMessage(Element validateNode,
			String sMessage, String aType) {

		if (validateNode != null) {
			Element messageNode = validateNode.getElement(XFA.MESSAGETAG,
					true, 0, true, false);
			if (messageNode != null) {
				boolean bFirst = true;

				if (messageNode.getFirstXFAChild() == null) {
					// need to append the new text node
					messageNode.getModel().createElement(messageNode, null,
							messageNode.getModel().getNS(), XFA.TEXT,
							XFA.TEXT, null, 0, null);
					messageNode.makeNonDefault(false);
				}

				boolean bFound = false;
				Node child;
				for (child = messageNode.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
					if (child instanceof TextValue) {
						TextValue text = (TextValue) child;
						String aName = child.getName();
						// if the name is equal to the type then stop looking
						if (aName == aType) {
							text.setValue(sMessage);
							bFound = true;
							break;
						}

						// default to the first nameless text node;
						if (bFirst && aName == "") {
							text.setValue(sMessage);
							bFound = true;
							bFirst = false;
						}
					}
				}

				// if not found then append a new text node
				if (!bFound) {
					messageNode.getModel().createElement(messageNode,
							messageNode.getLastXMLChild(),
							messageNode.getModel().getNS(), aType, aType,
							null, 0, null);

					TextValue text = (TextValue) child;
					assert (child.getName() == aType);
					text.setValue(sMessage);
				}
			}
		}
	}

	/**
	 * Namespace
	 * @return The template namespace URI string
	 *
	 * @exclude from published api.
	 */
	public static String templateNS() {
		if (Assertions.isEnabled) {
			// If this asserts fire, the head version has changed.
			// You will need to update this assert as well as make sure
			// STRS.XFATEMPLATENS_CURRENT has been updated to this new
			// head version.
			assert(Schema.XFAVERSION_35 == Schema.XFAVERSION_HEAD);
		}
		return STRS.XFATEMPLATENS_CURRENT;
	}

	/**
	 * @param nLegacy flags the current set of bitflag values.
	 * @param nFlag the specific flag to check.
	 */
	private static String updateLegacyString(String sPIValue , LegacyMask nLegacyFlags, LegacyMask nFlag, LegacyMask nDefaults) {
		// if the value is a default value don't write it out
		if (nLegacyFlags.and(nFlag).equals(nDefaults.and(nFlag)))
			return sPIValue;

		boolean bLegacy = (nLegacyFlags.and(nFlag)).isNonZero();

		String sOption = "";
		if (nFlag.equals(AppModel.XFA_LEGACY_POSITIONING)) {
			sOption = "LegacyPositioning";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_EVENTMODEL)) {
			sOption = "LegacyEventModel";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_PLUSPRINT)) {
			sOption = "LegacyPlusPrint";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_PERMISSIONS)) {
			sOption = "LegacyXFAPermissions";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_CALCOVERRIDE)) {
			sOption = "LegacyCalcOverride";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_RENDERING)) {
			sOption = "LegacyRendering";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V26_HIDDENPOSITIONED)) {
			sOption = "v2.6-hiddenPositioned";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V27_MULTIRECORDCONTEXTCACHE)) {
			sOption = "v2.7-multiRecordContextCache";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V27_LAYOUT)) {
			sOption = "v2.7-layout";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V27_SCRIPTING)) {
			sOption = "v2.7-scripting";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V27_EVENTMODEL)) {
			sOption = "v2.7-eventModel";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V27_TRAVERSALORDER)) {
			sOption = "v2.7-traversalOrder";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V27_XHTMLVERSIONPROCESSING)) {
			sOption = "v2.7-XHTMLVersionProcessing";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V27_ACCESSIBILITY)) {
			sOption = "v2.7-accessibility";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V28_LAYOUT)) {
			sOption = "v2.8-layout";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V28_SCRIPTING)) {
			sOption = "v2.8-scripting";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V29_LAYOUT)) {
			sOption = "v2.9-layout";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V29_SCRIPTING)) {
			sOption = "v2.9-scripting";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V30_LAYOUT)) {
			sOption = "v3.0-layout";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V30_SCRIPTING)) {
			sOption = "v3.0-scripting";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V29_TRAVERSALORDER)) {
			sOption = "v2.9-traversalOrder";
		} else if (nFlag.equals(AppModel.XFA_PATCH_B_02518)) {
			sOption = "patch-B-02518";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V29_FIELDPRESENCE)) {
			sOption = "v2.9-fieldPresence";
		} else if (nFlag.equals(AppModel.XFA_PATCH_W_2393121)) {
			sOption = "patch-W-2393121";
		} else if (nFlag.equals(AppModel.XFA_PATCH_W_2447677)) {
			sOption = "patch-W-2447677";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V32_SCRIPTING)) {
			sOption = "v3.2-scripting";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V31_LAYOUT)) {
			sOption = "v3.1-layout";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V32_LAYOUT)) {
			sOption = "v3.2-layout";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V32_RENDERING)) {
			sOption = "v3.2-rendering";
		} else if (nFlag.equals(AppModel.XFA_LEGACY_V32_CANONICALIZATION)) {
			sOption = "v3.2-canonicalization";
		} else if (nFlag.equals(AppModel.XFA_PATCH_W_2757988)) {
			sOption = "patch-W-2757988";
		} else {
			assert false; // should not get here
		}
		
		 // now update the PI
		sPIValue += ' ';
		sPIValue += sOption;
		if (bLegacy)
			sPIValue += ":1";
		else
			sPIValue += ":0";
		
		return sPIValue;
	}

	
	/**
	 * Protoable nodes containing extras/text/@name with this name represent an insertion point.
	 * @see #splice(List, boolean)
	 */
	private final static String INSERTION_POINT_NAME = "insertionPoint";
	
	/**
	 * Protoable nodes containing extras/text/@name with this name represent temporary placeholder 
	 * content for an insertion point.
	 * @see #splice(List, boolean)
	 */
	private final static String INSERTION_POINT_PLACEHOLDER_NAME = "insertionPointPlaceholder";

	

	/**
	 * This member should be set by the template model factory
	 * after the model is created.  If set, we'll apply fixups post load.
	 */
	private boolean mbApplyFixups;
	private boolean mbFixupRenderCache;

	private boolean mbCheckRelevant;

	private boolean mbHasServerSideScript;

	/**
	 *
	 * @exclude from published api.
	 */
	protected boolean mbIsSourceSetModelInitialized;
	
	private int meInputGenerator = Generator.XFAGen_Unknown;;

	/**
	 * For each bind element, we need to create a unique XML ID. We will
	 * use the word bind + the idCount and increment it each time, so that
	 * we will have a unique xmlId within the document for each find element.
	 *
	 * @exclude from published api.
	 */
	protected int mnIDCount;


	private final Storage<String> moGlobalFields = new Storage<String>();	// values are interned

	private final Storage<IndexTableElement> moIndexTable = new Storage<IndexTableElement>();
	
	private final Storage<String> moRelevantList = new Storage<String>();
	
	private LegacyMask mnLegacyInfo;
	private int mnOriginalTemplateVersion;
	
	private TemplateResolver moTemplateResolver = new TemplateResolver();
	
	/**
	 *
	 * @exclude from published api.
	 */
	public TemplateModel(Element parent, Node prevSibling) {
		super(parent, prevSibling, templateNS(), XFA.TEMPLATE, XFA.TEMPLATE,
				STRS.DOLLARTEMPLATE, XFA.TEMPLATETAG, XFA.TEMPLATE,
				getModelSchema());
		
		//mnLegacyInfo = 0;
		//mnOriginalVersion = 0;
		//mnOriginalTemplateVersion = 0;
		
		//mlIDCount = 0;
		//mbIsSourceSetModelInitialized = false;
		//mbCheckRelevant = false;

		//
		// The TemplateModel architecture expects us to have an
		// implementation for our childNodes jfArraylist.
		// If it doesn't then the TemplateModel code will not work
		//

	}

	/**
	 *
	 * @exclude from published api.
	 */
	public TemplateModel(Element parent, Node prevSibling, String uri, String QName, String name) {
		super(parent, prevSibling, uri, QName, name,
				STRS.DOLLARTEMPLATE, XFA.TEMPLATETAG, XFA.TEMPLATE,
				getModelSchema());
		
		//mnLegacyInfo = 0;
		//mnOriginalVersion = 0;
		
		// On the C++ side the initial template version is kept intact in the XML Dom, but 
		// since we don't have two DOM's on the Java side, the version might get updated
		// while loading the Template DOM. Instead we need to keep track of the original
		// version in case the Template version gets updated and an "OriginalXFAVersion"
		// processing instruction needs to be added.
		mnOriginalTemplateVersion = getCurrentVersion();
		
		//mlIDCount = 0;
		//mbIsSourceSetModelInitialized = false;
		//mbCheckRelevant = false;

		//
		// The TemplateModel architecture expects us to have an
		// implementation for our childNodes jfArraylist.
		// If it doesn't then the TemplateModel code will not work
		//
	}

	//this step is done while loading the dom peer on C++ side.
	private void initGenerator() {
		//set generator as the loading step might need this.
		Generator g = getGenerator();
		if (g != null)
			meInputGenerator = getGenerator().generator();
	}

	void addIndexEntry(ContainerIndex indexEntry, String nodeName,
			Object parent) {
		boolean parentFound = false;
		int i = 0;
		for (i = 0; i < moIndexTable.size(); i++) {
			if (moIndexTable.get(i).parent == parent) {
				parentFound = true;
				break;
			}

		}
		if (parentFound) {
			IndexTableElement ite = moIndexTable.get(i);
			// insert in sorted order
			int j = 0;
			for (j = 0; j < ite.likeChildren.size(); j++) {
				LikeNamedContainers likeChild = ite.likeChildren.get(j);
				if (likeChild.likeName.equals(nodeName)) {
					int listSize = likeChild.containers.size();
					int k = 0;
					while (k < listSize
							&& indexEntry.nContainerIndex >= likeChild.containers.get(k).nContainerIndex) {
						k++;
					}
					likeChild.containers.add(k, indexEntry);
					break;
				}
			}

		} 
		else {
			LikeNamedContainers newLikeNamedContainer = new LikeNamedContainers(nodeName);
			newLikeNamedContainer.containers.add(indexEntry);

			IndexTableElement newParentEntry = new IndexTableElement(parent);
			newParentEntry.likeChildren.add(newLikeNamedContainer);
			moIndexTable.add(newParentEntry);
		}
	}

	/**
	 * Applies fixups on the template model just after it has been loaded.
	 * <p>
	 * Note that this Java implementation does not handle the full set of
	 * fixups that are currently handled in the C++ implementation.
	 */
	public void applyFixups() {
		mbApplyFixups = true;
	}
	
	/**
	 * Removes render cache from the template model just after it has been loaded.
	 *
	 * @exclude from published api.
	 */
	public void fixupRenderCache() {
		mbFixupRenderCache = true;
	}
	
	/**
	 * Arrange XFA-Template DOM nodes in geographical order.
	 *
	 * @exclude from published api.
	 */
	public void convertToGeographicalOrder() {
		orderNodes(getModel());
	}

	/**
	 * @see com.adobe.xfa.Model#createElement(com.adobe.xfa.Element, com.adobe.xfa.Node, java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes, int, java.lang.String)
	 *
	 * @exclude from published api.
	 */
	@FindBugsSuppress(code="ES")
	public Element createElement(Element parent, Node prevSibling, String uri, String localName, String qName, Attributes attributes, int lineNumber, String fileName) {
		
		Element node = super.createElement(parent, prevSibling, uri, localName, qName,
				attributes, lineNumber, fileName);

		if (node.getElementClass() == XFA.INVALID_ELEMENT)
			return node;

		String aNodeName = node.getName();
		
		// Load dom node only if allowed by current template view settings.
		if (!isViewAdmissable(node, aNodeName)) {
			node.setClass(node.getClassName(), XFA.INVALID_ELEMENT);
		}
		else if (node instanceof Color) {
			Color oColor = (Color) node;
			getTemplateResolver().addActiveColor(oColor);
		}
		else if (localName == XFA.BIND && parent != null) {
			int index = node.findAttr(null, XFA.MATCH);
			if (index != -1 && node.getAttrVal(index).equals(STRS.GLOBAL)) {
				setAsGlobalField(parent);
			}
		}
		
		return node;
	}

	/**
	 * @see Model#createNode(int, Element, String, String, boolean)
	 *
	 * @exclude from published api.
	 */
	@FindBugsSuppress(code="ES")
	public Node createNode(int eTag, Element parent, String aName, String aNS, boolean bDoVersionCheck) {
		assert (aName != null);
		assert (aNS != null);
		Element retVal = getSchema().getInstance(eTag, this, parent, null, bDoVersionCheck);
		if (aName != null && aName != "") {
			retVal.setName(aName);
		}
		retVal.setNS(aNS);
		return retVal;
	}

	/**
	 * @exclude from public api.
	 */
	public ProtoableNode createLeaderTrailer(String sReference, Node oParentContainer, boolean bPeek) {
		//sRef is either an ID (in which case it it prefixed with '#'), or a SOM expression.
		//Fortunately findInternalProto() takes care of both automatically
		
		return findInternalProto(XFA.SUBFORMTAG, XFA.SUBFORMSETTAG, oParentContainer, sReference, bPeek);	
	}

//	/**
//	 *
//	 * @exclude from published api.
//	 */
//	protected void finalize() {
//		// Remove Image references from our List.
//		getTemplateResolver().cleanupImages();
//	}

	/**
	 * @see Model#getBaseNS()
	 *
	 * @exclude from published api.
	 */
	public String getBaseNS() {
	    return STRS.XFATEMPLATENS;
	}
	
	/**
	 * @see Model#getHeadNS()
	 *
	 * @exclude from published api.
	 */
	public String getHeadNS() {
		return TemplateModel.templateNS();
	}
	
	/**
	 * @see Model#getLegacySetting(AppModel.LegacyMask)
	 * @exclude from published api.
	 */
	public boolean getLegacySetting(LegacyMask nLegacyFlag) {
		// Set the original xfa version which populates mnLegacyInfo
	   	getOriginalXFAVersion();
		// If legacy positioning is turned on, make sure Legacy Render is on as well.
		if ((nLegacyFlag.equals(AppModel.XFA_LEGACY_RENDERING) && mnLegacyInfo.and(AppModel.XFA_LEGACY_POSITIONING).isNonZero()))
			return true;
		
		return mnLegacyInfo.and(nLegacyFlag).isNonZero();
	}
	
	/**
     * Normalize the namespaces of the nodes under this model given a namespace.
   	 * @param aNewNS - the namespace to change to.
   	 *
	 * @exclude from published api.
	 */
	public void normalizeNameSpaces(String aNewNS) {
	 	super.normalizeNameSpaces(aNewNS);
	 
	 	// watson bug 1521181, don't cache the default value because this could change at runtime
	 	// without updating moLegacyInfo.nXFAVersion, this only happens in designer
	 	// so blank out nXFAVersion so we can look it up again later.
	 	mnLegacyInfo = new LegacyMask(0);
	 }

	/**
	 * Gets the original version the template was authored with.
	 * @return the original XFA version
	 * @exclude
	 */
	public int getOriginalXFAVersion() {
	    if (mnOriginalVersion == 0) {
			// watson bug 1521181, don't cache the default value because this could change at runtime
			// without updating moLegacyInfo.nXFAVersion, this only happens in designer
			String sPIValue = getOriginalVersion(false);

			// Watson 2424681: Changed use of resolveNode() to "manual lookup" since when render a print document
			// resolveNode() would issue a version warning because in many cases the template version was 
			// below the config model version, and the output document version is based on the template version
			// in this case.
		    Node oBehaviorOverride = null;
		    
		    Node oConfig = getAppModel().getElement(XFA.CONFIGTAG, true, 0, false, false);
		    if ((oConfig != null) && (oConfig instanceof Element)) {
		    	Node oPresent = ((Element)oConfig).peekElement(XFA.PRESENTTAG, false, 0);
		    	if ((oPresent != null) && (oPresent instanceof Element)) 
		    		oBehaviorOverride = ((Element)oPresent).peekElement(XFA.BEHAVIOROVERRIDETAG, false, 0);
		    }
		    
			String sBehaviorOverride = null;
			if ((oBehaviorOverride != null) && (oBehaviorOverride instanceof Element)) {
				Node child = ((Element)oBehaviorOverride).getFirstXFAChild();
				if ((child != null) && (child instanceof TextNode))
			 		sBehaviorOverride = ((TextNode)child).getValue();
			}

			// Get the current version of the template and Legacy flags deprecated for that
			int nCurrentVersion = getCurrentVersion();
			LegacyMask nDeprecatedFlags = getXFALegacyDeprecatedForXFAVersion(nCurrentVersion);
			
			// If there isn't an original XFA version PI in the template and 
			// behaviorOverride is empty, then get the current version of the template.
			int nVersion;
			if (StringUtils.isEmpty(sPIValue)) {
				sPIValue = getOriginalVersion(true);
				nVersion = getVersion(sPIValue);
				mnLegacyInfo = getXFALegacyDefaultsForXFAVersion(nVersion);
				mnOriginalVersion = nVersion;
				if (StringUtils.isEmpty(sBehaviorOverride)){
					// update the Legacy flags according to current XFA version
					if (nDeprecatedFlags.isNonZero()){
						LegacyMask nDefaultFlags = mnLegacyInfo.and(nDeprecatedFlags.not());
						mnLegacyInfo = nDefaultFlags;
						updateOriginalXFAVersionPI();
					}
					return nVersion;
				}
			}
			else {
			
				// read in the settings
				// IF YOU CHANGE A DEFAULT OR ADD A NEW VALUE HERE 
				// MAKE SURE YOU UPDATE THE XFA_LEGACY_XX_DEFAULT setting in  XFATemplateModel.h
				
				// Get the defaults depending on the XFA version.
				nVersion = getVersion(sPIValue);
				mnLegacyInfo = getXFALegacyDefaultsForXFAVersion(nVersion);
				mnOriginalVersion = nVersion;
			}
			
			LegacyMask oLegacyInfo = new LegacyMask(mnLegacyInfo);
			
			updateLegacyInfo(sPIValue, "LegacyPositioning", AppModel.XFA_LEGACY_POSITIONING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "LegacyEventModel", AppModel.XFA_LEGACY_EVENTMODEL, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "LegacyPlusPrint", AppModel.XFA_LEGACY_PLUSPRINT, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "LegacyXFAPermissions", AppModel.XFA_LEGACY_PERMISSIONS, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "LegacyCalcOverride", AppModel.XFA_LEGACY_CALCOVERRIDE, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "LegacyRendering", AppModel.XFA_LEGACY_RENDERING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.6-hiddenPositioned", AppModel.XFA_LEGACY_V26_HIDDENPOSITIONED, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.7-multiRecordContextCache", AppModel.XFA_LEGACY_V27_MULTIRECORDCONTEXTCACHE, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.7-layout", AppModel.XFA_LEGACY_V27_LAYOUT, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.7-scripting", AppModel.XFA_LEGACY_V27_SCRIPTING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.7-eventModel", AppModel.XFA_LEGACY_V27_EVENTMODEL, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.7-traversalOrder", AppModel.XFA_LEGACY_V27_TRAVERSALORDER, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.7-XHTMLVersionProcessing", AppModel.XFA_LEGACY_V27_XHTMLVERSIONPROCESSING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.7-accessibility", AppModel.XFA_LEGACY_V27_ACCESSIBILITY, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.8-layout", AppModel.XFA_LEGACY_V28_LAYOUT, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.8-scripting", AppModel.XFA_LEGACY_V28_SCRIPTING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.9-layout", AppModel.XFA_LEGACY_V29_LAYOUT, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.9-scripting", AppModel.XFA_LEGACY_V29_SCRIPTING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v3.0-layout", AppModel.XFA_LEGACY_V30_LAYOUT, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v3.0-scripting", AppModel.XFA_LEGACY_V30_SCRIPTING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.9-traversalOrder", AppModel.XFA_LEGACY_V29_TRAVERSALORDER, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "patch-B-02518", AppModel.XFA_PATCH_B_02518, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v2.9-fieldPresence", AppModel.XFA_LEGACY_V29_FIELDPRESENCE, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "patch-W-2393121", AppModel.XFA_PATCH_W_2393121, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "patch-W-2447677", AppModel.XFA_PATCH_W_2447677, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "patch-W-2757988", AppModel.XFA_PATCH_W_2757988, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v3.2-scripting", AppModel.XFA_LEGACY_V32_SCRIPTING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v3.1-layout", AppModel.XFA_LEGACY_V31_LAYOUT, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v3.2-layout", AppModel.XFA_LEGACY_V32_LAYOUT, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v3.2-rendering", AppModel.XFA_LEGACY_V32_RENDERING, oLegacyInfo, sBehaviorOverride);
			updateLegacyInfo(sPIValue, "v3.2-canonicalization", AppModel.XFA_LEGACY_V32_CANONICALIZATION, oLegacyInfo, sBehaviorOverride);

			//Note: Update the layout legacy flags in ascending order since the older ones override earlier ones.
			//Also, if you add a new vX.Y-layout legacy flag you need to update ::updateLegacyFlags(...)

			mnLegacyInfo = new LegacyMask(oLegacyInfo);
			
			assert(mnOriginalVersion == nVersion);			

			// update the Legacy flags according to current XFA version
			if (nDeprecatedFlags.isNonZero()){
				LegacyMask nDefaultFlags = this.mnLegacyInfo.and(nDeprecatedFlags.not());
				this.mnLegacyInfo = nDefaultFlags;
				this.updateOriginalXFAVersionPI();
			}
			
			if (!StringUtils.isEmpty(sBehaviorOverride) || nDeprecatedFlags.isNonZero())
				updateOriginalXFAVersionPI();
	    }
	    else if (Assertions.isEnabled) {
			String sPIValue = getOriginalVersion(true);
			int nVersion = getVersion(sPIValue);
			// Javaport: If a fixup of the version was done (21 -> 22) (see
			// "Fix for Watson 1616502"), the mnOriginalVersion value is not
			// updated, so we have a situation with a pi that states 2.2 but \
			// a version value of 2.1. Because of legacy issues we can't change 
			// this. Maybe we should back-port this to C++.
			assert((mnOriginalVersion == nVersion) || ((mnOriginalVersion == 21) && (nVersion == 21)));
	    }
	    
	    return mnOriginalVersion;
	}
	
	private LegacyMask getXFALegacyDefaultsForXFAVersion(int nXFAVersion) {
		LegacyMask nVersion = new LegacyMask(0);
		
		if (Assertions.isEnabled) {
			// If this asserts fire, the head version has changed.
			// You will need to update this assert as well as the code
			// below to support this new version.
			assert(Schema.XFAVERSION_35 == Schema.XFAVERSION_HEAD);
		}
		
		// Set the default legacy flags depending on the version.
		switch (nXFAVersion)
		{
			case Schema.XFAVERSION_21:
				nVersion = AppModel.XFA_LEGACY_21_DEFAULT;
				break;
			case Schema.XFAVERSION_22:
				nVersion = AppModel.XFA_LEGACY_22_DEFAULT;
				break;
			case Schema.XFAVERSION_23:
				nVersion = AppModel.XFA_LEGACY_23_DEFAULT;
				break;
			case Schema.XFAVERSION_24:
				nVersion = AppModel.XFA_LEGACY_24_DEFAULT;
				break;
			case Schema.XFAVERSION_25:
				nVersion = AppModel.XFA_LEGACY_25_DEFAULT;
				break;
			case Schema.XFAVERSION_26:
				nVersion = AppModel.XFA_LEGACY_26_DEFAULT;
				break;
			case Schema.XFAVERSION_27:
				nVersion = AppModel.XFA_LEGACY_27_DEFAULT;
				break;
			case Schema.XFAVERSION_28:
				nVersion = AppModel.XFA_LEGACY_28_DEFAULT;
				break;
			case Schema.XFAVERSION_29:
				nVersion = AppModel.XFA_LEGACY_29_DEFAULT;
				break;
			case Schema.XFAVERSION_30:
				nVersion = AppModel.XFA_LEGACY_30_DEFAULT;
				break;
			case Schema.XFAVERSION_31:
				nVersion = AppModel.XFA_LEGACY_31_DEFAULT;
				break;
			case Schema.XFAVERSION_32:
				nVersion = AppModel.XFA_LEGACY_32_DEFAULT;
				break;
			case Schema.XFAVERSION_33:
				nVersion = AppModel.XFA_LEGACY_33_DEFAULT;
				break;
			case Schema.XFAVERSION_34:
				nVersion = AppModel.XFA_LEGACY_34_DEFAULT;
				break;
			case Schema.XFAVERSION_35:
				nVersion = AppModel.XFA_LEGACY_35_DEFAULT;
				break;
		default:
				// This is an unrecognized namespace.

				// Fix for Watson 1706661.
				// If the namespace is higher than our current namespace,
				// then default to our head namespace, otherwise set it to
				// the initial namespace.  In any case warn.
				int nDefault = 0;
				if (nXFAVersion > Schema.XFAVERSION_HEAD) {
					nVersion = AppModel.XFA_LEGACY_HEAD_DEFAULT;
					nDefault = Schema.XFAVERSION_HEAD;
				}
				else {
					nVersion = AppModel.XFA_LEGACY_21_DEFAULT;
					nDefault = Schema.XFAVERSION_21;
				}

				// Populate %0 for the message, which is the
				// unrecognized namespace.
				// Convert namespace to a decimal. i.e. 21 -> 2.1
				String sVer = "0.0";
				if (nXFAVersion > 0) {
					double dVersion = nXFAVersion / 10.0;
					sVer = Numeric.doubleToString(dVersion, 1, false);
				}
				
				// Populate %1 for the message, which is the
				// namespace we are defaulting to.
				// Convert namespace to a decimal. i.e. 21 -> 2.1
				String sVerDefault = "0.0";
				if (nDefault > 0) {
					double dDefault = nDefault / 10.0;
					sVerDefault = Numeric.doubleToString(dDefault, 1, false);
				}

				// Issue a warning that the namespace is unrecognized.
				// "Unrecognized namespace XFA %0, defaulting to XFA %1"
				MsgFormatPos oMessage = new MsgFormatPos(ResId.UnRecognizedNameSpaceWarning);
				oMessage.format(sVer);
				oMessage.format(sVerDefault);
				ExFull err = new ExFull(oMessage);
				addErrorList(err, LogMessage.MSG_WARNING, this);
		}
		
		return nVersion;
	}
	
	private LegacyMask getXFALegacyDeprecatedForXFAVersion(int nXFAVersion){
		if (Assertions.isEnabled) {
			// If this asserts fire, the head version has changed.
			// You will need to update this assert as well as the code
			// below to support this new version.
			assert(Schema.XFAVERSION_35 == Schema.XFAVERSION_HEAD);
		}
		
		LegacyMask nVersion = new LegacyMask(0);
		// Set the default legacy flags depending on the version.
		switch (nXFAVersion)
		{
			case Schema.XFAVERSION_21:
				nVersion = AppModel.XFA_LEGACY_21_DEPRECATED;
				break;
			case Schema.XFAVERSION_22:
				nVersion = AppModel.XFA_LEGACY_22_DEPRECATED;
				break;
			case Schema.XFAVERSION_23:
				nVersion = AppModel.XFA_LEGACY_23_DEPRECATED;
				break;
			case Schema.XFAVERSION_24:
				nVersion = AppModel.XFA_LEGACY_24_DEPRECATED;
				break;
			case Schema.XFAVERSION_25:
				nVersion = AppModel.XFA_LEGACY_25_DEPRECATED;
				break;
			case Schema.XFAVERSION_26:
				nVersion = AppModel.XFA_LEGACY_26_DEPRECATED;
				break;
			case Schema.XFAVERSION_27:
				nVersion = AppModel.XFA_LEGACY_27_DEPRECATED;
				break;
			case Schema.XFAVERSION_28:
				nVersion = AppModel.XFA_LEGACY_28_DEPRECATED;
				break;
			case Schema.XFAVERSION_29:
				nVersion = AppModel.XFA_LEGACY_29_DEPRECATED;
				break;
			case Schema.XFAVERSION_30:
				nVersion = AppModel.XFA_LEGACY_30_DEPRECATED;
				break;
			case Schema.XFAVERSION_31:
				nVersion = AppModel.XFA_LEGACY_31_DEPRECATED;
				break;
			case Schema.XFAVERSION_32:
				nVersion = AppModel.XFA_LEGACY_32_DEPRECATED;
				break;
			case Schema.XFAVERSION_33:
				nVersion = AppModel.XFA_LEGACY_33_DEPRECATED;
				break;
			case Schema.XFAVERSION_34:
				nVersion = AppModel.XFA_LEGACY_34_DEPRECATED;
				break;
			case Schema.XFAVERSION_35:
				nVersion = AppModel.XFA_LEGACY_35_DEPRECATED;
				break;				
			default:

				// This is an unrecognized namespace.

				// If the namespace is higher than our current namespace,
				// then default to our head namespace, otherwise set it to
				// the initial namespace.  In any case warn.
				int nDefault = 0;
				if (nXFAVersion > Schema.XFAVERSION_HEAD)
				{
					nVersion = AppModel.XFA_LEGACY_HEAD_DEPRECATED;
					nDefault = Schema.XFAVERSION_HEAD;
				}
				else
				{
					nVersion = AppModel.XFA_LEGACY_21_DEPRECATED;
					nDefault = Schema.XFAVERSION_21;
				}

				// Populate %0 for the message, which is the
				// unrecognized namespace.
				// Convert namespace to a decimal. i.e. 21 -> 2.1
				String sVer = "0.0";				
				if (nXFAVersion > 0) {
					double dVersion = nXFAVersion / 10.0;
					sVer = Numeric.doubleToString(dVersion, 1, false);
				}

				// Populate %1 for the message, which is the
				// namespace we are defaulting to.
				// Convert namespace to a decimal. i.e. 21 -> 2.1
				String sVerDefault = "0.0";
				if (nDefault > 0) {
					double dDefault = nDefault / 10.0;
					sVerDefault = Numeric.doubleToString(dDefault, 1, false);
				}

				// Issue a warning that the namespace is unrecognized.
				// "Unrecognized namespace XFA %0, defaulting to XFA %1"
				MsgFormatPos oMessage = new MsgFormatPos(ResId.UnRecognizedNameSpaceWarning);
				oMessage.format(sVer);
				oMessage.format(sVerDefault);
				ExFull err = new ExFull(oMessage);
				addErrorList(err, LogMessage.MSG_WARNING, this);
		}

		return nVersion;
	}
	
	/**
	 * @exclude from published api.
	 */
	public ScriptTable getScriptTable() {
		return TemplateModelScript.getScriptTable();
	}
	
	/**
	 * Gets the XFATemplateResolver.
	 * @return the XFATemplateResolver.
	 *
	 * @exclude from published api.
	 */
	TemplateResolver getTemplateResolver() {
		return moTemplateResolver;
	}

	/**
	 * indicate whether this template has any server-side scripts (marked runAt="server")
	 * @return true if there are server-side scripts.
	 * @exclude
	 */
	boolean hasServerSideScripts() {
		return mbHasServerSideScript;
	}

	void invalidMethod(char[] sName) {
		// we needed to add the form methods into the template (see below in
		// the scripting section) because Designer (for example) uses
		// intellisence
		// and since it doesn't have a form DOM it can show the $form
		// properties.
		// By adding the methods here they can actually populate the list of
		// properties
		// for $form.

		// I'm not sure if you should show an error message or do nothing.
		// to be consistent I think we should do nothing.

		// the template objects that don't implement certain form methods and/or
		// properties
		// don't output an error message (e.i field.resetData). The only place
		// where we
		// actually trow errors is in xfahostpseudomodel.cpp

		// the code below can be uncommented if we decide to show error
		// messages.

		// String gsTemplate("$template");

		// MsgFormatPos oMessage = new MsgFormatPos(InvalidMethodException);
		// oMessage << gsTemplate << sName;
		// throw new ExFull(oMessage);
	}

	/**
	* Helper function that compare if two Corners have the same setting.
	* @param pCornerL the corner on the LHS of the equality statement
	* @param pCornerR the corner on the RHS of the equality statement
	* @exclude
	*/
	boolean isCornerEqual(Element pCornerL, Element pCornerR) {
		return true;
	}
	/**
	* Helper function that compare if two edges have the same setting.
	* @param pEdgeL the edge on the LHS of the equality statement
	* @param pEdgeR the edge on the RHS of the equality statement
	* @exclude
	*/
	boolean isEdgeEqual(Element pEdgeL, Element pEdgeR) {
		return true;
	}
	
	boolean isViewAdmissable(Element node, String aNodeName) {
		//Strategy
		//Return true if node is to be loaded based on current view,
		// and false otherwise.
		
		// Get the view from the config DOM. Only do this once.
		if (!mbCheckRelevant) {
			Node oRelevant = resolveNode("config.present.common.template.relevant");
			if (oRelevant != null) {
				if(1 == oRelevant.getXFAChildCount()) {
					TextNode textNode = (TextNode)oRelevant.getFirstXFAChild();
					String sRelevant = textNode.getValue();
					
					// Store in a jfStorage to avoid string parsing
					if (!StringUtils.isEmpty(sRelevant)) {
						String[] tokens = sRelevant.split(" ");
						for (int i=0; i<tokens.length; i++) {
							moRelevantList.add(tokens[i]);
						}
					}
				}
			}
			mbCheckRelevant = true;
		}
		
		int nRelvantListSize = moRelevantList.size();
		
		// If there's no view specified, then all nodes are admissable.
		if (nRelvantListSize == 0)
			return true;
		
		//This is called for every dom node in the template
		//so we need to minimize the effect on performance,
		//Filter out the nodes that are not affected by view settings.
		//Yes this requires maintenance if the list of nodes that support
		//views changes but it's better than querying every dom node to see
		//if it contains the 'view' attribute - the vast majority won't.
		if(!isViewSensitiveNode(node, aNodeName))
			return true;
		
		// OK, node is view-sensitive.
		// 1. Check the relevant attribute of this node. This attribute contains
		// the view information, if specified.
		// 2. If the relevant attribute doesn't exist, then this node is not part
		// of a view and is therefore admissable.
		// 3. If the relevant attribute is present, then it may contain an empty
		// string, a single value, or a space delimited list of values. We
		// will parse it out and determine if node is admissable into the
		// current view.
		//
		// The syntax is: viewname or -viewname.
		//
		// relevant="manager employee" OK
		// relevant="-employee -english" OK
		// relevant="-employee manager" OK
		//
		//
		
		// Get the relevant attribute for this node
		int attr = node.findAttr(null, XFA.RELEVANT);
		
		// If no relevant attribute is present, then it's admissable.
		if (attr == -1)
			return true;
		
		// Get this nodes relevant attribute value (view info)
		String sRelevant = node.getAttrVal(attr).trim();
		
		// If the relevant attribute is empty, then it's admissable.
		if (StringUtils.isEmpty(sRelevant))
			return true;
		
		boolean bAdmissable = true; // Will this node be rendered?
		boolean bFound = false;
		
		String[] tokens = sRelevant.split(" ");
		int j = 0;
		while (bAdmissable && j< tokens.length) {
			String sViewName = tokens[j];
			boolean bIsMinus = sViewName.length() > 0 && sViewName.charAt(0) == '-';
			
			// Watson 1384530, the + is optional
			// and should be stripped.
			boolean bIsPlus = sViewName.length() > 0 && sViewName.charAt(0) == '+';

			if (bIsMinus || bIsPlus)
				sViewName = sViewName.substring(1);
			
			for (int i = 0; i < nRelvantListSize; i++) {
				if (sViewName.equals(moRelevantList.get(i))) {
					if (bIsMinus)
						bAdmissable = false;
					
					bFound = true;
					break;
				}
			}
			
			if (!bFound && !bIsMinus)
				bAdmissable = false;
			
			j++;
		}
		
		return bAdmissable;
	}

	@FindBugsSuppress(code="ES")
	boolean isViewSensitiveNode(Node node, String aNodeName) {
		// Strategy
		// Return true if node is a view sensitive node and false otherwise.
		if (node == null || aNodeName == "")
			return false;

		if (node instanceof Element
				&& (aNodeName == XFA.SUBFORM || aNodeName == XFA.SUBFORMSET
						|| aNodeName == XFA.CONTENTAREA
						|| aNodeName == XFA.PAGEAREA
						|| aNodeName == XFA.PAGESET || aNodeName == XFA.AREA
						|| aNodeName == XFA.VALUE || aNodeName == XFA.FIELD
						|| aNodeName == XFA.DRAW || aNodeName == XFA.BORDER || aNodeName == XFA.EXCLGROUP))
			return true;

		return false;
	}

	/**
	 *
	 * @exclude from published api.
	 */
	public void loadNode(Element parent, Node node, Generator generator) {
		// JavaPort: Most of the work that would have been done here in C++
		// is done during DOM creation. The code that used to be in doLoadNode 
		// is now in the overloaded createElement().
		
		if (node instanceof Element)
			doLoadNode((Element)node);
	}
	
	/** @exclude from published api. */
	@FindBugsSuppress(code="ES")
	public boolean loadSpecialAttribute(Attribute attribute, String aLocalName, Element element, boolean bCheckOnly) {
	
		if (aLocalName == XFA.MATCH &&
			element.getClassTag() == XFA.BINDTAG &&
			attribute.getAttrValue().equals(STRS.GLOBAL)) {
			setAsGlobalField(element.getXFAParent());
			return true;
		}

		// for PA only, cache if this doc has any server side scripts
		if (!mbHasServerSideScript &&
			aLocalName == XFA.RUNAT &&
			element.getClassTag() == XFA.SCRIPTTAG &&
			attribute.getAttrValue().equals(STRS.SERVER)) {
			mbHasServerSideScript = true;
			return true;
		}

		return super.loadSpecialAttribute(attribute, aLocalName, element, bCheckOnly);
	}

	/**
	 *
	 * @exclude from published api.
	 */
	public void nodeCleanup(Node node, boolean bHasAttrs, boolean bHasChildren) {
		
		// Javaport:  Move this logic earlier in the Java version so that the 
		// remaining method can assume we're dealing with an Element.

		// We used to check: pNode->isSameClass(XFA::TEXTNODETAG)
		// but now that we have SVGTextData, we need a more clever check...
		if (node instanceof Chars) {
			Chars x = (Chars)node;
			if (x.getData().length() == 0)
				node.remove(); // skip remove() level, which stacks an undo frame
			
			return;
		}
		
		Element element = (Element)node;
		
		//
		// All transient nodes should be removed...
		// Otherwise we occasionally get nodes that think they have children
		// when they really don't
		//
		boolean isTransient = element.isTransient();
		if (isTransient) {
			element.remove();
			return;
		}

		// Special handling for border/rectangle node:
		// if all four edges are the same, just keep one; if all four corners are the same, just keep one.
		if (element.isSameClass(XFA.BORDERTAG) || element.isSameClass(XFA.RECTANGLETAG)) {
			int nChildren = element.getXFAChildCount();
			if (nChildren > 3) {
				Element edges[] = new Element[4];
				Element corners[] = new Element[4];

				// check if there are four edges and four corners
				// If there are more than 4 edges, or 4 corners,
				// then delete them as it's a violation of the spec.
				int nEdges = 0;
				int nCorners = 0;
				Node child = element.getFirstXFAChild();
				while (child != null) {
					if (child.isSameClass(XFA.EDGETAG)) {
						if (nEdges<4) {
							edges[nEdges] = (Element)child;
							nEdges++;
						}
						else {
							// Remove edges > 4
							Node nextChild = child.getNextXFASibling();
							child.remove();
							child = nextChild;
							continue;
						}
					}
					else if (child.isSameClass(XFA.CORNERTAG)) {
						if (nCorners<4) {
							corners[nCorners] = (Element)child;
							nCorners++;
						}
						else {
							// Remove corners > 4
							Node nextChild = child.getNextXFASibling();
							child.remove();
							child = nextChild;
							continue;
						}
					}
					child = child.getNextXFASibling();
				}

				if (nEdges == 4) {
					// if four edges have the same setting, only keep one.
					if (isEdgeEqual(edges[0], edges[1]) && 
						isEdgeEqual(edges[0], edges[2]) && 
						isEdgeEqual(edges[0], edges[3])) {
						edges[1].remove();
						edges[2].remove();
						edges[3].remove();
					}
				}

				if (nCorners == 4) {
					// if four corners have the same setting, only keep one.
					if (isCornerEqual(corners[0], corners[1]) && 
						isCornerEqual(corners[0], corners[2]) && 
						isCornerEqual(corners[0], corners[3])) {
						corners[1].remove();
						corners[2].remove();
						corners[3].remove();
					}
				}
			}
		}
		
		// When we encounter a <caption> with no <value> child ... delete it!
		// However, the value could be bound dynamically. Therefore, the checks for
		// placement, font, para should be done before deleting the node.
		// Refer to Watson bug #1837258.
		// Also added  cleanups for the parent elements for font and para are the same
		if (element.isSameClass(XFA.CAPTIONTAG)) {
			// check for font before removing the node. Watson 1837258
	        Element parent = element.getXFAParent();
			Node child = element.locateChildByClass(XFA.FONTTAG, 0);

			if (child != null) {
				Element parentElem = parent.getElement(XFA.FONTTAG, true, 0, false, false);
				if (parentElem != null) { 
					// parent does not have same element.
					if (child.compareVersions(parentElem, null, null))
						child.remove();
				}
			}

			//unless it has a reserve attribute ... don't delete it
			Measurement oReserve = (Measurement)element.getAttribute(XFA.RESERVETAG, true, false);
			if (oReserve != null)
				return;

			//or unless the field has a rollover or button down caption ... don't delete it
			Node items = element.getXFAParent().getElement(XFA.ITEMSTAG,true,0,false,false);
			if (items != null) {
				Node oItem = items.getFirstXFAChild();
				while (oItem != null) {
					if (oItem instanceof Element) {
						Element item = (Element)oItem;
						if (item.peekAttribute(XFA.NAMETAG) != null) {
							String sName = item.peekAttribute(XFA.NAMETAG).toString();
							if (sName.equals("rollover") || sName.equals("down"))
								return;
						}
					}
					oItem = oItem.getNextXFASibling();
				}
			}
			NodeList children = element.getNodes();
			int n = children.length();
			for (int i=0; i<n; i++) {
				if (children.item(i).isSameClass(XFA.VALUE))
					return;
			}
			element.remove();
			return;
		}
		
		// We have have just removed this node
		if (element.getXFAParent() == null)
			return;
		
		int nodeTag = element.getClassTag();
		int parentTag = element.getXFAParent().getClassTag();
		
		// fill elements with a hidden presence can be eliminated
		if (!bHasChildren && nodeTag == XFA.FILLTAG && parentTag != XFA.FONTTAG) {
			EnumValue presence = (EnumValue)element.getAttribute(XFA.PRESENCETAG, true,false);
			
			if (presence != null && (presence.getInt() == EnumAttr.PRESENCE_HIDDEN ||
					                 presence.getInt() == EnumAttr.PRESENCE_INVISIBLE ||
					                 presence.getInt() == EnumAttr.PRESENCE_INACTIVE)) {
				element.remove(); // skip remove() level, which stacks an undo frame
				return;
			}
		}
		
		if (bHasAttrs || bHasChildren) {
			return;
		}
		
		// We may safely delete <fill> tags only if they're the child of a <font>
		if (nodeTag == XFA.FILLTAG && parentTag != XFA.FONTTAG)
			return;
		
		
		// We may safely delete <margin> tags only if they're the child of a
		// <field> or <subform> or <draw> or <border> or <caption>
		if ((nodeTag == XFA.MARGINTAG) &&
				(parentTag != XFA.FIELDTAG) &&
				(parentTag != XFA.DRAWTAG) &&
				(parentTag != XFA.SUBFORMTAG) &&
				(parentTag != XFA.BORDERTAG) &&
				(parentTag != XFA.CAPTIONTAG))
		{
			return;
		}
		
		
		if (nodeTag == XFA.CORNERTAG) {
			boolean isDefault = isDefault(false);
			if (isDefault) {
				element.remove();
			}
			return;
		}
		
		// We can't delete empty edge, border and corner tags, since the default
		// is something different when not specified.
		if (nodeTag == XFA.EDGETAG || nodeTag == XFA.BORDERTAG || nodeTag ==
			XFA.CORNERTAG || nodeTag == XFA.COMBTAG)
			return;
		
		element.remove(); // skip remove() level, which stacks an undo frame
	}

	/**
	 * Return TRUE if the given attribute can be removed, aka cleaned up from a given node. 
	 * Return FALSE otherwise. This was private in C++.
	 *
	 * @exclude from published api.
	 */
	public boolean doAttributeCleanup(Node pNode, int eAttributeTag, String sAttrValue)	{
		// H and W have complicated default mechanisms, so it's not really safe to remove them.
		// Ditto for leaders and trailers (watson 2322044)
		if( eAttributeTag == XFA.HTAG    ||
			eAttributeTag == XFA.WTAG    ||
			eAttributeTag == XFA.INITIALNUMBERTAG ||
			eAttributeTag == XFA.OVERFLOWLEADERTAG ||  
			eAttributeTag == XFA.OVERFLOWTRAILERTAG ||
			eAttributeTag == XFA.LEADERTAG ||
			eAttributeTag == XFA.TRAILERTAG)
		{
			return false;
		}

		return super.doAttributeCleanup(pNode, eAttributeTag, sAttrValue);
	}
	
	private void orderContainers(Element node) {
		
		List<Container> containers = new ArrayList<Container>();
		
		// Accumulate Containers.
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child instanceof Container) {
				containers.add((Container)child);
			}
		}

		// Sort the containers.
		Collections.sort(containers, new Comparator<Container>() {
			
			public int compare(Container left, Container right) {
			
				// Get the y ordinate of the left and right containers
				double nYLeft = ((Measurement)left.getAttribute(XFA.YTAG)).getValueAsUnit(EnumAttr.MILLIMETER);
				double nYRight = ((Measurement) right.getAttribute(XFA.YTAG)).getValueAsUnit(EnumAttr.MILLIMETER);

				if (nYLeft < nYRight)
					return -1;
				
				if (nYLeft > nYRight)
					return 1;

				// Get the x ordinate of the left and right containers
				double nXLeft = ((Measurement) left.getAttribute(XFA.XTAG)).getValueAsUnit(EnumAttr.MILLIMETER);
				double nXRight = ((Measurement) right.getAttribute(XFA.XTAG)).getValueAsUnit(EnumAttr.MILLIMETER);

				return nXLeft < nXRight ? -1 : nXLeft > nXRight ? 1 : 0;
			}
		});

		// Re-insert the containers in order.
		for (Node child : containers) {
			node.appendChild(child, false);
			((Element)child).setModel(this);
		}
	}

	//
	// Tail recursive function that orders the Template DOM nodes geographically starting
	// from the bottom up.
	//
	private void orderNodes(Element node) {
		
		if (node.getFirstXFAChild() == null)
			return;
		
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			//
			// Tail recursion.
			//
			if (child instanceof Element)
				orderNodes((Element) child);
		}
		
		orderContainers(node);
	}

	/**
	 *
	 * @exclude from published api.
	 */
	public void postLoad() {
		
		Element startNode = this;
		
		isLoading(true);
		
		initGenerator();
		//fix for watson bug#2426694 //do this versioning thing before loading the dom
		
		// Fix for Watson 1616502.
		// When you call mergedXDP and save out the packets, the XDP that
		// we return back has a generator of XFA2_4.  The generator shouldn't matter
		// after this point, since we've already updated all of the models during
		// the mergedXDP step, with one exception!
		// If the original generator was Adobe Designer 7.0 and the template namespace
		// version is 2.1 then we want to bump this to 2.2 due to a bug
		// that the original 7.0 Designer that didn't update the namespace.
		String sOriginalNS = getOriginalVersion(false);
		if (StringUtils.isEmpty(sOriginalNS)) {
			// if generator was 7.0 and the version is 2.1  then return 2.2
			// this is due to a bug that the original 7.0 designer didn't update the NS
			if ((meInputGenerator == Generator.XFAGen_AdobeDesignerV70) && 
				(mnOriginalTemplateVersion == 21)) {
				// update the document to ensure we have the proper namespace settings
				setCurrentVersion(Schema.XFAVERSION_22);
				setNeedsNSNormalize(true);
				mnOriginalTemplateVersion = 22;
				setNameSpaceURI(STRS.XFATEMPLATENS_TWODOTTWO, false, false, true);			
			}
		}

		loadNode(getXFAParent(), startNode, getGenerator());
		
		// Watson 1396446: When importing old xft files, that don't have pagesets,
		// the master page is put at the end of the hierarchy.  This is because we
		// append the pageset when we create it for these old files.  I've tried
		// doing the insertBefore in the fixup code, but there's to many other
		// fixups that are happening that messes things up, so I'm doing the change
		// here after the template model has been loaded.

		// Strategy:
		// The template model is now loaded, so..
		// 1. get the top level subform
		// 2. find the pageset
		// 3. if the pageset is not the first object under the top level subform,
		//    then insert it before the first object under the top level subform.
		//
		// NOTE: IF WE EVER SUPPORT MORE THAN ONE TOP LEVEL SUBFORM THEN WE MAY
		//       NEED TO REVISIT THIS CODE.

		// Get the number of children for this template model.
		Node pChild = getFirstXFAChild();
		if (pChild != null) {
			// Init some variables
			boolean bFoundTopLevelSubform = false;

			// Find the first top level subform, there could be extras
			// at this level as well.
			while (pChild != null) {
				if (pChild.isSameClass(XFA.SUBFORMTAG)) {
					bFoundTopLevelSubform = true;
					break;
				}
				pChild = pChild.getNextXFASibling();
			}

			if (bFoundTopLevelSubform) {
				// OK, got the top level subform.
				Element pTopLevelSubform = (Element) pChild;

				pChild = pTopLevelSubform.getFirstXFAChild();

				if (pChild != null) {
					boolean bFound = false;
					int i = 0;

					// Find the pageSet
					while (pChild != null) {
						if (pChild.isSameClass(XFA.PAGESETTAG)) {
							bFound = true;
							break;
						}
						i++;
						pChild = pChild.getNextXFASibling();
					}

					// If we found the pageSet, and it isn't the first child of
					// the
					// top level subform, then insert the pageSet before it.
					if (bFound && i > 0)
						pTopLevelSubform.insertChild(pChild, pTopLevelSubform.getFirstXFAChild(), false);

				} // if nTopLevelSubformChildren > 0
			} // bFoundTopLevelSubform
		} // if nTemplateChildren > 0


		
		// If the SourceSet Model was initialized, then
		// walk through the list of data <source> children
		// and load them.
		
		
		// TODO JavaPort: Don't expect sourceSet to work, since it's all about ADO,
		// and we're Java...
//		if (mbIsSourceSetModelInitialized) {
//			jfDomDocument doc = mpoSourceSetModel.getDomPeer().getOwnerDocument();
//			
//			
//			int nCount = moSourceList.size();
//			for (int i=0; i < nCount; i++) {
//				// Get the source name
//				jfDomElement oSource(moSourceList.get(i));
//				// TODO use model NS
//				doc.addId(String.EmptyString(), XFA.idTag(),oSource);
//				
//				// Ask the SourceSetModel to load the source node.
//				mpoSourceSetModel.loadNode(mpoSourceSetModel, oSource, genTag);
//			}
//		}

		
		// Resolve nodes that refer elsewhere
		resolveProtos(false);

		// Javaport: Since no fix-ups are done to the template model 
		// during the load (only one DOM - remember?) we need to do 
		// them after the fact.
		TemplateModelFixup tmf = null;
		if (mbApplyFixups) {
			tmf = new TemplateModelFixup(this, mbFixupRenderCache);
			tmf.processFixups();
		}
		
		if (mbApplyFixups && tmf != null) {
			tmf.applyFixups(this);
		}
		
		
		isLoading(false);
	}
	
	private void doLoadNode(Element element) {
		try {
			// Javaport: This method does some of the work that is done in XFATemplateModelImpl::doLoadNode.
			// Look for ExData elements contained within fields, and ensure that 
			// the content that it contains has the correct class assigned.
			// There is other work that the C++ implementation does (raising exceptions,
			// fiddling with whitespace) that is not reproduced here.
			
			final int nAttributes = element.getNumAttrs();
			for (int i = 0; i < nAttributes; i++) {
				Attribute attr = element.getAttr(i);
				loadSpecialAttribute(attr, attr.getLocalName(), element, false);
			}
			
			if (element instanceof ExDataValue) {
				
				ExDataValue exData = (ExDataValue)element;
				
				boolean bIsRichText = false;
				boolean bIsXmlContent = false;
				int index = exData.findAttr(null, XFA.CONTENTTYPE);
				if (index != -1) {
					String contentType = exData.getAttrVal(index);
					
					if (contentType.equals(STRS.TEXTXML))
						bIsXmlContent = true;
					else if (contentType.equals(STRS.TEXTHTML))
						bIsRichText = true;
				}
				
				if (bIsXmlContent) {
					// currently, we should only encounter an exData with a contentType
					// of text/xml for the value of a multiSelect choiceList. If this
					// is not the case, let it fall through.

					bIsXmlContent = false;
					
					Element parent = element.getXFAParent();
					
					if (parent instanceof Value) {
						
						Element grandParent = parent.getXFAParent();
						
						if (grandParent instanceof Field) {
		
							// check for a choiceList ui with an open attribute of multiSelect
							Element ui = (Element)grandParent.locateChildByClass(XFA.UITAG, 0);
							if (ui != null) {
								Element choiceList = (Element)ui.locateChildByClass(XFA.CHOICELISTTAG, 0);
								if (choiceList != null) {
									index = choiceList.findAttr(null, XFA.OPEN);
									if (index != -1) {
										String sOpen = choiceList.getAttrVal(index);
										if (sOpen.equals("multiSelect"))
											bIsXmlContent = true;
									}
								}
							}
						}
					}
				}
				
				boolean previousWillDirty = getDocument().getWillDirty();
				getOwnerDocument().setWillDirty(false);
				// preLoad children
				// This is required to ensure that whitespace removal and consolidation is done
				for (Node child = exData.getFirstXMLChild(); child != null; )
					child = preLoadNode(element, child, getGenerator());
				
				// Remove all the children of element so that we can add them again, with validation.
				Node [] children = new Node[exData.getXMLChildCount()];
				Node nextChild;
				int i = 0;
				for (Node child = exData.getFirstXMLChild(); child != null; child = nextChild, i++) {
					nextChild = child.getNextXMLSibling();
					children[i] = child;
					child.remove();
				}
				
				for (i = 0; i < children.length; i++) {
					Node child = children[i];
					
					if (!(child instanceof TextNode) && !(child instanceof Element)) {
						exData.appendChild(child);
						continue;
					}
					
					boolean bLoadNode = true;
					
					if (bIsRichText) {
						
						if (child instanceof TextNode && ((TextNode)child).isXMLSpace()) {
							// remove leading whitespace nodes
							continue;
						}
						
						if (child instanceof Element && ((Element)child).getNS() == STRS.XHTMLNS) {
							
							// Replace this generic Element with a RichTextNode
							RichTextNode newRichTextNode = new RichTextNode(null, null);
							if (convertGenericContentElement(exData, (Element)child, newRichTextNode)) {				
							
								// Do we need to bump the template version?
								// watson bug 1648640, don't bump the xfa version while we load in xml content by default
								// watson bug 1744498 - Designer needs to bump version by setting mbBumpXFAVersionLoadingRichText
								RichTextNode newRichText = (RichTextNode)newRichTextNode;
								if (!validateUsage(newRichText.getVersionRequired(), 0 , getAppModel().bumpVersionOnRichTextLoad())) {
									// TODO need better xhtml load error here
									// We've already loaded the xhtml, AXTE will take care of ignoring anything unwanted
									MsgFormatPos reason = new MsgFormatPos(ResId.InvalidChildVersionException);
									reason.format("");
									reason.format(newRichText.getClassAtom());
									addErrorList(new ExFull(reason), LogMessage.MSG_WARNING, newRichText);
								}
							}
							
							bLoadNode = false;
						}
					}
					
					if (bIsXmlContent) {
						
						if (child instanceof TextNode && ((TextNode)child).isXMLSpace()) {
							// remove leading whitespace nodes
							continue;
						}
						
						if (child instanceof Element) {
							// Replace this generic Element with an XMLMultiSelectNode
							XMLMultiSelectNode newMultiSelectNode = new XMLMultiSelectNode(null, null);
							convertGenericContentElement(exData, (Element)child, newMultiSelectNode);
							
							bLoadNode = false;
						}
					}
					
					if (bLoadNode) {
						if (child instanceof Element) {
							Element childElement = (Element)child;
							// This section approximates the code that appears in the C++ at the top of this method.
							// In this implementation, we know that it is only being applied to element children of exData, 
							// so ....
							
							// ignore any element belonging to a namespace other than XFATemplate
							
							int eTag = getSchema().getElementTag(childElement.getNS(), childElement.getLocalName());
							if (eTag != -1 && (childElement.getNS() == "" || childElement.getNS().startsWith(STRS.XFATEMPLATENS))) {					
								// If we get here, we know that the generic element that was created
								// at parse time is invalid for an exData, so we can simply log an error.
								// Note that this exception will end up out of order with respect to the C++.
		                        MsgFormatPos msg = new MsgFormatPos(ResId.InvalidChildAppendException, exData.getClassName());
		                        msg.format(((Element) child).getLocalName());
		                        ExFull ex = new ExFull(msg);
		                        addXMLLoadErrorContext(child, ex);
		                        getModel().addErrorList(ex, LogMessage.MSG_VALIDATION_ERROR, exData);
							}
							else {
								childElement.inhibitPrettyPrint(true);
							}
							
							// ...and append it anyway since we keep invalid nodes in exData content
							exData.appendChild(childElement, false);
						}
						else if (child instanceof TextNode) {
							
							// Attempt to insert the node to see if it will be allowed
							try {
								exData.appendChild(child, true);
							}
							catch (ExFull ex) {
								// Append the child anyway, without validation.
								// Note that it won't be part of the parent exData's value, but this approximates what the
								// C++ implementation does by leaving the invalid nodes in the XML DOM.
								exData.appendChild(child, false);
								throw ex;
							}
						}
					}
				}
				getDocument().setWillDirty(previousWillDirty);
			}
			else {
				
				for (Node child = element.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
					if (!(child instanceof Element))
						continue;
					
					doLoadNode((Element)child);
				}
			}
		} catch (ExFull ex) {
			getModel().addErrorList(ex, LogMessage.MSG_VALIDATION_ERROR, null);
		}
	}
	
	/**
	 * Converts a generic Element created at parse time to a specific derived class.
	 * Only contentElement itself it converted - the child nodes are copied, but are
	 * not converted.
	 * @param contentElement the Element to be converted.
	 * @param newNode the Element of some derived type that contentElement is to be copied into.
	 * @return <code>true</code> if the conversion succeeded.
	 */
	private boolean convertGenericContentElement(Element parent, Element contentElement, Element newNode) {
		assert contentElement.getClassTag() == XFA.INVALID_ELEMENT;
		assert newNode.getClassTag() == XFA.RICHTEXTNODETAG || newNode.getClassTag() == XFA.XMLMULTISELECTNODETAG;
		
		newNode.setModel(this);
		newNode.setDocument(getOwnerDocument());
		
		newNode.setDOMProperties(contentElement.getNS(), contentElement.getLocalName(), contentElement.getXMLName(), null);
		newNode.setLineNumber(contentElement.getLineNumber());
		
		for (int i = 0; i < contentElement.getNumAttrs(); i++) {
			Attribute a = contentElement.getAttr(i);
			newNode.setAttribute(a.getNS(), a.getQName(), a.getLocalName(), a.getAttrValue(), false);
		}
		
		// Attempt to insert the node to see if it will be allowed before moving any children
		try {
			parent.appendChild(newNode, true);
		}
		catch (ExFull ex) {
			getModel().addErrorList(ex, LogMessage.MSG_VALIDATION_ERROR, null);
			
			// Leave the generic element in place
			// Note that it won't be part of the parent exData's value, but this approximates what the
			// C++ implementation does by leaving the invalid nodes in the XML DOM.
			parent.appendChild(contentElement, false);
			return false;
		}
		
		// Move the XFA children of contentElement to newNode
		Node nextSibling;
		for (Node child = contentElement.getFirstXMLChild(); child != null; child = nextSibling) {
			nextSibling = child.getNextXMLSibling();
			newNode.appendChild(child, false);
		}
		
		contentElement.remove();
		
		return true;
	}

	// add a global field
	void setAsGlobalField(Node poNodeImpl) {
		String aName = poNodeImpl.getName();
		for (int i = 0; i < moGlobalFields.size(); i++) {
			if (moGlobalFields.get(i) == aName)
				return;
		}

		moGlobalFields.add(poNodeImpl.getName());
	}
	
	/**
	 * Sets the boolean value of a particular legacy settings
	 * @param nLegacyFlags the specific legacy settings to update
	 * @param bValue the new boolean value for the legacy setting
	 *
	 * @exclude from published api.
	 */
	public void setLegacySetting(LegacyMask nLegacyFlags, boolean bValue) {
		// call getOriginalXFAVersion to populate mnLegacyInfo and mnOriginalVersion
		mnOriginalVersion = getOriginalXFAVersion();
		LegacyMask	oLegacyInfo = new LegacyMask(mnLegacyInfo);

		updateLegacyFlags(bValue, nLegacyFlags, oLegacyInfo); // turn on or off the necessary bits
		mnLegacyInfo = new LegacyMask(oLegacyInfo);

		// update the pi
		updateOriginalXFAVersionPI();
	}

	/**
	 * Removes the originalXFAVersion PI if the default legacy flags are equal to the current legacy flags.
	 * @param nIgnoreLegacyFlags - the legacy flags that should be ignored in the comparison.
	 * @return TRUE if the default legacy flags are equal to the current legacy flags, otherwise FALSE.
	 *
	 * @exclude from published api.
	 */
	public boolean cleanupLegacySetting(LegacyMask nIgnoreLegacyFlags /*= 0*/) {
		// call getOriginalXFAVersion() to ensure that mnLegacyInfo is set correctly
		getOriginalXFAVersion();

		int nCurrentVersion = getCurrentVersion();
		LegacyMask nDefaultFlags = new LegacyMask(getXFALegacyDefaultsForXFAVersion(nCurrentVersion));
		LegacyMask nCurrentFlags = new LegacyMask(mnLegacyInfo);

		// if we have flags to ignore set turn them off for both the default and current doc
		if (nIgnoreLegacyFlags.isNonZero()) {
			LegacyMask nDefaultFlagsHolder = new LegacyMask(nDefaultFlags);
			updateLegacyFlags(false, nIgnoreLegacyFlags, nDefaultFlagsHolder);
			nDefaultFlags = new LegacyMask(nDefaultFlagsHolder);
			LegacyMask nCurrentFlagsHolder = new LegacyMask(nCurrentFlags);
			updateLegacyFlags(false, nIgnoreLegacyFlags, nCurrentFlagsHolder);
			nCurrentFlags = new LegacyMask(nCurrentFlagsHolder);
		}

		// the are the same we can remove the original xfa version PI.
		if (nCurrentFlags.equals(nDefaultFlags)) {
			removeOriginalVersionNode();
			return true;
		}
		
		return false;		
	}
	
	/**
	 * @exclude from published api.
	 */
	public void setOriginalXFAVersion(int nXFAVersion) {
		// if 0 restore the defaults and remove the PI
		if (nXFAVersion == 0) {
			mnLegacyInfo = new LegacyMask(0);
		}
		super.setOriginalXFAVersion(nXFAVersion);
	}
	
	/**
	 * Set the Template Resolver to use to collect info used for generating the
	 * form.
	 * 
	 * @param oResolver
	 *            the XFATemplate resolver.
	 */
	void setTemplateResolver(TemplateResolver oResolver) {
		moTemplateResolver = oResolver;
	}
		
	/**
	 * A class to define a mapping between a key and template fragment reference. 
	 * 
	 * @see TemplateModel#splice(List, boolean)
	 */
	public static class Insertion {
		
		private final String key;
		private final String reference;
		private final String containedBySom;
		
		/**
		 * Instantiates an insertion from the given.
		 */
		public Insertion(String key, String reference) {
			this.key = key;
			this.reference = reference;
			this.containedBySom = null;
		}
		
		/**
		 * Instantiates an insertion from the given.
		 */
		public Insertion(String key, String reference, String somContext) {
			this.key = key;
			this.reference = reference;
			this.containedBySom = somContext;
		}
		
		/**
		 * Gets the name that identifies an insertion point.
		 * @return a name that identifies an insertion point.
		 */
		public final String getKey() {
			return key;
		}
		
		/**
		 * Gets the reference that defines a fragment.
		 * @return the reference that defines a fragment.
		 */
		public final String getReference() {
			return reference;
		}
		
		/**
		 * Gets the SOM expression that defines the subtree(s) within the
		 * template that must contain the insertion point in order to be matched.
		 * If the SOM expression is <code>null</code> or empty, this insertion
		 * can be matched anywhere the key matches in the template. 
		 * @return a SOM expression that defines the subtree(s) within the
		 * template where this insertion can be matched.
		 */
		public final String getContainedBySom() {
			return containedBySom;
		}
	}
	
	/*
	 * AdobePatentID="B1081"
	 */
	@SuppressWarnings("unused")
	private final static String Adobe_patent_B1081 = "AdobePatentID=\"B1081\"";
	
	/**
	 * Splices external template fragments into this template.
	 * 
	 * The points at which insertions are to be inserted are indicated by those
	 * elements containing an <code>extras</code> element that in turn contain
	 * a <code>text</code> element with the name "insertionPoint".
	 * <p/>
	 * Elements containing an <code>extras</code> element that in turn contains
	 * a <code>text</code> element with the name "insertionPointPlaceholder"
	 * are removed.
	 * <p/>
	 * The value contained within the <code>text</code> node is an insertion
	 * key, and it is matched against each key in <code>insertions</code>. The
	 * corresponding replacement reference identifies a template fragment that
	 * is loaded and inserted into the template. Each insertion is done by
	 * making a copy of the insertion point and inserting that copy before the
	 * insertion point. The template fragment reference is then resolved, and the
	 * resolved fragment is merged into the insertion point copy.
	 * <p/>
	 * The values in <code>insertions</code> reference template fragments
	 * using the same syntax that the <code>usehref</code> attribute uses to
	 * reference external prototype fragments:
	 * <ul>
	 * <li>URI#ID</li>
	 * <li>URI#som(expression)</li>
	 * <li>URI</li>
	 * </ul>
	 * <p/>
	 * Any errors generated while loading the referenced fragment are copied to
	 * this <code>TemplateModel</code>.
	 * <p/>
	 * The caller is responsible for registering an <code>HrefHandler</code>
	 * with the <code>AppModel</code> to allow any external references to be
	 * resolved.
	 * 
	 * @param insertions
	 *            the insertions to be spliced into the template.
	 * @param removeInsertionPoints
	 *            if <code>true</code>, then any insertion points in the
	 *            template are removed.
	 * @return a <code>List</code> containing the entries from
	 *         <code>insertions</code> for which no corresponding insertion
	 *         point was found, or <code>null</code> if all insertions were
	 *         used.
	 * 
	 * @see AppModel#getErrorList()
	 * @see AppModel#setHrefHandler(HrefHandler handler)
	 */
	public List<Insertion> splice(
			List<Insertion> insertions,
			boolean removeInsertionPoints) {
		
		Set<String> usedKeys = new HashSet<String>(insertions.size());
		List<String> insertionStack = new ArrayList<String>();
		
		spliceInternal(this, insertions, removeInsertionPoints, usedKeys, insertionStack, "");
		
		// Build a list of all insertions that weren't used
		List<Insertion> result = null;
		for (Insertion insertion : insertions) {
			if (!usedKeys.contains(insertion.getKey())) {
				if (result == null)
					result = new ArrayList<Insertion>();
				
				result.add(insertion);
			}				
		}
		
		return result;
	}
	
	private void spliceInternal(
			Element startNode,
			List<Insertion> insertions,
			boolean removeInsertionPoints,
			Set<String> usedKeys,
			List<String> insertionStack,
			String basePath) {
		
		// Traverse startNode's children and look for insertion points and placeholders.
		List<TextValue> insertionPoints = new ArrayList<TextValue>();
		List<TextValue> insertionPointPlaceholders = new ArrayList<TextValue>();
		findInsertionPoints(startNode, insertionPoints, insertionPointPlaceholders);
		
		// Splice in insertions
		for (Insertion insertion : insertions) {
			
			if (insertion.getKey() == null || 
				insertion.getReference() == null || insertion.getReference().trim().length() == 0)
				continue;
			
			for (TextValue textValue : insertionPoints) {
				
				if (!insertion.getKey().equals(textValue.toString()))
					continue;
					
				Element insertionPoint = textValue.getXFAParent().getXFAParent();
				Element insertionPointParent = insertionPoint.getXFAParent();
				
				if (insertionPointParent == null)
					continue;
				
				if (insertion.getContainedBySom() != null && insertion.getContainedBySom().trim().length() != 0) {
					if (!isElementContainedWithinSom(insertionPoint, insertion.getContainedBySom()))
						continue;
				}
				
				// Detect circular references in splicing
				if (insertionStack.contains(insertion.getKey())) {
					StringBuilder buf = new StringBuilder("splice: '");
					for (String key2 : insertionStack) {
						buf.append(key2);
						buf.append("'->'");
					}
					buf.append(insertion.getKey());
					buf.append('\'');
					
					ExFull ex = new ExFull(new MsgFormatPos(ResId.CircularProtoException, buf.toString()));
					addErrorList(ex, LogMessage.MSG_WARNING, insertionPoint);
					return;
				}
				
				// Clone the insertion point, and insert the clone before the insertion point
				ProtoableNode clonedInsertionPoint = (ProtoableNode)insertionPoint.clone(null, true);
				insertionPointParent.insertChild(clonedInsertionPoint, insertionPoint, false);
				
				// Remove the extras that defined the insertion point from the clone
				Element extras = clonedInsertionPoint.getElement(XFA.EXTRASTAG, 0);
				for (Node child = extras.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
					if (child instanceof TextValue) {
						TextValue clonedInsertionPointTextValue = (TextValue)child;
						
						if (clonedInsertionPointTextValue.getName().equals(INSERTION_POINT_NAME)) {
							clonedInsertionPointTextValue.remove();
							break;
						}
					}
				}
				
				// Remove placeholder content now since it could potentially get merged
				// with the insertion point content, then get removed later when all
				// remaining placeholder content is removed.
				removeInsertionPointPlaceholders(clonedInsertionPoint);
				
				if (extras.getFirstXFAChild() == null)
					extras.remove();
				
				// Set the usehref attribute - this has the side-effect of resolving the proto.
				clonedInsertionPoint.setAttribute(new StringAttr(XFA.USEHREF, insertion.getReference()), XFA.USEHREFTAG);

				// Recursively clear the fragment flag on the resolved fragment so that
				// it can be saved with the template.
				clonedInsertionPoint.isFragment(false, true);
				
				// XFA's fragment references aren't URIs, so purge any #XML_ID
				// or #som(SOM_expr) from the reference.
				String uri = insertion.getReference();
				int sharp = uri.indexOf('#');
				if (sharp >= 0)
				    uri = uri.substring(0, sharp);
				// assemble the path to this fragment
				URI ref = URI.create(uri).normalize();
				String path = ref.getPath();
				int index = path.lastIndexOf('/');
				path = index == -1 ? "" : path.substring(0, index + 1);
				
				basePath = ref.isAbsolute() ? 
						ref.toString() : 
						URI.create(basePath + path).normalize().toString();
				
				// Watson 2414396
				// If the reference to the fragment just loaded is relative, then we may
				// need to adjust any relative URLs in the fragment.
				if (basePath != null && !StringUtils.isEmpty(basePath))
					rebaseRelativeReferencesInFragment(clonedInsertionPoint, basePath);
				
				// Recursively resolve any insertions in the resolved fragment
				insertionStack.add(insertion.getKey());
				
				spliceInternal(clonedInsertionPoint, insertions, removeInsertionPoints, usedKeys, insertionStack, basePath);
				insertionStack.remove(insertionStack.size() - 1);
				
				usedKeys.add(insertion.getKey());
			}
		}
		
		// Remove insertion points, if requested
		if (removeInsertionPoints) {
			for (TextValue textValue : insertionPoints) {
				textValue.getXFAParent().getXFAParent().remove();
			}
		}
		
		// Remove all placeholders.
		// Placeholder content in insertion points that were inserted will already
		// have been removed, but any insertion points that haven't have content
		// inserted into them still need to be removed.
		for (TextValue textValue : insertionPointPlaceholders)
			textValue.getXFAParent().getXFAParent().remove();
	}
	
	/**
	 * Determines whether a given element is contained within the subtree(s)
	 * defined by a SOM expression. The element is considered to be contained
	 * within a subtree if it is the same node as the resolved SOM expression,
	 * if the resolved SOM expression is one of its ancestors.
	 * @param element the element to be tested
	 * @param containedBySom the SOM expression that defines the subtrees that
	 * are tested for containing element.
	 * @return <code>true</code> if containedBySom identifies subtree(s) that
	 * contain element.
	 */
	private boolean isElementContainedWithinSom(Element element, String containedBySom) {
		
		NodeList nodes = resolveNodes(containedBySom, true, false, false);
		
		for (int i = 0; i < nodes.length(); i++) {
			Node contextNode = (Node)nodes.item(i);
			
			Element parent = element;
			while (parent != null) {
				if (parent == contextNode)
					return true;
				
				parent = parent.getXFAParent();
			}
		}
		
		return false;
	}
	
	/**
	 * Searches recursively for any extras children containing TextValues that define
	 * insertion points or insertion point temporary placeholder content.
	 * 
	 * @param element the <code>Element</code> to be searched.
	 * @param insertionPoints a list that any insertion points found are added to
	 * @param insertionPointPlaceholders a list that any insertion point placeholders found are added to
	 */
	private void findInsertionPoints(Element element, List<TextValue> insertionPoints, List<TextValue> insertionPointPlaceholders) {
		
		// Look for an extras child of element that define splice points.
		// The parent must be a ProtoableNode since that is the mechanism used
		// to splice in insertions (i.e., via proto resolution).
		if (element.getClassTag() == XFA.EXTRASTAG && 
			element.getXFAParent() instanceof ProtoableNode) {
			
			for (Node child = element.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
				if (child instanceof TextValue) {
					TextValue textValue = (TextValue)child;
					
					if (textValue.getName().equals(INSERTION_POINT_NAME))
						insertionPoints.add(textValue);
					else if (textValue.getName().equals(INSERTION_POINT_PLACEHOLDER_NAME))
						insertionPointPlaceholders.add(textValue);
				}
			}
		}
		
		for (Node child = element.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child instanceof Element) {
				findInsertionPoints((Element)child, insertionPoints, insertionPointPlaceholders);
			}
		}
	}
	
	/**
	 * Searches recursively for any extras containing TextValues that define
	 * insertion point placeholders, and removes the ProtoableNode that contains it.
	 * @param node the node to remove insertion point placeholders from.
	 */
	private void removeInsertionPointPlaceholders(Node node) {
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child instanceof TextValue) {
				TextValue clonedInsertionPointTextValue = (TextValue)child;
				
				if (clonedInsertionPointTextValue.getName().equals(INSERTION_POINT_PLACEHOLDER_NAME)) {
					clonedInsertionPointTextValue.getXFAParent().getXFAParent().remove();
				}
			}
			
			removeInsertionPointPlaceholders(child);
		}
	}
	
	/**
	 * Recursively, rebase relative references (other than usehref) to be relative to 
	 * a given base path.
	 * @param node the Element to be rebased, including its children.
	 * @param base the relative base path, which is non-empty and terminates in a '/'.
	 */
	private static void rebaseRelativeReferencesInFragment(Element node, String base) {
		
		// The URI references that need to be rebased (that we know about) are:
		// 	<image href="...">
		// 	<exData href="...">
		// 	<exObject codeBase="...">
		
		// TODO: What about <a href="..."> in rich text content?
		
		assert base.length() > 1 && base.endsWith("/");
		
		String schemaAttrName = null;
		
		if (node.getClassTag() == XFA.IMAGETAG || node.getClassTag() == XFA.EXDATATAG)
			schemaAttrName = XFA.HREF;
		else if (node.getClassTag() == XFA.EXOBJECTTAG)
			schemaAttrName = XFA.CODEBASE;
		
		if (schemaAttrName != null) {
			int index = node.findSchemaAttr(schemaAttrName); 
		
			if (index != -1) {				
				try {
					Attribute attr = node.getAttr(index); 
					URI uri = new URI(attr.getAttrValue());
					//Don't rebase empty uri
					if (!uri.toString().equals("") && !uri.isAbsolute()) {
						String rebasedURI = URI.create(base + attr.getAttrValue()).normalize().toString();
						node.setAttribute(attr.getNS(), attr.getQName(), attr.getLocalName(), rebasedURI, false);
					}
				} 
				catch (URISyntaxException e) {
					// quietly ignore any invalid URIs we find
				}
			}
		}
		
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			if (child instanceof Element)
				rebaseRelativeReferencesInFragment((Element)child, base);
		}
	}
	
	/**
	 * Synchronize this template model with the peer x:xmpmeta packet.
	 * MetaData information is copied from the older to the newer (by
	 * checking the timestamps in each).
	 * @param sCreatorTool - The name of the tool that created the form (may be empty). 
	 * @param sProducer - The producer that created the form (may be empty).
	 * @throws ExFull if synchronizing the metadata fails. 
	 *
	 * @exclude from published api.
	 */
	public void synchronizeXMPMetaData(String sCreatorTool, String sProducer) {
		
		AppModel appModel = getAppModel();
		if (appModel == null)
			return;
		
		try {

			final XMPHelper oXMP = new XMPHelper(appModel, XMPHelper.getXMPPacket(appModel), "", true);
			oXMP.setCreatorTool(sCreatorTool, false);
			oXMP.setProducer(sProducer, false);
	
			// Sync template and x:xmpmeta, passing TRUE to allow the x:xmpmeta packet
			// to be updated from the template if the template is newer.
			oXMP.synchronize(true);
		}
		catch (Exception ex) {	// XMPException - but avoid referencing xmpcore.jar
			throw new ExFull(ex);
		}
	}
	
	/**
	 * @exclude from published api.
	 */
	protected void updateOriginalXFAVersionPI() {
		
		if (!ACROBAT_PLUGIN){
			String sPIValue = getNS(mnOriginalVersion);
			
			int nVersion = getVersion(sPIValue);
			LegacyMask nDefaults = getXFALegacyDefaultsForXFAVersion(nVersion);
			
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_POSITIONING, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_EVENTMODEL, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_PLUSPRINT, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_PERMISSIONS, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_CALCOVERRIDE, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_RENDERING, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V26_HIDDENPOSITIONED, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V27_MULTIRECORDCONTEXTCACHE, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V27_LAYOUT, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V27_SCRIPTING, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V27_EVENTMODEL, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V27_TRAVERSALORDER, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V27_XHTMLVERSIONPROCESSING, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V27_ACCESSIBILITY, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V28_LAYOUT, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V28_SCRIPTING, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V29_LAYOUT, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V29_SCRIPTING, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V30_LAYOUT, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V30_SCRIPTING, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V29_TRAVERSALORDER, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_PATCH_B_02518, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V29_FIELDPRESENCE, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_PATCH_W_2393121, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_PATCH_W_2447677, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_PATCH_W_2757988, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V32_SCRIPTING, nDefaults);
			sPIValue = updateLegacyString(sPIValue, mnLegacyInfo, AppModel.XFA_LEGACY_V32_CANONICALIZATION, nDefaults);
	
			ProcessingInstruction oPI = getOriginalVersionNode(false, "");
	
			// update the pi
			if (oPI == null)
				getOriginalVersionNode(true, sPIValue);	// create
			else
				oPI.setData(sPIValue);
		}
	}

	/**
	 * check if a node is a global field
	 * 
	 * @exclude from public api.
	 */
	public boolean validateGlobalField(Element poNodeImpl) {

		int eMatch = EnumAttr.MATCH_ONCE;

		if (poNodeImpl.isSameClass(XFA.FIELDTAG)) {
			Element poBind = poNodeImpl.getElement(XFA.BINDTAG, true, 0, false,
					false);
			if (poBind != null) {
				EnumValue eValue = (EnumValue) poBind.getAttribute(XFA.MATCHTAG);

				eMatch = eValue.getInt();
			}
		}

		String aName = poNodeImpl.getName();
		for (int i = 0; i < moGlobalFields.size(); i++) {
			if (moGlobalFields.get(i) == aName) {
				if (eMatch == EnumAttr.MATCH_GLOBAL)
					return true;
				else
					return false;
			}
		}
		return true;
	}

	/**
	 * we needed to add the form methods into the template (see below in 
	 * the scripting section) because Designer (for example) uses intellisence
	 * and since it doesn't have a form DOM it can show the $form properties.
	 * By adding the methods here they can actually populate the list of properties 
	 * for $form.
	 *
	 * @exclude from published api.
	 */
	public void invalidMethod(String sName) {
		// I'm not sure if you should show an error message or do nothing.
		// to be consistent I think we should do nothing.

		// the template objects that don't implement certain form methods and/or properties 
		// don't output an error message (e.i field.resetData). The only place where we 
		// actually trow errors is in xfahostpseudomodel.cpp

		// the code below can be uncommented if we decide to show error messages.

		//	String gsTemplate = "$template";
		//	MsgFormatPos oMessage = new MsgFormatPos(ResId.InvalidMethodException);
		//	oMessage.format(gsTemplate).format(sName);
		//	throw new ExFull(oMessage);
	}

	/**
	 * Make any adjustments necessary for the specified version.  If nVersion is 0, then the
	 * current version (i.e. getCurrentVersion()) is used.
	 *
	 * @exclude from published api.
	 */
	public void	adjustForVersion(int nVersion) {
		if (nVersion == 0)
			nVersion = getCurrentVersion();

		// Currently no adjustments are required when adjusting to a version >= XFAVERSION_26.
		// We leave IDs as IDs even though SOM expressions are supported in XFAVERSION_26 and above.
		// This may change if we decide to covert IDs to SOM expressions for XFAVERSION_26 and above.
		if (nVersion >= Schema.XFAVERSION_26)
			return;
		adjustForVersion(this, nVersion);
	}

	// Currently the sole purpose is to convert any SOM-based expressions
	// into ID-based expressions so that older processors can handle them.
	private void adjustForVersion(Element element, int nVersion)
	{
		if (element.isSameClass(XFA.BREAKTAG)) {
			Element pageSet = findPageSet(element);
			Element subform = element.getXFAParent();
			assert(subform == null || subform instanceof Subform);
			adjustAttributeForVersion(element, XFA.AFTERTARGETTAG, pageSet, nVersion);
			adjustAttributeForVersion(element, XFA.BEFORETARGETTAG, pageSet, nVersion);
			adjustAttributeForVersion(element, XFA.OVERFLOWTARGETTAG, subform, nVersion);
			adjustAttributeForVersion(element, XFA.OVERFLOWLEADERTAG, subform, nVersion);
			adjustAttributeForVersion(element, XFA.OVERFLOWTRAILERTAG, subform, nVersion);
			adjustAttributeForVersion(element, XFA.BOOKENDLEADERTAG, subform, nVersion);
			adjustAttributeForVersion(element, XFA.BOOKENDTRAILERTAG, subform, nVersion);
		}
		else if (element.isSameClass(XFA.BREAKAFTERTAG) ||
				 element.isSameClass(XFA.BREAKBEFORETAG) ||
				 element.isSameClass(XFA.OVERFLOWTAG)) {
			Element contextNode = null;	// pageSet or Subform
			if (element.isSameClass(XFA.OVERFLOWTAG)) {
				Element subform = element.getXFAParent();
				assert(subform == null || subform instanceof Subform);
				contextNode = subform;
			}
			else
				contextNode = findPageSet(element);
			adjustAttributeForVersion(element, XFA.TARGETTAG, contextNode, nVersion);
			adjustAttributeForVersion(element, XFA.LEADERTAG, contextNode, nVersion);
			adjustAttributeForVersion(element, XFA.TRAILERTAG, contextNode, nVersion);
		}
		else if (element.isSameClass(XFA.BOOKENDTAG)) {
			Element subform = element.getXFAParent();
			assert(subform == null || subform instanceof Subform);
			adjustAttributeForVersion(element, XFA.LEADERTAG, subform, nVersion);
			adjustAttributeForVersion(element, XFA.TRAILERTAG, subform, nVersion);
		}

		// Recursively apply to children.
		Node child = element.getFirstXFAChild();
		while (child != null) {
			if (child instanceof Element)
				adjustForVersion((Element)child, nVersion);
			child = child.getNextXFASibling();
		}
	}

	private void adjustAttributeForVersion(Element element, int eTag, 
										   Element contextNode, int version) {
		Attribute oAttr = null;
		if (element.isPropertySpecified(eTag, false, 0)) // false = don't check protos
			oAttr = element.getAttribute(eTag);
		if (oAttr != null) {
			String sAttrValue = oAttr.getAttrValue();
			if (!sAttrValue.startsWith("#")) {
				String sSOM = sAttrValue;
				if ((sSOM.startsWith("som(")) && sSOM.endsWith(")"))
					sSOM = sSOM.substring(4, sSOM.length() - 1);
				Node oTargetNode = null;
				if (contextNode != null)
					oTargetNode = contextNode.resolveNode(sSOM);
				if (oTargetNode instanceof Element) {
					//Attribute attr1 = element.getAttribute(eTag);
					String sID = ((Element)oTargetNode).establishID();
					element.setAttribute(new StringAttr(XFA.VALUE, "#" + sID), eTag);					
				}
				else {
					// If the SOM expression isn't valid, do nothing.  It won't be resolved
					// on the client, but that's a form design problem.  Log a warning here.
					// "The SOM expression '%0' referenced on the node '%1' cannot be resolved.  
					// The form may not render correctly on older clients."
					MsgFormatPos oMessage = new MsgFormatPos(ResId.CannotConvertSOMToID);
					oMessage.format(sSOM);
					oMessage.format(element.getSOMExpression());

					Model model = element.getModel();
					if (model != null)
						model.addErrorList(new ExFull(oMessage), LogMessage.MSG_WARNING, this);
				}
			}
		}
	}

	// Locate a pageset, given a node.  May return a null node.
	private Element findPageSet(Element element) {
		// Determine the active pageSet by scanning up the parent lineage looking
		// for a subform that contains a pageSet.
		Element parent = element;
		while (parent != null) {
			if (parent instanceof Subform) {
				// If this ancestor subform contains a pageset, that's what we
				// want to return;
				Node child = parent.getFirstXFAChild();
				while (child != null) {
					if (child instanceof PageSet)
						return (Element)child;	// Found the pageset.
					child = child.getNextXFASibling();
				}
			}
			parent = parent.getXFAParent();
		}
		
		return null;
	}
	
	/**
	 * Publish the model to an Application Storage facility. This involves
	 * updating all external references (such as image hrefs) such that they
	 * point to local collateral. The actual details of this are left up to the
	 * implementer of the class derived from Model.Publisher specified in the
	 * oPublisher parameter.
	 * 
	 * What publish() itself does is recursively traverse the tree structure of
	 * the model. When an external reference is encountered (for example, the
	 * href attribute of an image node), the updateExternalRef() method is
	 * called. The node, attribute identifier and original value of the external
	 * reference are passed to updateExternalRef(). The derived class should
	 * copy the collateral to the application storage area (if desired), and
	 * return the updated value of sExternalRefValue to indicate the new value
	 * for the external reference (probably some sort of relative path). If an
	 * external reference stored as an element value (for example, a uri node)
	 * is encountered, updateExternalRef is called with node and the eAttribute
	 * arg is set to XFA.TEXTNODETAG.
	 * 
	 * Two or more instances of an identical external reference result in only
	 * one call to updateExternalRef(), because returned values are cached and
	 * reused.
	 *
	 * If called on the AppModel, this method recursively calls publish
	 * on all contained models.
	 *
	 * @param publisher
	 *            an instance of a class derived from TemplateModel.Publisher.
	 * 
	 * @see com.adobe.xfa.Model#publish(Publisher oPublisher)
	 * @exclude from published api.
	 */
	public boolean publish(Model.Publisher publisher) {
		Map<String, String> hrefCache = new HashMap<String, String>();
		publishNode(publisher, this, hrefCache);
		return super.publish(publisher);
	}

	private void publishNode(Model.Publisher publisher, Node node, Map<String, String> hrefCache) {
		if (node instanceof ProtoableNode) {
			ProtoableNode protoableNode = (ProtoableNode)node;
			if (protoableNode.isPropertySpecified(XFA.USEHREFTAG, true, 0)) {
				String sURL = protoableNode.getAttribute(XFA.USEHREFTAG).getAttrValue();
				int nHash = sURL.indexOf('#');
				String sFragIdent = "";
				if (nHash != -1) {
					sFragIdent = sURL.substring(nHash);
					sURL = sURL.substring(0, nHash);
				}
				String sNewURL = hrefCache.get(sURL);
				if (StringUtils.isEmpty(sNewURL)) {
					// call publisher's virtual method updateExternalRef() to get a new value
					sNewURL = publisher.updateExternalRef(node, XFA.USEHREFTAG, sURL);
					hrefCache.put(sURL, sNewURL);
				}
				if (!sURL.equals(sNewURL)) {
					sNewURL += sFragIdent;
					protoableNode.setAttribute(new StringAttr(XFA.USEHREF, sNewURL), XFA.USEHREFTAG);
				}
			}

			if (protoableNode.isPropertySpecified(XFA.HREFTAG, true, 0)) {
				// not inline; get hRef value
				String sHref = protoableNode.getAttribute(XFA.HREFTAG).getAttrValue();
				String sNewHRef = hrefCache.get(sHref);
				if (StringUtils.isEmpty(sNewHRef)) {
					// call publisher's virtual method updateExternalRef() to get a new value
					sNewHRef = publisher.updateExternalRef(node, XFA.HREFTAG, sHref);
					hrefCache.put(sHref, sNewHRef);
				}
				if (!sHref.equals(sNewHRef)) 
					protoableNode.setAttribute(new StringAttr(XFA.HREF, sNewHRef), XFA.HREFTAG);
			}
			
		}
		
		/*
		if (node instanceof ImageValue) {
			ImageValue imageValue = (ImageValue)node;
			if (imageValue.isPropertySpecified(XFA.HREFTAG, true, 0)) {
				// not inline; get hRef value
				String sHref = imageValue.getAttribute(XFA.HREFTAG).getAttrValue();
				String sNewHRef = hrefCache.get(sHref);
				if (StringUtils.isEmpty(sNewHRef)) {
					// call publisher's virtual method updateExternalRef() to get a new value
					sNewHRef = publisher.updateExternalRef(node, XFA.HREFTAG, sHref);
					hrefCache.put(sHref, sNewHRef);
				}
				if (!sHref.equals(sNewHRef)) 
					imageValue.setAttribute(new StringAttr(XFA.HREF, sNewHRef), XFA.HREFTAG);
			}
		}
		*/

		if (node instanceof RichTextNode) {
			RichTextNode richTextNode = (RichTextNode)node;
			List<Element> anchors = new ArrayList<Element>();
			getElementsByTagNamesNS(this, richTextNode.getNS(), "a", anchors);
			int nAnchors = anchors.size();
			for (int j = 0; j < nAnchors; j++) {
				Element oAnchor = anchors.get(j);
				Attribute href = oAnchor.getAttribute(XFA.HREFTAG);
				if (href != null) {
					String sHref = href.getAttrValue();
					String sQuery = "";
					int nFoundAt = sHref.indexOf('?');
					
					if (nFoundAt == -1)
						nFoundAt = sHref.indexOf('#');
					
					if (nFoundAt != -1) {
						sQuery = sHref.substring(nFoundAt);
						sHref = sHref.substring(0, nFoundAt);
					}
					
					String sNewHRef = hrefCache.get(sHref);
					
					if (StringUtils.isEmpty(sNewHRef)) {
						// call oPublisher's virtual method updateExternalRef() to get a new value
						sNewHRef = publisher.updateExternalRef(node, XFA.HREFTAG, sHref);
						hrefCache.put(sHref, sNewHRef);
					}
					
					if (!sHref.equals(sNewHRef)) {
						sNewHRef += sQuery;
						richTextNode.setAttribute(new StringAttr(XFA.HREF, sNewHRef), XFA.HREFTAG);
					}
				}
			}
		}
		
		for (Node child = node.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
			publishNode(publisher, child, hrefCache);
		}
	}

	private void getElementsByTagNamesNS(Node node, String ns, String name, List<Element> anchors) {
		for (Node child = node.getFirstXMLChild(); child != null; child = child.getNextXMLSibling()) {
			if (child instanceof Element) {
				Element element = (Element)child;
				if (element.getNS().equals(ns) && element.getLocalName().equals(name))
					anchors.add(element);
			}
			
			getElementsByTagNamesNS(child, ns, name, anchors);
		}
	}

	/**
	 * @exclude from published api.
	 */
	public Node preLoadNode(
			Element parent,
			Node node, 
			Generator genTag) {
		
		Node nextSibling = node.getNextXMLSibling();
		
		if (node instanceof TextNode) {
			TextNode textNode = (TextNode)node;
			ChildReln childReln = parent.getChildReln(XFA.TEXTNODETAG);
			
			if (childReln == null || parent instanceof ExDataValue) {
				// Remove whitespace nodes
				// In XFA4J, the parser doesn't generate consecutive #text nodes, so
				// we don't need to scan forward.
				
				if (textNode.isXMLSpace()) {				
					nextSibling = textNode.getNextXMLSibling();
					textNode.remove();
				}
			}
			else if (childReln.getMax() == 1) {
				
				// Consolidate any following #text nodes into textNode
				Node nextChild;
				for (Node child = nextSibling; child != null; child = nextChild) {
					nextChild = child.getNextXMLSibling();
					
					if (child instanceof TextNode) {
						textNode.setText(textNode.getText() + ((TextNode)child).getText());
						child.remove();
					}
				}
				
				nextSibling = textNode.getNextXMLSibling();
			}
		}
		
		// Javaport: In C++ we know if there is an OriginalXFAVerson PI present 
		// upfront when creating the template model. In Java we don't know until 
		// the very end when the whole template has already been loaded.
		// Model.preLoadNode() is where we detect the OriginalXFAVersion PI and it
		// will update the original version node but will not change the version
		// information or the legacy flags kept by the templateModel.
		// If original version changes, we need to force an update, and if anything
		// should have been loaded differently because of version/legacy flags,  
		// it needs to be corrected in the fixup stage.
		String original = getOriginalVersion(false);
		super.preLoadNode(parent, node, genTag);
		if (!getOriginalVersion(false).equals(original))
			mnOriginalVersion = 0;
		
		return nextSibling;
	}

}
