package com.adobe.xfa.text;

//import com.adobe.xfa.font.FontInstance;
import com.adobe.xfa.ut.Storage;
//import com.adobe.xfa.ut.UnitSpan;

/**
 * @exclude from published api.
 */

class LineDesc {
	float moMarginL;
	float moMarginR;
	float moSpecial;
	int meJustifyH;
	int mnTrailingSpaces;
	int mnInternalSpaces;

	LineDesc () {
	}

	LineDesc (LineDesc source) {
		moMarginL = source.moMarginL;
		moMarginR = source.moMarginR;
		moSpecial = source.moSpecial;
		meJustifyH = source.meJustifyH;
		mnTrailingSpaces = source.mnTrailingSpaces;
		mnInternalSpaces = source.mnInternalSpaces;
	}
}

//----------------------------------------------------------------------
//
//		class DispLine
//
//----------------------------------------------------------------------
class DispLine {
	static class PosnInfo {
		int index;			// general-purpose character index (stream, line)
		int auxInfo;		// general-purpose additional information
		int posnIndex = -1;	// index into position map (not always used)
	}

	static class GlyphMaker {
		private final DispLine mLine;
		private float mWidth = 0;
		private float mTrailingWidth = 0;
		private float mCharSpacing;
		private float mWordSpacing;
		private DispRun mCurrentRun;

		GlyphMaker (DispLine line) {
			mLine = line;
		}

		void addGlyph (AFEElement afeElement, int charIndex) {
			int glyphIndex = mLine.getGlyphCount();

			if ((mCurrentRun == null)
			 || (charIndex < mCurrentRun.getMapIndex())
			 || (charIndex >= (mCurrentRun.getMapIndex() + mCurrentRun.getMapLength()))) {
				mCurrentRun = mLine.getMappedRun (charIndex);
				TextAttr attr = mCurrentRun.getAttr();
				if (attr.charSpacingEnable()) {
					mCharSpacing = Units.toFloat (attr.charSpacing().getLength());
				} else {
					mCharSpacing = 0;
				}
				if (attr.wordSpacingEnable()) {
					mWordSpacing = Units.toFloat (attr.wordSpacing().getLength());
				} else {
					mWordSpacing = 0;
				}
			}
			float charWidth = afeElement.getScaledXAdvance() + mCharSpacing;
			if (mLine.getBreakClass (charIndex) == TextCharProp.BREAK_SP) {
				charWidth += mWordSpacing;
			}

			Glyph glyph = new Glyph();
			glyph.setGlyph (afeElement.elementAt());

			glyph.setOriginalX (mWidth);
			mWidth += charWidth;
			glyph.setOriginalNextX (mWidth);

			glyph.setRenderByGlyphID (afeElement.renderByGlyphID());
			glyph.setRTL (afeElement.isRTL());

			mLine.add (glyph);

			float offsetX = afeElement.getScaledXPlacement();
			float offsetY = afeElement.getScaledYPlacement();
			if ((offsetX != 0) || (offsetY != 0)) {
				GlyphExtra glyphExtra = mLine.forceExtra (glyphIndex);
				glyphExtra.setOffsetX (offsetX);
				glyphExtra.setY (offsetY);
				mLine.getGlyph(glyphIndex).setShifted (true);
			}

			if ((! mLine.isVisualChar (charIndex))
			 && (mLine.getBreakClass (charIndex) == TextCharProp.BREAK_SP)) {
				mTrailingWidth += charWidth; 
			}
		}

		void applyWidths () {
			mLine.setWidth (mWidth);
			mLine.setTrailingWidth (mTrailingWidth);
		}
	}

	static final int CARET_INVALID = 0;
	static final int CARET_CONDITIONAL = 1;
	static final int CARET_PRESENT = 2;

	static final int INTERNAL_LINE = 0;
	static final int TAB_TO_END = 1;
	static final int HARD_NEW_LINE = 2;
	static final int REAL_LAST_LINE = 3;

	static final int LINE_BREAK_NORMAL = 0;
	static final int LINE_BREAK_HYPHEN_SUPPRESS = 1;
	static final int LINE_BREAK_HYPHEN = 2;
	static final int LINE_BREAK_FORCED = 3;

	static final int POSN_TYPE_INVALID = 0;
	static final int POSN_TYPE_RUN = 1;
	static final int POSN_TYPE_LIGATURE = 2;
	static final int POSN_TYPE_MULTIPLE = 3;

	private TextDisplay mpoDisplay;
	private TextFrame mpoFrame;
	private DispMapSet mpoMaps;

	private float moWidth;
	private float moTrailingWidth;
	private int mnVisualCharCount;

	private Storage<GlyphExtra> moGlyphExtra;
	private int[] moGlyphLocOrder;

	private boolean mbVerticalOrientation;
	private boolean mbHasBIDI;
	private boolean mbOptycaJustify;	// TODO: no Optyca in Java
	private boolean mbOptycaMapping;
	private boolean mbIsFirstLineInStream;
	private boolean mbIsLastLineInStream;
	private boolean mbIsFirstParaLine;
	private boolean mbHasDecoration;
	private boolean mbHasSingleColour;
	private boolean mbHasAXTEMappings;
	private boolean mbIsLayoutLine;
	private boolean mbIsRTL;
	private boolean mbEndInMiddle;

	private int meLastParaLine;
	private int meStartBreak;
	private int meEndBreak;

	//private static final UnitSpan gLegacyWidthRound = new UnitSpan (UnitSpan.INCHES_72K, 6);

	DispLine (TextFrame poFrame) {
		mpoMaps = poFrame.getDisposableMaps();
		initialize (poFrame);
	}

	DispLine (DispMapSet poMaps) {		// TODO: obsolete, since cannot be called from DispLineWrapped in Java
		mpoMaps = poMaps;
// TODO: C++ implementation doesn't call Initialize(), so neither does this
	}

	DispLine () {
	}

	void detach () {
		mpoFrame.releaseDisposableMaps (mpoMaps);
		mpoMaps.clear();
	}

	void initialize (TextFrame poFrame) {
		mpoFrame = poFrame;
		mpoDisplay = poFrame.display();
		mbHasBIDI = false;
		meLastParaLine = INTERNAL_LINE;
		meStartBreak = LINE_BREAK_NORMAL;
		meEndBreak = LINE_BREAK_NORMAL;
		mbIsRTL = mpoDisplay.isRTL();
	}

	TextSparseStream stream () {
		return mpoFrame.getStream();
	}

	TextDisplay display () {
		return mpoDisplay;
	}

	TextFrame frame () {
		return mpoFrame;
	}

	void setFrame (TextFrame poFrame) {
		mpoFrame = poFrame;
	}

	void setVerticalOrientation (boolean bVerticalOrientation) {
		mbVerticalOrientation = bVerticalOrientation;
	}

	boolean verticalOrientation () {
		return mbVerticalOrientation;
	}

	boolean hasBIDI () {
		return mbHasBIDI;
	}

	void setHasBIDI (boolean bHasBIDI) {
		mbHasBIDI = bHasBIDI;
	}

	boolean optycaJustify () {
		return mbOptycaJustify;
	}

	void setOptycaJustify (boolean bOptycaJustify) {
		mbOptycaJustify = bOptycaJustify;
	}

	boolean optycaMapping () {
		return mbOptycaMapping;
	}

	void setOptycaMapping (boolean bOptycaMapping) {
		mbOptycaMapping = bOptycaMapping;
	}

// TBD: these need to be renamed for frame
	boolean isFirstLineInStream () {
		return mbIsFirstLineInStream;
	}

	void setFirstLineInStream (boolean bIsFirstLineInStream) {
		mbIsFirstLineInStream = bIsFirstLineInStream;
	}

	boolean isLastLineInStream () {
		return mbIsLastLineInStream;
	}

	void setLastLineInStream (boolean bIsLastLineInStream) {
		mbIsLastLineInStream = bIsLastLineInStream;
	}

	boolean isFirstParaLine () {
		return mbIsFirstParaLine;
	}

	void setFirstParaLine (boolean bIsFirstParaLine) {
		mbIsFirstParaLine = bIsFirstParaLine;
	}

	boolean isLastParaLine () {
		return getLastParaLine() >= REAL_LAST_LINE;
	}

	int getLastParaLine () {
		return meLastParaLine;
	}

	void setLastParaLine (int eLast) {
		meLastParaLine = eLast;
	}

	int getStartBreak () {
		return meStartBreak;
	}

	void setStartBreak (int eStartBreak) {
		meStartBreak = eStartBreak;
	}

	int getEndBreak () {
		return meEndBreak;
	}

	void setEndBreak (int eEndBreak) {
		meEndBreak = eEndBreak;
	}

	boolean hasDecoration () {
		return mbHasDecoration;
	}

	void setHasDecoration (boolean bHasDecoration) {
		mbHasDecoration = bHasDecoration;
	}

	boolean hasSingleColour () {
		return mbHasSingleColour;
	}

	void setHasSingleColour (boolean bHasSingleColour) {
		mbHasSingleColour = bHasSingleColour;
	}

	boolean hasAXTEMappings () {
		return mbHasAXTEMappings;
	}

	void setHasAXTEMappings (boolean bHasAXTEMappings) {
		mbHasAXTEMappings = bHasAXTEMappings;
	}

	boolean isLayoutLine () {
		return mbIsLayoutLine;
	}

	void setLayoutLine (boolean bIsLayoutLine) {
		mbIsLayoutLine = bIsLayoutLine;
	}

	boolean isRTL () {
		return mbIsRTL;
	}

	void setRTL (boolean bIsRTL) {
		mbIsRTL = bIsRTL;
	}

	boolean isEndInMiddle () {
		return mbEndInMiddle;
	}

	void setEndInMiddle (boolean bIsEndInMiddle) {
		mbEndInMiddle = bIsEndInMiddle;
	}

	boolean allowLastPosition (int eAffinity) {
		return mbIsLastLineInStream || (mbEndInMiddle && (eAffinity == TextPosnBase.AFFINITY_BEFORE));
	}

	int getCombCells () {
		return mpoFrame.getActualCombCells();
	}

	int getLegacyLevel () {
		return mpoFrame.getLegacyLevel();
	}

	boolean legacyPositioning () {
		return mpoFrame.getLegacyLevel() == TextLegacy.LEVEL_V6;
	}

	int getCharCount () {
		return mpoMaps.mnCharCount;
	}

	int[] getCharArray () {
		return mpoMaps.getCharArray();
	}

	int getChar (int nIndex) {
		return mpoMaps.getChar (nIndex);
	}

	int[] getBreakArray () {
		return mpoMaps.getBreakArray();
	}

	int getBreakData (int nIndex) {
		return mpoMaps.getBreakData (nIndex);
	}

	int getBreakClass (int nIndex) {
		return TextCharProp.getBreakClass (mpoMaps.getBreakData (nIndex));
	}

	int getWidthClass (int nIndex) {
		return TextCharProp.getWidthClass (mpoMaps.getBreakData (nIndex));
	}

	int getGraphemeClass (int nIndex) {
		return TextCharProp.getGraphemeClass (mpoMaps.getBreakData (nIndex));
	}

	int getWordClass (int nIndex) {
		return TextCharProp.getWordClass (mpoMaps.getBreakData (nIndex));
	}

	void setChar (int nIndex, int c, int eBreak) {
		mpoMaps.setChar (nIndex, c, eBreak);
		testBIDI (c, eBreak);
	}

	void setBreak (int nIndex, int eBreak) {
		mpoMaps.setBreakData (nIndex, eBreak);
		testBIDI (mpoMaps.getChar (nIndex), eBreak);
	}

	void preAllocChars (int nChars) {
		mpoMaps.preAllocChars (nChars);
	}

	int getLastGlyphChar () {
		if ((getGlyphCount() == 0) || (getCharCount() == 0)) {
			return '\0';
		}
		GlyphLoc oGlyphLoc = getOrderedGlyphLoc (getGlyphCount() - 1);
		int nCharIndex = oGlyphLoc.getMapIndex() + oGlyphLoc.getMapLength() - 1;
		return getChar (nCharIndex);
	}

	int getVisualCharCount () {
		return mnVisualCharCount;
	}

	void setVisualCharCount (int nCount) {
		mnVisualCharCount = nCount;
	}

	boolean isVisualChar (int nCharIndex) {
		return nCharIndex < mnVisualCharCount;
	}

// cannot position after last char in line, except for last line
	boolean isValidPosition (int nCharIndex, int eAffinity) {
		return allowLastPosition (eAffinity) ? (nCharIndex <= getCharCount()) : (nCharIndex < getCharCount());
	}
	boolean isValidPosition (int nCharIndex) {
		return isValidPosition (nCharIndex, TextPosn.AFFINITY_AFTER);
	}

	DispMap getPositionMap () {
		return mpoMaps.moPosnMap;
	}

	int getPositionCount () {
		return mpoMaps.moPosnMap.size();
	}

	DispPosn getPosition (int nIndex) {
		return mpoMaps.getPosition (nIndex);
	}

	DispPosn getMappedPosition (int nCharIndex) {
		int nMapIndex = mpoMaps.moPosnMap.findItem (nCharIndex);
		assert (getPositionMap().isValidMapIndex (nMapIndex));
		return mpoMaps.getPosition (nMapIndex);
	}

	static int getPositionStreamCount (DispPosn oPosition) {
		return (oPosition.getStreamCount() == 0)
			 ? oPosition.getMapLength()
			 : oPosition.getStreamCount();
	}

	static int getCharStreamCount (DispPosn oPosition) {
		int streamCount = oPosition.getStreamCount();
		return (streamCount == 0) ? 1 : streamCount;
	}

	void clearPositionMap () {
		mpoMaps.moPosnMap.empty();
	}

	int charToStreamIndex (int nCharIndex, int nStreamOffset) {
		return charToStreamIndex (getMappedPosition (nCharIndex), nCharIndex, nStreamOffset);
	}

	static int charToStreamIndex (DispPosn oPosition, int nCharIndex, int nStreamOffset) {
		int nCharOffset = nCharIndex - oPosition.getMapIndex();
		if (getPositionType (oPosition) == POSN_TYPE_MULTIPLE) {
			if (nCharOffset < oPosition.getMapLength()) {
				nCharOffset = 0;
			}
		}

		return nCharOffset + oPosition.pp().index() + nStreamOffset;
	}

	void charToStreamInfo (int nCharIndex, PosnInfo info) {
		charToStreamInfo (getMappedPosition (nCharIndex), nCharIndex, info);
	}

	static void charToStreamInfo (DispPosn oPosition, int nCharIndex, PosnInfo info) {
		info.index = charToStreamIndex (oPosition, nCharIndex, 0);
		info.auxInfo = 1;

		if (getPositionType (oPosition) == POSN_TYPE_LIGATURE) {
			info.auxInfo = getPositionStreamCount (oPosition);
		}
	}

	TextPosnBase getStreamPosition (int nCharIndex, int nStreamOffset) {
		DispPosn oBasePosn = getMappedPosition (nCharIndex);
		return new TextPosnBase (oBasePosn.pp().stream(), charToStreamIndex (oBasePosn, nCharIndex, nStreamOffset));
	}

	static int streamToCharIndex (DispPosn oPosition, int nStreamIndex, int nCharOffset) {
		int nCharRelative = nStreamIndex - oPosition.pp().index();
		if (getPositionType (oPosition) == POSN_TYPE_LIGATURE) {
			nCharRelative = 0;
		}

		return nCharRelative + oPosition.getMapIndex() + nCharOffset;
	}

	static void streamToCharIndex (DispPosn oPosition, int nStreamIndex, PosnInfo info) {
		info.index = streamToCharIndex (oPosition, nStreamIndex, 0);
		info.auxInfo = 0;
		if (getPositionType (oPosition) == POSN_TYPE_LIGATURE) {
			info.auxInfo = nStreamIndex - oPosition.pp().index();
		}
	}

	static void streamToCharInfo (DispPosn oPosition, int nStreamIndex, PosnInfo info) {
		info.index = streamToCharIndex (oPosition, nStreamIndex, 0);
		info.auxInfo = 1;
		if (getPositionType (oPosition) == POSN_TYPE_MULTIPLE) {
			info.auxInfo = oPosition.getMapLength();
		}
	}

	static int getPositionType (DispPosn oPosn) {
		if (oPosn.getStreamCount() == 0) {
			return POSN_TYPE_RUN;
		}
		if (oPosn.getMapLength() == 1) {
			return POSN_TYPE_LIGATURE;
		}
		if (oPosn.getStreamCount() == 1) {
			return POSN_TYPE_MULTIPLE;
		}
		return POSN_TYPE_INVALID;
	}

	static boolean isValidPosition (DispPosn oPosn) {
		return getPositionType (oPosn) != POSN_TYPE_INVALID;
	}

	DispMap getRunMap () {
		return mpoMaps.moRunMap;
	}

	int getRunCount () {
		return mpoMaps.moRunMap.size();
	}

	DispRun getRun (int nIndex) {
		return mpoMaps.getRun (nIndex);
	}

	DispRun getMappedRun (int nCharIndex) {
		int nMapIndex = mpoMaps.moRunMap.findItem (nCharIndex);
		assert (getRunMap().isValidMapIndex (nMapIndex));
		return mpoMaps.getRun (nMapIndex);
	}

	TextAttr getMappedAttr (int nCharIndex) {
		return getMappedRun (nCharIndex).getAttr();
	}

	int getGlyphCount () {
		return mpoMaps.moGlyphs.size();
	}

	Glyph getGlyph (int nIndex) {
		return mpoMaps.getGlyph (nIndex);
	}

	Glyph getMappedGlyph (int nCharIndex) {
		GlyphLoc oGlyphLoc = getMappedGlyphLoc (nCharIndex);
		return getGlyph (oGlyphLoc.getGlyphIndex());
	}

	int nextGlyphBaseAccentIndex (int nStart) {
		int nGlyphs = getGlyphCount();
		if (nStart >= nGlyphs) {
			return nGlyphs;
		}

		Glyph poGlyph = getGlyph (nStart);
		int nIndex = nStart;

		if (! poGlyph.isRTL()) {
			nIndex++;
		}

		for (; nIndex < nGlyphs; nIndex++) {
			GlyphLoc oGlyphLoc = getOrderedGlyphLoc (nIndex);
			int nCharIndex = oGlyphLoc.getMapIndex();
			if (getBreakClass (nCharIndex) != TextCharProp.BREAK_CM) {
				if (poGlyph.isRTL()) {
					nIndex++;
				}
				break;
			}
		}

		return nIndex;
	}

	void preAllocGlyphs (int nGlyphs, boolean bAllocGlyphLocs) {
		mpoMaps.preAllocGlyphs (nGlyphs, bAllocGlyphLocs);
	}

	GlyphExtra forceExtra (int nGlyphIndex) {
		Glyph oGlyph = getGlyph (nGlyphIndex);
		GlyphExtra poExtra = oGlyph.getExtra (this);
		if (poExtra == null) {
			if (moGlyphExtra == null) {
				moGlyphExtra = new Storage<GlyphExtra>();
			}
			int nIndex = moGlyphExtra.size();
			moGlyphExtra.setSize (moGlyphExtra.size() + 1);	// TODO: probably not as efficient in Java
			poExtra = new GlyphExtra();						// TODO: need to try preserving these
			moGlyphExtra.set (nIndex, poExtra);
			oGlyph.setExtraIndex (nIndex);
		}
		poExtra.setWidth (oGlyph.getOriginalNextX() - oGlyph.getOriginalX());
		return poExtra;
	}

	GlyphExtra getGlyphExtra (int nGlyphIndex) {
		return moGlyphExtra.get (nGlyphIndex);
	}

	void preAllocExtra (int nSize) {
		if (nSize > 0) {
			if (moGlyphExtra == null) {
				moGlyphExtra = new Storage<GlyphExtra>();
			}
			if (nSize > moGlyphExtra.size()) {
				moGlyphExtra.ensureCapacity (nSize);
			}
		}
	}

	DispMap getGlyphLocMap () {
		return mpoMaps.moGlyphLocMap;
	}

	GlyphLoc getGlyphLoc (int nIndex) {
		if (mpoMaps.moGlyphLocMap.size() == 0) {
			return new GlyphLoc (nIndex, nIndex, 1);	// TODO: is it OK to do all this newing?
		}

		return mpoMaps.getGlyphLoc (nIndex);
	}

	GlyphLoc getMappedGlyphLoc (int nCharIndex) {
		if (mpoMaps.moGlyphLocMap.size() == 0) {
			return getGlyphLoc (nCharIndex);
		}

		int nMapIndex = mpoMaps.moGlyphLocMap.findItem (nCharIndex);
		assert (getGlyphLocMap().isValidMapIndex (nMapIndex));
		return mpoMaps.getGlyphLoc (nMapIndex);
	}

	int getGlyphLocCharIndex (int nGlyphLocIndex) {
		if (mpoMaps.moGlyphLocMap.size() == 0) {
			return nGlyphLocIndex;
		} else {
			return mpoMaps.getGlyphLoc(nGlyphLocIndex).getMapIndex();
		}
	}

	GlyphLoc getOrderedGlyphLoc (int nGlyphIndex) {
		return getGlyphLoc (getGlyphLocOrder (nGlyphIndex));
	}

	int getGlyphLocOrder (int nGlyphIndex) {
		return ((moGlyphLocOrder == null) || (moGlyphLocOrder.length == 0)) ? nGlyphIndex : moGlyphLocOrder[nGlyphIndex];
	}

	void setGlyphLocOrder (int nGlyphIndex, int nGlyphLocIndex) {
		moGlyphLocOrder[nGlyphIndex] = nGlyphLocIndex;
	}

	void allocateGlyphLocOrder (int glyphCount) {
		if ((moGlyphLocOrder == null) || (moGlyphLocOrder.length < glyphCount)) {
			moGlyphLocOrder = new int [glyphCount];
		}
	}

	DispMap getEmbedMap () {
		return mpoMaps.moEmbedMap;
	}

	int getEmbedCount () {
		return mpoMaps.moEmbedMap.size();
	}

	DispEmbed getEmbed (int nIndex) {
		return mpoMaps.getEmbed (nIndex);
	}

	DispEmbed isObject (int nCharIndex) {
		int nEmbedMapIndex = findObject (nCharIndex);
		if (nEmbedMapIndex < 0) {
			return null;
		}

		DispEmbed oEmbed = mpoMaps.getEmbed (nEmbedMapIndex);
		if (oEmbed.getMapIndex() != nCharIndex) {
			return null; // don't allow match at end posn
		}

		return oEmbed;
	}

	DispEmbed isObject (GlyphLoc oGlyphLoc) {
		if (oGlyphLoc.getMapLength() > 1) {
			return null;
		}
		return isObject (oGlyphLoc.getMapIndex());
	}

	int findObject (int nCharIndex) {
		int nEmbedMapIndex = mpoMaps.moEmbedMap.findItem (nCharIndex);
		if (! mpoMaps.moEmbedMap.isValidMapIndex (nEmbedMapIndex)) {
			return -1;
		}
		return nEmbedMapIndex;
	}

	int findObject (GlyphLoc oGlyphLoc) {
		if (oGlyphLoc.getMapLength() > 1) {
			return -1;
		}
		return findObject (oGlyphLoc.getMapIndex());
	}

	DispTab tabAt (int nCharIndex) {
		if (mpoMaps.moEmbedMap.size() == 0) {
			return null;
		}

		DispEmbed dispEmbed = isObject (nCharIndex);
		if (dispEmbed == null) {
			return null;
		}

		TextEmbed embed = dispEmbed.getEmbed();
		if (! (embed instanceof DispTab)) {
			return null;
		}

		return (DispTab) embed;
	}

//	TextDirectionLevel allocateDirectionLevels (int nSize, int nPreserve) {
//		return mpoMaps.moDirectionLevels.allocate (nSize, nPreserve);
//	}

	void addChar (int c, int eBreak) {
		mpoMaps.addChar (c, eBreak);
		mnVisualCharCount = mpoMaps.mnCharCount;
		testBIDI (c, eBreak);
	}

	void add (DispEmbed oAdd, int nStart) {
		mpoMaps.moEmbedMap.add (oAdd, nStart);
	}

	void add (DispPosn oAdd, int nStart, int nLength) {
		mpoMaps.moPosnMap.add (oAdd, nStart, nLength);
		if (getPositionType (getPosition (getPositionCount() - 1)) != POSN_TYPE_RUN) {
			setHasAXTEMappings (true);
		}
	}

	void add (DispRun oAdd, int nStart, int nLength) {
		mpoMaps.moRunMap.add (oAdd, nStart, nLength);

// Performance (see also AddChar() method): Certain run types may disable
// the optyca bypass even if all the characters are acceptable for AXTE
// simple layout.  In particular, right-to-left (RTL) text, indic digits,
// discretionary ligatures, and invisible text with invisible character
// outside the supported ranges.
		if (mbHasBIDI) {
			return;
		}

		if (oAdd.isRTL()) {
			mbHasBIDI = true;
		}

		else {
			TextAttr poAttr = oAdd.getAttr();

			int eDigits = display().getDigits (poAttr);
			if ((eDigits != TextAttr.DIGITS_LOCALE) && (eDigits != TextAttr.DIGITS_ARABIC)) {
				mbHasBIDI = true;							// TODO: does this alter algorithm?
			}

			if (poAttr != null) {
				if ((poAttr.direction() == TextAttr.DIRECTION_RTL)
				 || (poAttr.paraDirection() == TextAttr.DIRECTION_RTL)) {
					mbHasBIDI = true;
				} else if (poAttr.invisibleEnable() && poAttr.invisible()) {
					char c = poAttr.invisChar();
					testBIDI (c, TextCharProp.getCharProperty (c));
				}
			}
		}
	}

	void add (Glyph oAdd) {
		mpoMaps.moGlyphs.add (oAdd);
	}

	void add (GlyphLoc oAdd, int nStart, int nLength) {
		mpoMaps.moGlyphLocMap.add (oAdd, nStart, nLength);
	}

	float getWidth () {
		return moWidth;
	}

	void setWidth (float oWidth) {
		moWidth = oWidth;
	}

	float getTrailingWidth () {
		return moTrailingWidth;
	}

	void setTrailingWidth (float oWidth) {
		moTrailingWidth = oWidth;
	}

	int findPosition (TextPosnBase oPosn, PosnInfo info) {
		int eCaret = CARET_INVALID;
		DispPosn matchPosn = null;
		info.index = 0;
		info.auxInfo = 0;
		info.posnIndex = 0;

// The given position may abut an attribute change.  We need to normalize
// it in order to compare character positions in the line.
		TextPosnBase oFwdPosn = new TextPosnBase (oPosn);
		oFwdPosn.tighten (true);
		TextPosnBase oBkwdPosn = null; // assigned only if needed

// Run through the line's position map, testing each position range.
		for (int i = 0; i < getPositionCount(); i++) {
			DispPosn oTestPosn = getPosition (i);
			int nAfterIndex = oTestPosn.pp().index() + getPositionStreamCount (oTestPosn);

// If the given position is in or after this range, we need to look
// further.
			if ((oFwdPosn.stream() == oTestPosn.pp().stream())
			 && (oFwdPosn.index() >= oTestPosn.pp().index())) {

// If the position is cleanly before the end of this position range,
// we've found it and can stop.
				if (oFwdPosn.index() < nAfterIndex) {
					matchPosn = oTestPosn;
					streamToCharIndex (oTestPosn, oFwdPosn.index(), info);
					eCaret = CARET_PRESENT;
					info.posnIndex = i;
					break;
				}

// Otherwise, the position may be beyond this range, or it may be right
// at the end (which requires special processing for the position at a
// line break).
				else {
					TextPosnBase poEqualTest = null;

// Figure out which of our normalized positions to test.
					if (oFwdPosn.index() == nAfterIndex) {
						poEqualTest = oFwdPosn;
					} else {
						if (oBkwdPosn == null) {
							oBkwdPosn = new TextPosnBase (oPosn);
							oBkwdPosn.tighten (false);
						}
						if (oBkwdPosn.index() == nAfterIndex) {
							poEqualTest = oBkwdPosn;
						}
					}

// If we've found it, do the special line break check and get out.
					if (poEqualTest != null) {
						if ((i + 1 < getPositionCount()) || allowLastPosition (oPosn.affinity())) {
							eCaret = CARET_PRESENT;
						} else {
							eCaret = CARET_CONDITIONAL;
						}
						matchPosn = oTestPosn;
						streamToCharIndex (oTestPosn, poEqualTest.index(), info);
						info.posnIndex = i;
						break;
					}
				}
			}
		}

		if (matchPosn != null) {
			if (! isValidPosition (info.index, oPosn.affinity())) {
				eCaret = CARET_CONDITIONAL;
			}
		}

		return eCaret;
	}

	static boolean isUnicodeFont (TextAttr poAttr) {
//		FontItem poFontItem = oFontInstance.getFontItem();	// TODO:
//
//		if (poFontItem != null) {
//			FontItem.ENCODING eEncoding = poFontItem.getEncoding();
//			switch (eEncoding) {
//				case FontItem.UTF_8:
//				case FontItem.UTF_16:
//				case FontItem.UCS_2:
//					return true;
//			}
//		}

		return false;
	}

	void removeContent (int nStart, int nLength) {
		if (nLength == 0) {
			return;
		}

		int nOldLength = getCharCount();
		int nRestIndex = nStart + nLength;
		assert (nRestIndex <= nOldLength);
		int nNewLength = nOldLength - nLength;

// RemoveLineData() is a templated function that removes nLength items at
// position nStart from an array of nOldLength items, sliding the
// remaining items back over the removed ones.	Note that PreAllocChars()
// must be called for the smaller number of characters to get the
// character property array repositioned properly.
//		RemoveLineData (mpoMaps.mpcCharData, nStart, nLength, nOldLength);	// TODO:
//		RemoveLineData (mpoMaps.mpeBreakData, nStart, nLength, nOldLength);
		mpoMaps.preAllocChars (nNewLength, true);

// The following code removes the glyph objects, which may not correspond
// 1:1 with the characters being removed if Optyca was used.  In such a
// case, we need to find contiguous sequences of glyphs and remove them.
		float oWidthRemoved = 0.0f;
		if (! mbHasBIDI) {
			oWidthRemoved = removeGlyphs (nStart, nLength);
		} else {
			int nCharRangeStart = 0;
			int nCharRangeLimit = 0;
			int nGlyphStart = 0;
			int nGlyphCount = 0;
			for (int nGlyph = 0; nGlyph < getGlyphCount(); nGlyph++) {
				GlyphLoc oGlyphLoc = getOrderedGlyphLoc (nGlyph);
				//Glyph oGlyph = getGlyph (nGlyph);
				int nCharIndex = oGlyphLoc.getMapIndex();
				if ((nCharIndex >= nStart) && (nCharIndex < nRestIndex)) {
					int nCharLimit = nCharIndex + oGlyphLoc.getMapLength();
					assert (nCharLimit <= nRestIndex);
					if (nGlyphCount == 0) {
						nCharRangeStart = nCharIndex;
						nCharRangeLimit = nCharLimit;
						nGlyphStart = nGlyph;
						nGlyphCount = 1;
					} else if (nCharIndex == nCharRangeLimit) {
						nCharRangeLimit = nCharLimit;
						nGlyphCount++;
					} else if (nCharLimit == nCharRangeStart) {
						nCharRangeStart = nCharIndex;
						nGlyphCount++;
					} else {
						oWidthRemoved += removeGlyphs (nGlyphStart, nGlyphCount);
						nCharRangeLimit = nCharRangeStart;
						nGlyphCount = 0;
					}
				}
			}
			oWidthRemoved += removeGlyphs (nGlyphStart, nGlyphCount);
		}
		moWidth -= oWidthRemoved;

// Remove the range of characters from the four character-based maps.
		mpoMaps.moEmbedMap.removeRange (nStart, nLength);
		mpoMaps.moPosnMap.removeRange (nStart, nLength /*, DispPosn.split */);
		mpoMaps.moRunMap.removeRange (nStart, nLength);
		if (mpoMaps.moGlyphLocMap.size() > 0) {
			mpoMaps.moGlyphLocMap.removeRange (nStart, nLength);
			for (int i = nStart; i < mpoMaps.moGlyphLocMap.size(); i++) {
				getGlyphLoc(i).setGlyphIndex (i);
			}
		}

//		if ((mpoMaps.moCharWidths.size() >= nOldLength) && mbBypassOptyca) {
// If character widths were cached, remove the corresponding widths.
//			RemoveLineData (mpoMaps.moCharWidths.getBlock(), nStart, nLength, nOldLength);	// TODO:
//			mpoMaps.moCharWidths.setSize (nNewLength);
//		}

// If direction levels were cached, remove the corresponding levels.
//		TextDirectionLevelArray poLevels = mpoMaps.moDirectionLevels;						// TODO:
//		if (poLevels.getAlloc() >= nOldLength) {
//			RemoveLineData (poLevels.getData(), nStart, nLength, nOldLength);
//		}

		mnVisualCharCount -= nLength;
	}

	void insertContent (DispLine poSource, int nIndex) {
		assert (nIndex <= getCharCount());
		int nInsert = poSource.getCharCount();
		if (nInsert == 0) {
			return;
		}
		int i;
		int nOldLength = getCharCount();
		int nNewLength = nOldLength + nInsert;

// InsertLineData() is a templated function that inserts items from one
// array into another, taking care to first open up space by shifting
// existing items in the destination array.
		mpoMaps.preAllocChars (nNewLength, true);
//		InsertLineData (mpoMaps.mpcCharData, nOldLength, nIndex, poSource.mpoMaps.mpcCharData, nInsert);	// TODO:
//		InsertLineData (mpoMaps.mpeBreakData, nOldLength, nIndex, poSource.mpoMaps.mpeBreakData, nInsert);

// Update the three easy character-based maps.
		mpoMaps.moEmbedMap.insertMap (poSource.mpoMaps.moEmbedMap, nIndex, nInsert);
		mpoMaps.moPosnMap.insertMap (poSource.mpoMaps.moPosnMap, nIndex, nInsert);
		mpoMaps.moRunMap.insertMap (poSource.mpoMaps.moRunMap, nIndex, nInsert);

// Figure out where the insert point is in the glyph array as well as
// X-coordinate shifts for the inserted glyphs and the ones after the
// insertion.  Note that glyph array sizes and indexes may not correspond
// to character sizes and indexes.
		GlyphLoc oGlyphLoc = getMappedGlyphLoc(nIndex);
		int nGlyphIndex = oGlyphLoc.getGlyphIndex();
		int nGlyphRest = nGlyphIndex + poSource.getGlyphCount();
		//float oShiftRest = poSource.getWidth();
		float oShiftNew = (nGlyphIndex >= getGlyphCount()) ? getWidth() : getGlyph(nGlyphIndex).getOriginalX();

// Actually insert the glyph objects.
		int nOldGlyphs = getGlyphCount();
		int nInsertGlyphs = poSource.getGlyphCount();
		mpoMaps.moGlyphs.setSize (nOldGlyphs + nInsertGlyphs);
//		InsertLineData (mpoMaps.moGlyphs.getBlock(), nOldGlyphs, nGlyphIndex, poSource.mpoMaps.moGlyphs.getBlock(), nInsertGlyphs);	// TODO:

// Apply the X-coordinate shifts
		for (i = nGlyphIndex; i < nGlyphRest; i++) {
			getGlyph(i).shift (oShiftNew);
		}
		for (i = nGlyphRest; i < getGlyphCount(); i++) {
			getGlyph(i).shift (oShiftNew);
		}

		if (! mbHasBIDI) {
// If this line bypassed Optyca, it will have cached character widths for
// simple layout.
			if (poSource.mbHasBIDI) {
// If the insertion did not bypass Optyca, we'll have to treat this line
// as if it didn't as well.  This means cancelling the cached character
// widths, setting the direction levels.
//				mpoMaps.moCharWidths.setSize (0);
//				mpoMaps.moDirectionLevels.allocate (nNewLength);	// TODO:
//				mpoDisplay.getWRS().defaultLevels (nOldLength, mbIsRTL, mpoMaps.moDirectionLevels.getData());
			}

//			else if (mpoMaps.moCharWidths.size() >= nOldLength) {
// On the other hand, if the insertion also bpasses Optyca, the character
// widths wull have to be updated to account for the insertion.
//				mpoMaps.moCharWidths.setSize (nNewLength);
//				InsertLineData (mpoMaps.moCharWidths.getBlock(), nOldLength, nIndex, poSource.mpoMaps.moCharWidths.getBlock(), nInsert);	// TODO:
//			}
		}

		if ((mpoMaps.moGlyphLocMap.size() > 0) || (poSource.mpoMaps.moGlyphLocMap.size() > 0)) {
// If either line has a glyph location map, insert the corresponding map
// from the source line.  Note that a default map may have to be
// manufactured for this line or the source.
// TBD: this assumes no reordering which is probably OK for current usage
			DispMap oTempMap = new DispMap();
			DispMap poSourceMap = poSource.mpoMaps.moGlyphLocMap;
			if (poSourceMap.size() == 0) {
				poSourceMap = oTempMap;
				manufactureGlyphLocMap (nInsert, oTempMap);
			}

			if (mpoMaps.moGlyphLocMap.size() == 0) {
				manufactureGlyphLocMap (nOldLength, mpoMaps.moGlyphLocMap);
			}

			mpoMaps.moGlyphLocMap.insertMap (poSourceMap, nIndex, nInsert);

			for (i = nIndex; i < getGlyphCount(); i++) {
				getGlyphLoc(i).setGlyphIndex (i);
			}
		}

// If direction levels have been cached, insert the new ones.
//		TextDirectionLevelArray poLevels = mpoMaps.moDirectionLevels;	// TODO:
//		TextDirectionLevelArray poInsertLevels = poSource.mpoMaps.moDirectionLevels;
//		if ((poLevels.getAlloc() >= nOldLength) && (poInsertLevels.getAlloc() >= nInsert)) {
//			poLevels.allocate (nNewLength, nOldLength);
//			InsertLineData (poLevels.getData(), nOldLength, nIndex, poInsertLevels.getData(), nInsert);
//		}

		moWidth += poSource.moWidth;
		mnVisualCharCount += nInsert;
	}

	TextAttr getLastAttr () {
		TextPosnBase oPosn = new TextPosnBase (stream(), Integer.MAX_VALUE);	// TODO: there has to be a more efficient way
		return oPosn.attributePtr();
	}

	void clear () {
		mpoMaps.clear();
	}

	void setMaps (DispMapSet maps) {
		mpoMaps = maps;
	}

	DispMapSet getMaps () {
		return mpoMaps;
	}

	private float removeGlyphs (int nStart, int nRemoveLength) {
		float oWidthRemoved = 0.0f;

		if (nRemoveLength == 0) {
			return oWidthRemoved;
		}

		int nNewLength = getGlyphCount() - nRemoveLength;

		float oStartX = getGlyph(nStart).getOriginalX();

//		RemoveLineData (mpoMaps.moGlyphs.getBlock(), nStart, nRemoveLength, GetGlyphCount());	// TODO:
		mpoMaps.moGlyphs.setSize (nNewLength);

		if (nStart >= getGlyphCount()) {
			return oWidthRemoved;
		}

		float oShift = oStartX - getGlyph (nStart).getOriginalX();
		for (int i = nStart; i < getGlyphCount(); i++) {
			Glyph oGlyph = getGlyph (i);
			oWidthRemoved += oGlyph.getOriginalNextX() - oGlyph.getOriginalX();
			oGlyph.shift (oShift);
		}

		return oWidthRemoved;
	}

//	private float getCharWidth (DispRun poRun, int nIndex, int c, boolean bApplySpacing, boolean bUseHorizontalGlyphs) {
//		DispEmbed poDispEmbed = isObject (nIndex);
//
//		if (poDispEmbed != null) {
//			return Units.toFloat (poDispEmbed.getEmbed().width());
//		}
//
//		else {
//			assert (poRun != null);
//			TextAttr poAttr = poRun.getAttr();
//			if (poAttr == null) {
//				return 0.0f;
//			}
//
//			FontInstance oFontInstance = poAttr.fontInstance();
//			if (oFontInstance != null) {
//				float oWidth = oFontInstance.getCharWidth (c, bUseHorizontalGlyphs);
//				if ((float) oWidth < 0.0) {
//					int oNotDefID = oFontInstance.getFontItem().getNotDefGlyphID();
//					oWidth = oFontInstance.getGlyphWidth (oNotDefID, bUseHorizontalGlyphs);
//				}
//
//// Adjust width by character spacing settings.
//				if (bApplySpacing) {
//					if (poAttr.charSpacingEnable()) {
//						oWidth += Units.toFloat (poAttr.charSpacing().getLength());
//					}
//					if (poAttr.wordSpacingEnable() && (getBreakClass (nIndex) == TextCharProp.BREAK_SP)) {
//						oWidth += Units.toFloat (poAttr.wordSpacing().getLength());
//					}
//				}
//
//				if (! legacyPositioning()) {
//// If not compatibility mode, we're done.
//					return oWidth;
//				}
//
//// If in compatibility mode, simulate the truncation and rounding of
//// horizontal metrics that used to occur.
//				UnitSpan oFontUnitSize = oFontInstance.getSize();
//
//// Determine the actual font size in floating-point points.
//				double nFontSize = oFontUnitSize.value() / 1000.0;
//
//// Scale the char width for a 1,000pt font and truncate to a whole point
//// value (like the old font service).  Conveniently, this is equivalent
//// to the 1pt font character width if the units were POINTS_1K;
//// Fix for Watson#1379884.	Use new getDoubleCharWidth method to simulate the results of getCharWidth
//// in a previous compiler version that generated a slightly different output.
//				double dWidth = oFontInstance.getDoubleCharWidth (c, bUseHorizontalGlyphs);
//				long nTruncWidth = (long) (1000.0 * dWidth / nFontSize);
//
//// Scale that truncated value by the current font size, in a unit span.
//				UnitSpan oResult = new UnitSpan (UnitSpan.POINTS_1K, (int) Math.round (nTruncWidth * nFontSize));
//
//// Round the resulting value to the nearest 1,200dpi position, just like
//// the old GFXFont class.
//				oResult.round (gLegacyWidthRound);
//				return Units.toFloat (oResult);
//			}
//		}
//
//		return 0.0f;
//	}

	private void testBIDI (int c, int eBreak) {
		if (mbHasBIDI)
			return;

		switch (TextCharProp.getBIDIClass (eBreak)) {
			case TextCharProp.BIDI_AL:
			case TextCharProp.BIDI_AN:			// TODO: does this alter algorithm?
			case TextCharProp.BIDI_R:
			case TextCharProp.BIDI_RLE:
			case TextCharProp.BIDI_RLO:
				mbHasBIDI = true;
				break;
		}
	}

	private static void manufactureGlyphLocMap (int nSize, DispMap oMap) {
		for (int i = 0; i < nSize; i++) {
			GlyphLoc oGlyphLoc = new GlyphLoc (i);
			oMap.add (oGlyphLoc, i, 1);
		}
	}
}
