/*
 * File: ParamInfoGroup.java
 * ************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.xmp.schema.rng.model;

import java.util.Stack;

/**
 * Defines a group of constraints on a property value.
 * 
 * @author pwollek
 */
public class ParamInfoGroup implements ParamInfo
{
	/**
	 * Operator to define relatioship between constraints defined by a stack of {@link ParamInfo} objects
	 * 
	 * @author pwollek
	 */
	public enum Operator
	{
		/**
		 * or operator
		 */
		kOR,
		/**
		 * and operator
		 */
		kAND;
	}

	private Stack<ParamInfo> mParamInfos; // ParamInfoImpl and/or ParamInfoGroup

	private final Operator mOperator;
	
	private boolean isChoice; // Represents if this group is defined inside a <rng:choice/>

	/**
	 * 
	 * Constructs a new ParamInfoGroup.
	 * 
	 * @param operator
	 *            Operator which defines relationship between a stack of {@link ParamInfo} objects
	 */
	public ParamInfoGroup(Operator operator)
	{
		mOperator = operator;
		mParamInfos = new Stack<ParamInfo>();
	}

	/**
	 * @return the mOperator
	 */
	public Operator getOperator()
	{
		return mOperator;
	}

	/**
	 * Pushes the {@link ParamInfo} object in this {@link ParamInfoGroup}
	 * 
	 * @param info
	 *            {@link ParamInfo} object to push
	 */
	public void push(ParamInfo info)
	{
		boolean exists = false;

		ParamInfo newInfo = info;

		if (info instanceof ParamInfoGroup)
		{
			ParamInfoGroup group = (ParamInfoGroup) info;

			if (group.size() == 1)
				newInfo = group.pop();
		}

		for (ParamInfo param : mParamInfos)
		{
			exists |= param.equals(newInfo);
		}

		if (!exists)
			mParamInfos.push(newInfo);
	}

	ParamInfo pop()
	{
		return mParamInfos.pop();
	}

	/**
	 * Tests if this ParamInfoGroup is empty.
	 * 
	 * @return <code>true</code> if and only if this ParamInfoGroup contains no ParamInfo items; <code>false</code>
	 *         otherwise.
	 */
	public boolean empty()
	{
		return mParamInfos.empty();
	}

	int size()
	{
		return mParamInfos.size();
	}

	/**
	 * @return Stack of {@link ParamInfo} objects contained in this {@link ParamInfoGroup}
	 */
	public Stack<ParamInfo> getParams()
	{
		return mParamInfos;
	}

	public boolean equals(ParamInfo param)
	{
		boolean ret = false;

		if (param instanceof ParamInfoGroup)
		{
			ParamInfoGroup group = (ParamInfoGroup) param;

			if (this.size() == group.size())
			{
				for (int i = 0; i < this.size(); i++)
				{
					ret |= mParamInfos.get(i).equals(group.mParamInfos.get(i));
				}
			}
		}

		return ret;
	}
	
	/**
	 * Set if this param group is defined inside a rng:choice
	 * @param isChoice boolean
	 */
	public void setChoice(boolean isChoice)
	{
		this.isChoice = isChoice;
	}
	
	/**
	 * Gets if this param group is defined inside a rng:choice
	 * @return boolean
	 */
	public boolean isChoice()
	{
		return isChoice;
	}
	
	@Override
	public String toString()
	{
		StringBuilder str = new StringBuilder();
		str.append("========== ParamInfoGroup =========");
		str.append("\nOperator : ").append(mOperator);
		if(mParamInfos != null)
		{
			for(ParamInfo param : mParamInfos)
				str.append(param.toString());
		}
		return str.toString();
	}
}
