/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.day.cq.dam.commons.util;

import com.adobe.granite.security.user.util.AuthorizableUtil;
import com.adobe.granite.toggle.api.ToggleRouter;
import com.adobe.granite.ui.preview.PreviewEnabledService;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.i18n.I18n;
import com.day.cq.replication.Agent;
import com.day.cq.replication.AgentManager;
import com.day.cq.replication.ReplicationActionType;
import com.day.cq.replication.ReplicationQueue;
import com.day.cq.replication.ReplicationStatus;
import com.day.cq.workflow.exec.Workflow;
import com.day.cq.workflow.exec.WorkflowData;
import com.day.cq.workflow.metadata.MetaDataMap;
import com.day.cq.workflow.status.WorkflowStatus;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.scripting.SlingScriptHelper;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;

import javax.servlet.http.HttpServletRequest;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Map;

import static com.day.cq.workflow.job.AbsoluteTimeoutHandler.ABS_TIME;
import static com.day.cq.commons.servlets.AbstractListServlet.ListItem.SCHEDULED_ACTIVATION_WORKFLOW_ID;
import static com.day.cq.commons.servlets.AbstractListServlet.ListItem.SCHEDULED_DEACTIVATION_WORKFLOW_ID;

public class AssetPublicationUtil {

    private static final String AGENT_ID = "agentId";
    private static final String AGENT_ID_PREVIEW = "preview";
    private static final String AGENT_ID_PUBLISH = "publish";

    private static final String BUILD_ENTRY_MAP = "buildEntryMap";
    private static final String BUILD_ENTRY_MAP_PREVIEW = "buildEntryMapPreview";

    private static final String SCHEDULED_ACTIVATION_WORKFLOW_ID_NEW = "/var/workflow/models/scheduled_activation";
    private static final String SCHEDULED_TREE_ACTIVATION_WORKFLOW_ID_VAR = "/var/workflow/models/scheduled_tree_activation";
    private static final String SCHEDULED_DEACTIVATION_WORKFLOW_ID_NEW = "/var/workflow/models/scheduled_deactivation";

    private static final String FT_ASSETS_PREVIEW = "ft-sites-125";

    private static String getStatus(int maxQueuePos, String actionType, I18n i18n) {
        String status = "";
        if (maxQueuePos > 0) {
            if ("Activate".equals(actionType)) {
                status = i18n.get("Publication Pending. #{0} in the queue.", "0 is the position in the queue", maxQueuePos);
            } else {
                status = i18n.get("Un-publication Pending. #{0} in the queue.", "0 is the position in the queue", maxQueuePos);
            }
        }
        return status;
    }

    /**
     * Publication Related Methods
     */
    public static String getPendingStatusFromReplication(ReplicationStatus replicationStatus, I18n i18n) {
        if (replicationStatus == null) return "";
        String actionType = "";
        int maxQueuePos = -1;
        for (ReplicationQueue.Entry e : replicationStatus.getPending()) {
            if (e.getQueuePosition() > maxQueuePos) {
                maxQueuePos = e.getQueuePosition();
                actionType = e.getAction().getType().getName();
            }
        }
        maxQueuePos = maxQueuePos + 1;

        return getStatus(maxQueuePos, actionType, i18n);
    }

    /**
     * @return pending status of an asset
     */
    private static String getPendingStatusUsingAgents(SlingScriptHelper sling, Resource resource, I18n i18n, String agentId) {
        String path = resource.getPath();
        ReplicationStatus replicationStatus = resource.adaptTo(ReplicationStatus.class);
        if(replicationStatus != null && StringUtils.isNotEmpty(agentId)) {
            replicationStatus = replicationStatus.getStatusForAgent(agentId);
        }
        AgentManager agentMgr = sling.getService(AgentManager.class);

        if (replicationStatus == null) return "";
        String actionType = "";
        int maxQueuePos = -1;
        ReplicationActionType type = replicationStatus.getLastReplicationAction();
        Calendar last = replicationStatus.getLastPublished();
        if(type != null && last != null) {
            long time = last.getTimeInMillis();
            for(Agent agent: agentMgr.getAgents().values()) {
                if(!agent.isInMaintenanceMode()) {
                    // Excluding preview agent for default(current implementation) publication station.
                    if(StringUtils.isEmpty(agentId) && !StringUtils.equals(agent.getId(), AGENT_ID_PREVIEW)) {
                        maxQueuePos = Math.max(findMaxPositionInAgentQueue(agent, path, type, time), maxQueuePos);
                    } else if(StringUtils.equals(agent.getId(), agentId)) {
                        maxQueuePos = Math.max(findMaxPositionInAgentQueue(agent, path, type, time), maxQueuePos);
                        break;
                    }
                }
            }
        }
        maxQueuePos = maxQueuePos + 1;

        return getStatus(maxQueuePos, actionType, i18n);
    }

    private static int findMaxPositionInAgentQueue(Agent agent, String path, ReplicationActionType type, long time) {
        int maxQueuePos = -1;
        ReplicationQueue queue = agent.getQueue();
        if(queue != null) {
            for(ReplicationQueue.Entry e: queue.entries(path)) {
                if(type == e.getAction().getType() && time == e.getAction().getTime() && e.getQueuePosition() > maxQueuePos) {
                    maxQueuePos = e.getQueuePosition();
                }
            }
        }
        return maxQueuePos;
    }

    /**
     * @return returns the publication pending status.
     */
    public static String getPendingStatusFromCache(HttpServletRequest request, Resource resource , I18n i18n) {
        return computePendingStatusUsingCache(request, resource, i18n, null);
    }

    /**
     * @return return preview publication pending status.
     */
    public static String getPendingStatusOfPreviewFromCache(HttpServletRequest request, Resource resource , I18n i18n) {
        return computePendingStatusUsingCache(request, resource, i18n, AGENT_ID_PREVIEW);
    }

    /**
     * If agentId is null then it return publication status (i.e value of PUBLISHED column in list view)
     * If agentId is AGENT_ID_PREVIEW then it return preview publication status
     */
    private static String computePendingStatusUsingCache(HttpServletRequest request, Resource resource , I18n i18n, String agentId) {
        String actionType = "";
        int maxQueuePos = -1;
        String path = resource.getPath();
        String entryMapKey = BUILD_ENTRY_MAP;
        if(StringUtils.equals(agentId, AGENT_ID_PREVIEW)) {
            entryMapKey = BUILD_ENTRY_MAP_PREVIEW;
        }
        Map<String, Pair <String,Integer>> buildEntryMap =  (Map<String, Pair <String,Integer>>)request.getAttribute(entryMapKey);

        if(buildEntryMap != null && buildEntryMap.containsKey(path)) {
            Pair<String, Integer> pair = buildEntryMap.get(path);
            actionType = pair.getKey();
            maxQueuePos = pair.getValue();
        }

        maxQueuePos = maxQueuePos + 1;

        return getStatus(maxQueuePos, actionType, i18n);
    }

    public static String formatTime(Locale locale, Long time) {
        DateTimeFormatter pattern = DateTimeFormatter.ofLocalizedDateTime(FormatStyle.MEDIUM);
        ZonedDateTime zonedDateTime = ZonedDateTime.ofInstant(new Date(time).toInstant(), ZoneId.systemDefault());
        return zonedDateTime.format(pattern);
    }

    public static Calendar getLastReplicated(ValueMap properties) {
        if(properties == null) {
            return null;
        }
        return properties.get(DamConstants.PN_PAGE_LAST_REPLICATED, Calendar.class);
    }

    public static Calendar getLastReplicatedPreview(ValueMap properties) {
        if(properties == null) {
            return null;
        }
        String lastReplicatedProperty = String.join("_", DamConstants.PN_PAGE_LAST_REPLICATED, AGENT_ID_PREVIEW);
        return properties.get(lastReplicatedProperty, Calendar.class);
    }

    public static String getLastReplicatedBy(ValueMap properties) {
        if(properties == null) {
            return null;
        }
        return properties.get(DamConstants.PN_PAGE_LAST_REPLICATED_BY, String.class);
    }

    public static String getLastReplicatedByPreview(ValueMap properties) {
        if(properties == null) {
            return null;
        }
        String lastReplicatedByProperty = String.join("_", DamConstants.PN_PAGE_LAST_REPLICATED_BY, AGENT_ID_PREVIEW);
        return properties.get(lastReplicatedByProperty, String.class);
    }

    public static String getLastReplicationAction(ValueMap properties) {
        if(properties == null) {
            return null;
        }
        return properties.get(DamConstants.PN_PAGE_LAST_REPLICATION_ACTION, String.class);
    }

    public static String getLastReplicationActionPreview(ValueMap properties) {
        if(properties == null) {
            return null;
        }
        String lastReplicationActionProperty = String.join("_", DamConstants.PN_PAGE_LAST_REPLICATION_ACTION, AGENT_ID_PREVIEW);
        return properties.get(lastReplicationActionProperty, String.class);
    }

    public static String getPendingStatus(HttpServletRequest httpRequest, Resource resource, I18n i18n,
                                          SlingScriptHelper sling) {
        Map<String, Pair <String,Integer>>  entryMap = (Map<String, Pair <String,Integer>>)httpRequest.getAttribute(BUILD_ENTRY_MAP);

        String pendingStatus;
        if(entryMap != null) {
            pendingStatus = getPendingStatusFromCache(httpRequest, resource, i18n);
        } else if(sling == null) {
            ReplicationStatus replicationStatus = resource.adaptTo(ReplicationStatus.class);
            pendingStatus = getPendingStatusFromReplication(replicationStatus, i18n);
        } else {
            pendingStatus = getPendingStatusUsingAgents(sling, resource, i18n, null);
        }
        return pendingStatus;
    }

    public static String getPendingStatusOfPreview(HttpServletRequest httpRequest, Resource resource, I18n i18n,
                                          SlingScriptHelper sling) {
        Map<String, Pair <String,Integer>>  entryMap = (Map<String, Pair <String,Integer>>)httpRequest.getAttribute(BUILD_ENTRY_MAP_PREVIEW);

        String pendingStatus;
        if(entryMap != null) {
            pendingStatus = getPendingStatusOfPreviewFromCache(httpRequest, resource, i18n);
        } else {
            pendingStatus = getPendingStatusUsingAgents(sling, resource, i18n, AGENT_ID_PREVIEW);
        }
        return pendingStatus;
    }

    public static String getScheduledStatus(List<Workflow> scheduledWorkflows, I18n i18n, Resource resource,
                                      SlingHttpServletRequest httpServletRequest) {
        String resourcePath = resource.getPath();
        StringBuilder status = new StringBuilder();
        int i = 0;

        for (Workflow scheduledWorkflow : scheduledWorkflows) {
            if (i > 0) {
                status.append("\n\n");
            }

            if (isScheduledActivationWorkflow(scheduledWorkflow)) {
                status.append(i18n.get("Publication Pending")).append("\n");
                try {
                    String versions = scheduledWorkflow.getWorkflowData().getMetaDataMap().get("versions", String.class);
                    if (versions != null) {
                        JSONObject jsonObject = new JSONObject(versions);
                        String version = jsonObject.get(resourcePath).toString();
                        if (StringUtils.isNotEmpty(version)) {
                            status.append(i18n.get("Version")).append(": ").append(version).append("\n");
                        }
                    }
                } catch (JSONException jse) {
                    // ignore
                }
            } else {
                status.append(i18n.get("Un-publication Pending")).append("\n");
            }

            status.append(i18n.get("Scheduled")).append(": ");
            status.append(
                    formatTime(httpServletRequest.getLocale(),
                            scheduledWorkflow.getWorkflowData().getMetaDataMap().get(ABS_TIME, Long.class)
                    ));
            status.append(" (")
                    .append(AuthorizableUtil.getFormattedName(
                            httpServletRequest.getResourceResolver(), scheduledWorkflow.getInitiator()))
                    .append(")");
            i++;
        }

        return status.toString();
    }

    public static List<Workflow> getScheduledWorkflows(WorkflowStatus workflowStatus) {
        return computeScheduledWorkflows(workflowStatus, null);
    }

    public static List<Workflow> getScheduledWorkflowsOfPreview(WorkflowStatus workflowStatus) {
        return computeScheduledWorkflows(workflowStatus, AGENT_ID_PREVIEW);
    }

    private static List<Workflow> computeScheduledWorkflows(WorkflowStatus workflowStatus, String agentId) {
        List<Workflow> scheduledWorkflows = new LinkedList<Workflow>();

        // Get the scheduled workflows
        if (workflowStatus != null) {
            List<Workflow> workflows = workflowStatus.getWorkflows(false);
            for (Workflow workflow : workflows) {
                WorkflowData workflowData = workflow.getWorkflowData();
                MetaDataMap metaDataMap = workflowData != null ? workflow.getMetaDataMap(): null;
                if (metaDataMap != null) {
                    String agentIdMetaData = metaDataMap.get(AGENT_ID, String.class);
                    if (StringUtils.isNotEmpty(agentId)) {
                        if (!StringUtils.equals(agentId, agentIdMetaData)) {
                            continue;
                        }
                    } else if(StringUtils.equals(agentIdMetaData, AGENT_ID_PREVIEW)) {
                        continue;
                    }
                    if (isScheduledActivationWorkflow(workflow) || isScheduledDeactivationWorkflow(workflow)) {
                        scheduledWorkflows.add(workflow);
                    }
                }
            }
        }

        // Sort the scheduled workflows by time started
        Collections.sort(scheduledWorkflows, new Comparator<Workflow>() {
            public int compare(Workflow o1, Workflow o2) {
                return o1.getTimeStarted().compareTo(o2.getTimeStarted());
            }
        });

        return scheduledWorkflows;
    }

    // TODO: Replace constants with a workflow API tracked by CQ-4222642
    public static boolean isScheduledActivationWorkflow(Workflow workflow) {
        return (workflow != null
                && (workflow.getWorkflowModel().getId().equals(SCHEDULED_ACTIVATION_WORKFLOW_ID)
                || workflow.getWorkflowModel().getId().equals(SCHEDULED_ACTIVATION_WORKFLOW_ID_NEW)
                || workflow.getWorkflowModel().getId().equals(SCHEDULED_TREE_ACTIVATION_WORKFLOW_ID_VAR))
        );
    }

    public static boolean isScheduledDeactivationWorkflow(Workflow workflow) {
        return (workflow != null
                && (workflow.getWorkflowModel().getId().equals(SCHEDULED_DEACTIVATION_WORKFLOW_ID)
                || workflow.getWorkflowModel().getId().equals(SCHEDULED_DEACTIVATION_WORKFLOW_ID_NEW))
        );
    }

    public static boolean isPreviewEnabled(SlingScriptHelper sling) {
        PreviewEnabledService previewEnabledService = sling.getService(PreviewEnabledService.class);
        ToggleRouter toggleRtr = sling.getService(ToggleRouter.class);
        return previewEnabledService!= null && previewEnabledService.isEnabled() && toggleRtr!= null &&
                toggleRtr.isEnabled(FT_ASSETS_PREVIEW);
    }
}