/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.mcm.campaign;

import com.day.cq.wcm.api.Page;
import org.apache.sling.api.resource.PersistenceException;

/**
 * Interface providing methods to interact with and modify Adobe Campaign newsletter
 * pages. All methods (except {@link #isNewsletter(com.day.cq.wcm.api.Page)} must be
 * called with a newsletter page, or else they will throw a {@link NewsletterException}.
 */
public interface NewsletterManager {

    /**
     * Checks if the specified page is a newsletter, i.e. if its resource type
     * inherits from the base campaign newsletter component.
     *
     * @param page A page
     * @return true if the specified page is a newsletter, false otherwise
     */
    boolean isNewsletter(Page page);

    /**
     * Returns true if the newsletter is linked to the specified delivery.
     *
     * @param newsletter A newsletter
     * @param deliveryId A delivery id
     * @return true if the newsletter is linked to the specified delivery, false otherwise
     * @throws NewsletterException if the specified page is not a newsletter
     */
    boolean isLinked(Page newsletter, String deliveryId) throws NewsletterException;

    /**
     * Returns an array with currently linked deliveries.
     *
     * @param newsletter The newsletter page to check
     * @return An array with linked deliveries; empty array if no deliveries are linked
     * @throws NewsletterException If the specified page is not a newsletter
     */
    String[] getLinkedDeliveries(Page newsletter) throws NewsletterException;

    /**
     * Links a newsletter with an Adobe Campaign delivery.
     *
     * @param newsletter The newsletter page to link
     * @param deliveryId The id of the delivery to link with the newsletter
     * @return The uuid identifying the newsletter
     * @throws NewsletterException if the delivery is already linked with the newsletter
     * @throws PersistenceException
     */
    String link(Page newsletter, String deliveryId) throws NewsletterException, PersistenceException;

    /**
     * Unlinks a newsletter from an Adobe Campaign delivery.
     *
     * @param newsletter The newsletter page to unlink
     * @param deliveryId The id of the delivery to unlink the newsletter from
     * @throws NewsletterException if the newsletter is not linked to the delivery or
     *          if the delivery has been used to lock the newsletter
     * @throws PersistenceException
     */
    void unlink(Page newsletter, String deliveryId) throws NewsletterException, PersistenceException;

    /**
     * Checks if a newsletter has been modified after a specified date.
     *
     * @param newsletter A newsletter page
     * @param timestamp The date to check against
     * @return true if the newsletter has been modified since, false otherwise
     * @throws NewsletterException
     */
    boolean isModifiedSince(Page newsletter, long timestamp) throws NewsletterException;

    /**
     * Returns the date of the last modification of a newsletter.
     *
     * @param newsletter A newsletter page
     * @return The timestamp of the last modification of the specified newsletter
     * @throws NewsletterException
     */
    long getLastModified(Page newsletter) throws NewsletterException;

    /**
     * Gets a suitable subject for the specified newsletter.
     *
     * @param newsletter A newsletter page
     * @return The subject of the specified newsletter
     * @throws NewsletterException if the subject could not be determined
     */
    String getSubject(Page newsletter) throws NewsletterException;

    /**
     * Gets the plain text variant of the specified newsletter (if available).
     *
     * @param newsletter A newsletter page
     * @return The plain text variant of the specified newsletter; <code>null</code> if no
     *         text variant is available
     * @throws NewsletterException
     */
    String getPlainText(Page newsletter) throws NewsletterException;

    /**
     * Checks if a newsletter is approved.
     *
     * @param newsletter A newsletter page
     * @return true if the specified newsletter is approved, false otherwise
     * @throws NewsletterException
     */
    boolean isApproved(Page newsletter) throws NewsletterException;

    /**
     * Marks a newsletter as approved.
     *
     * @param newsletter A newsletter page to approve
     * @throws NewsletterException if the newsletter has already been approved
     * @throws PersistenceException
     */
    void approve(Page newsletter) throws NewsletterException, PersistenceException;

    /**
     * Revokes the approval of a newsletter.
     *
     * @param newsletter A newsletter page to disapprove
     * @throws NewsletterException if the newsletter is not currently approved
     * @throws PersistenceException
     */
    void disapprove(Page newsletter) throws NewsletterException, PersistenceException;

    /**
     * Marks a newsletter as sent by a delivery.
     *
     * @param newsletter A newsletter page
     * @param deliveryId The delivery used to send the newsletter
     * @throws NewsletterException if the newsletter is not linked to the delivery
     *          or has already been marked as sent
     * @throws PersistenceException
     */
    void markAsSent(Page newsletter, String deliveryId) throws NewsletterException, PersistenceException;

    /**
     * Locks a newsletter to the currently active session. The lock is associated to the
     * specified delivery, meaning that the same delivery must be used to unlock the
     * newsletter.
     *
     * @param newsletter A newsletter page to lock
     * @param deliveryId The delivery requesting to lock the newsletter
     * @throws NewsletterException if the newsletter is already locked, not linked
     *          to the delivery or if an error occurred
     */
    void lock(Page newsletter, String deliveryId) throws NewsletterException;

    /**
     * Unlocks a newsletter. Must be called with the same delivery that was used
     * to lock the newsletter.
     *
     * @param newsletter A newsletter page to unlock
     * @param deliveryId The delivery requesting to unlock the newsletter
     * @throws NewsletterException if the newsletter is not currently locked, was
     *          locked by another delivery or an error occurred
     */
    void unlock(Page newsletter, String deliveryId) throws NewsletterException;

    /**
     * Replicates a newsletter and all its referenced resources to the publish instance(s).
     *
     * @param newsletter A newsletter page to publish
     * @throws NewsletterException if session is not authorized to replicate the
     *          newsletter or if the replication did not succeed
     */
    void publish(Page newsletter) throws NewsletterException;

}
