/*
 * ************************************************************************
 *  *
 *  * ADOBE CONFIDENTIAL
 *  * __________________
 *  *
 *  *  Copyright 2023 Adobe Incorporated
 *  *  All Rights Reserved.
 *  *
 *  * NOTICE:  All information contained herein is, and remains
 *  * the property of Adobe Incorporated and its suppliers,
 *  * if any.  The intellectual and technical concepts contained
 *  * herein are proprietary to Adobe Incorporated and its
 *  * suppliers and are protected by trade secret or copyright law.
 *  * Dissemination of this information or reproduction of this material
 *  * is strictly forbidden unless prior written permission is obtained
 *  * from Adobe Incorporated.
 *  *************************************************************************
 */
package com.adobe.cq.wcm.launches.cf;

import com.adobe.cq.launches.api.LaunchException;
import com.adobe.cq.launches.api.LaunchManager;
import com.adobe.cq.launches.api.LaunchSource;
import com.day.cq.wcm.api.Page;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.sling.api.resource.Resource;

import java.util.List;

import org.osgi.annotation.versioning.ProviderType;

/**
 * Content fragment manager to perform operations like rebasing / getting differences between production and launch /
 * promoting to production / deleting to a launch and to the resource contained inside.
 */
@ProviderType
public interface ContentFragmentLaunchManager extends LaunchManager {

    /**
     * Rebase a launch based on the main content (brings changes from the main content into the Launch).
     * @param launch Content Fragment Launch
     * @param params Parameters of the rebase operation of the Content Fragment Launch
     * @throws LaunchException If the rebase operation fails
     */
    void rebaseLaunch(@Nonnull ContentFragmentLaunch launch, @Nullable ContentFragmentLaunchOperationParameters params) throws LaunchException;

    /**
     * Gets the differences between the Content Fragment Launch and main content.
     * @param cfLaunch Content Fragment Launch
     * @param params Parameters used to compute differences of a Content Fragment Launch
     * @return list of rebase diff entries
     */
    List<RebaseDiffEntry> getDiffs(@Nonnull ContentFragmentLaunch cfLaunch, @Nullable ContentFragmentLaunchOperationParameters params);

    /**
     * Moves a resource to a new destination inside a Launch
     * This method accounts for the source resource title, avoids name collisions and
     * enables certain limitations that only apply to resources inside Content Fragment Launch.
     * @param srcResource Source resource
     * @param destinationResourcePath Resource path of the destination
     * @param dstTitle title of the moved resource
     * @return moved resource
     */
    Resource moveResource(@Nonnull Resource srcResource, @Nonnull String destinationResourcePath, @Nullable String dstTitle);

    /**
     * Copies a resource to a new destination inside a Launch.
     * This method accounts for the source resource title, avoids name collisions and
     * enables certain limitations that only apply to resources inside Content Fragment Launch.
     * @param srcResource Source resource
     * @param destinationResourcePath Resource path of the destination
     * @param dstTitle title of the copied resource
     * @return copied resource
     */
    Resource copyResource(@Nonnull Resource srcResource, @Nonnull String destinationResourcePath, @Nullable String dstTitle);

    /**
     * Deletes a resource present inside a Launch.
     * This method enables certain limitations that only apply to resources inside Content Fragment Launch.
     * @param resource Resource to be deleted
     */
    void deleteResource(@Nonnull Resource resource);

    /**
     * Updates the list of Content Fragments sources of a Content Fragment Launch.
     * @param launchContentResource The content resource of a Launch (which holds the Content Fragment Launch sources)
     * @param inputLaunchResourceList List of new Content Fragment Launch resources to replace the old one as sources
     * @throws LaunchException If the rebase operation fails
     */
    void updateSourceListInLaunch(@Nonnull Resource launchContentResource, @Nonnull List<LaunchSource> inputLaunchResourceList) throws LaunchException;

    /**
     * Performs actions like:
     * - setting the last rebase field
     * - updating references
     * to a Content Fragment Launch and its resources after creating them.
     * @param launchPageResource The root page resource of a Content Fragment Launch
     * @param launchContentResources List of content resources previously added to the Content Framgnet Launch
     */
    void updateLaunchAfterCreatingResources(@Nonnull Resource launchPageResource, @Nonnull List<Resource> launchContentResources);

    /**
     * Creates content resources inside the Content Fragment Launch.
     * @param inputLaunchResourceList The list of sources used for creating the Content Fragment Launch content resources
     * @param launchPageResource The root page resource of a Content Fragment Launch
     * @param launchPage The root page of a Content Fragment Launch
     * @return The list of created content resources
     */
    List<Resource> createLaunchContentResources(@Nonnull List<LaunchSource> inputLaunchResourceList, @Nonnull Resource launchPageResource, @Nonnull Page launchPage);
}
