/* ****************************************************************************
 *
 *	File: PDFStandardFontDescriptors.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.util;

import java.util.HashMap;

import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFUnableToCompleteOperationException;
import com.adobe.internal.pdftoolkit.core.types.ASDictionary;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASNumber;
import com.adobe.internal.pdftoolkit.core.types.ASRectangle;

abstract class FDCreator
{
	abstract ASDictionary create(ASName name);
}

final class Helvetica_FDCreator extends FDCreator
{
	Helvetica_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(48));
		fd.put(ASName.k_FontBBox, new ASRectangle(-166, -225, 1000, 931));
		fd.put(ASName.k_ItalicAngle, new ASNumber(0));
		fd.put(ASName.k_Ascent, new ASNumber(718));
		fd.put(ASName.k_Descent, new ASNumber(-207));
		fd.put(ASName.k_CapHeight, new ASNumber(718));
		fd.put(ASName.k_XHeight, new ASNumber(523));
		fd.put(ASName.k_StemV, new ASNumber(88));
		fd.put(ASName.k_StemH, new ASNumber(76));
		return fd;
	}
}

final class HelveticaOblique_FDCreator extends FDCreator
{
	HelveticaOblique_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(112));
		fd.put(ASName.k_FontBBox, new ASRectangle(-170, -225, 1116, 931));
		fd.put(ASName.k_ItalicAngle, new ASNumber(-12));
		fd.put(ASName.k_Ascent, new ASNumber(718));
		fd.put(ASName.k_Descent, new ASNumber(-207));
		fd.put(ASName.k_CapHeight, new ASNumber(718));
		fd.put(ASName.k_XHeight, new ASNumber(523));
		fd.put(ASName.k_StemV, new ASNumber(88));
		fd.put(ASName.k_StemH, new ASNumber(76));
		return fd;
	}
}

final class HelveticaBold_FDCreator extends FDCreator
{
	HelveticaBold_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(48));
		fd.put(ASName.k_FontBBox, new ASRectangle(-170, -228, 1003, 962));
		fd.put(ASName.k_ItalicAngle, new ASNumber(0));
		fd.put(ASName.k_Ascent, new ASNumber(718));
		fd.put(ASName.k_Descent, new ASNumber(-207));
		fd.put(ASName.k_CapHeight, new ASNumber(718));
		fd.put(ASName.k_XHeight, new ASNumber(532));
		fd.put(ASName.k_StemV, new ASNumber(140));
		fd.put(ASName.k_StemH, new ASNumber(118));
		return fd;
	}
}

final class HelveticaBoldOblique_FDCreator extends FDCreator
{
	HelveticaBoldOblique_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(112));
		fd.put(ASName.k_FontBBox, new ASRectangle(-174, -228, 1114, 962));
		fd.put(ASName.k_ItalicAngle, new ASNumber(-12));
		fd.put(ASName.k_Ascent, new ASNumber(718));
		fd.put(ASName.k_Descent, new ASNumber(-207));
		fd.put(ASName.k_CapHeight, new ASNumber(718));
		fd.put(ASName.k_XHeight, new ASNumber(532));
		fd.put(ASName.k_StemV, new ASNumber(88));
		fd.put(ASName.k_StemH, new ASNumber(76));
		return fd;
	}
}

final class TimesRoman_FDCreator extends FDCreator
{
	TimesRoman_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(50));
		fd.put(ASName.k_FontBBox, new ASRectangle(-168, -218, 1000, 898));
		fd.put(ASName.k_ItalicAngle, new ASNumber(0));
		fd.put(ASName.k_Ascent, new ASNumber(683));
		fd.put(ASName.k_Descent, new ASNumber(-217));
		fd.put(ASName.k_CapHeight, new ASNumber(662));
		fd.put(ASName.k_XHeight, new ASNumber(450));
		fd.put(ASName.k_StemV, new ASNumber(84));
		fd.put(ASName.k_StemH, new ASNumber(28));
		return fd;
	}
}

final class TimesItalic_FDCreator extends FDCreator
{
	TimesItalic_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(114));
		fd.put(ASName.k_FontBBox, new ASRectangle(-169, -217, 1010, 883));
		fd.put(ASName.k_ItalicAngle, new ASNumber(-15.5));
		fd.put(ASName.k_Ascent, new ASNumber(683));
		fd.put(ASName.k_Descent, new ASNumber(-217));
		fd.put(ASName.k_CapHeight, new ASNumber(653));
		fd.put(ASName.k_XHeight, new ASNumber(441));
		fd.put(ASName.k_StemV, new ASNumber(76));
		fd.put(ASName.k_StemH, new ASNumber(32));
		return fd;
	}
}

final class TimesBold_FDCreator extends FDCreator
{
	TimesBold_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(50));
		fd.put(ASName.k_FontBBox, new ASRectangle(-168, -218, 1000, 935));
		fd.put(ASName.k_ItalicAngle, new ASNumber(0));
		fd.put(ASName.k_Ascent, new ASNumber(683));
		fd.put(ASName.k_Descent, new ASNumber(-217));
		fd.put(ASName.k_CapHeight, new ASNumber(676));
		fd.put(ASName.k_XHeight, new ASNumber(461));
		fd.put(ASName.k_StemV, new ASNumber(139));
		fd.put(ASName.k_StemH, new ASNumber(44));
		return fd;
	}
}

final class Courier_FDCreator extends FDCreator
{
	Courier_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(51));
		fd.put(ASName.k_FontBBox, new ASRectangle(-23, -250, 715, 805));
		fd.put(ASName.k_ItalicAngle, new ASNumber(0));
		fd.put(ASName.k_Ascent, new ASNumber(629));
		fd.put(ASName.k_Descent, new ASNumber(-157));
		fd.put(ASName.k_CapHeight, new ASNumber(562));
		fd.put(ASName.k_XHeight, new ASNumber(426));
		fd.put(ASName.k_StemV, new ASNumber(51));
		fd.put(ASName.k_StemH, new ASNumber(51));
		return fd;
	}
}

final class CourierOblique_FDCreator extends FDCreator
{
	CourierOblique_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(115));
		fd.put(ASName.k_FontBBox, new ASRectangle(-27, -250, 849, 805));
		fd.put(ASName.k_ItalicAngle, new ASNumber(-12));
		fd.put(ASName.k_Ascent, new ASNumber(629));
		fd.put(ASName.k_Descent, new ASNumber(-157));
		fd.put(ASName.k_CapHeight, new ASNumber(562));
		fd.put(ASName.k_XHeight, new ASNumber(426));
		fd.put(ASName.k_StemV, new ASNumber(51));
		fd.put(ASName.k_StemH, new ASNumber(51));
		return fd;
	}
}

final class CourierBold_FDCreator extends FDCreator
{
	CourierBold_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(51));
		fd.put(ASName.k_FontBBox, new ASRectangle(-113, -250, 749, 801));
		fd.put(ASName.k_ItalicAngle, new ASNumber(0));
		fd.put(ASName.k_Ascent, new ASNumber(629));
		fd.put(ASName.k_Descent, new ASNumber(-157));
		fd.put(ASName.k_CapHeight, new ASNumber(562));
		fd.put(ASName.k_XHeight, new ASNumber(439));
		fd.put(ASName.k_StemV, new ASNumber(106));
		fd.put(ASName.k_StemH, new ASNumber(84));
		return fd;
	}
}

final class CourierBoldOblique_FDCreator extends FDCreator
{
	CourierBoldOblique_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(115));
		fd.put(ASName.k_FontBBox, new ASRectangle(-57, -250, 869, 801));
		fd.put(ASName.k_ItalicAngle, new ASNumber(-12));
		fd.put(ASName.k_Ascent, new ASNumber(629));
		fd.put(ASName.k_Descent, new ASNumber(-157));
		fd.put(ASName.k_CapHeight, new ASNumber(562));
		fd.put(ASName.k_XHeight, new ASNumber(439));
		fd.put(ASName.k_StemV, new ASNumber(106));
		fd.put(ASName.k_StemH, new ASNumber(84));
		return fd;
	}
}


final class TimesBoldItalic_FDCreator extends FDCreator
{
	TimesBoldItalic_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(114));
		fd.put(ASName.k_FontBBox, new ASRectangle(-200, -218, 996, 921));
		fd.put(ASName.k_ItalicAngle, new ASNumber(-15));
		fd.put(ASName.k_Ascent, new ASNumber(683));
		fd.put(ASName.k_Descent, new ASNumber(-217));
		fd.put(ASName.k_CapHeight, new ASNumber(669));
		fd.put(ASName.k_XHeight, new ASNumber(462));
		fd.put(ASName.k_StemV, new ASNumber(121));
		fd.put(ASName.k_StemH, new ASNumber(42));
		return fd;
	}
}

final class Symbol_FDCreator extends FDCreator
{
	Symbol_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(20));
		fd.put(ASName.k_FontBBox, new ASRectangle(-180, -293, 1090, 1010));
		fd.put(ASName.k_ItalicAngle, new ASNumber(0));
		fd.put(ASName.k_Ascent, new ASNumber(0));
		fd.put(ASName.k_Descent, new ASNumber(0));
		fd.put(ASName.k_CapHeight, new ASNumber(0));
		fd.put(ASName.k_XHeight, new ASNumber(0));
		fd.put(ASName.k_StemV, new ASNumber(85));
		fd.put(ASName.k_StemH, new ASNumber(92));
		return fd;
	}
}

final class ZapfDingbats_FDCreator extends FDCreator
{
	ZapfDingbats_FDCreator() {}

	@Override
	ASDictionary create(ASName name)
	{
		ASDictionary fd = new ASDictionary();
		fd.put(ASName.k_Type, ASName.k_FontDescriptor);
		fd.put(ASName.k_FontName, name);
		fd.put(ASName.k_Flags, new ASNumber(20));
		fd.put(ASName.k_FontBBox, new ASRectangle(-1, -143, 981, 820));
		fd.put(ASName.k_ItalicAngle, new ASNumber(0));
		fd.put(ASName.k_Ascent, new ASNumber(0));
		fd.put(ASName.k_Descent, new ASNumber(0));
		fd.put(ASName.k_CapHeight, new ASNumber(0));
		fd.put(ASName.k_XHeight, new ASNumber(0));
		fd.put(ASName.k_StemV, new ASNumber(90));
		fd.put(ASName.k_StemH, new ASNumber(28));
		return fd;
	}
}

/**
 *  
 * PDFStandardFontDescriptors encapulates the font descriptors for
 * the standard 14 fonts.
 *
 * @author dpond
 */
public class PDFStandardFontDescriptors
{
	private PDFStandardFontDescriptors() {}

	/**
	 * Returns already created font descriptor dictionary corresponding to fonr name passed here. 
	 */
	public static ASDictionary get(ASName name)
		throws PDFInvalidParameterException
	{
		try
		{
			ASDictionary dict = descriptors.getDictionary(name);
			if(dict == null)
				throw new PDFInvalidParameterException("Font name does not correspond to a standard 14 font.");
			return dict;
		}
		catch (PDFUnableToCompleteOperationException e) {
			throw new PDFInvalidParameterException("Font name does not correspond to a standard 14 font.", e);
		}
	}

	private static ASDictionary create(ASName name)
	{
		return ((FDCreator)creators.get(name)).create(name);
	}

	private static HashMap creators = new HashMap();
	private static ASDictionary descriptors = new ASDictionary();

	static {
		creators.put(ASName.k_Helvetica, new Helvetica_FDCreator());
		creators.put(ASName.k_Helvetica_Oblique, new HelveticaOblique_FDCreator());
		creators.put(ASName.k_Helvetica_Bold, new HelveticaBold_FDCreator());
		creators.put(ASName.k_Helvetica_BoldOblique, new HelveticaBoldOblique_FDCreator());

		creators.put(ASName.k_Times_Roman, new TimesRoman_FDCreator());
		creators.put(ASName.k_Times_Italic, new TimesItalic_FDCreator());
		creators.put(ASName.k_Times_Bold, new TimesBold_FDCreator());
		creators.put(ASName.k_Times_BoldItalic, new TimesBoldItalic_FDCreator());

		creators.put(ASName.k_Courier, new Courier_FDCreator());
		creators.put(ASName.k_Courier_Oblique, new CourierOblique_FDCreator());
		creators.put(ASName.k_Courier_Bold, new CourierBold_FDCreator());
		creators.put(ASName.k_Courier_BoldOblique, new CourierBoldOblique_FDCreator());

		creators.put(ASName.k_Symbol, new Symbol_FDCreator());
		creators.put(ASName.k_ZapfDingbats, new ZapfDingbats_FDCreator());

		descriptors.put(ASName.k_Helvetica, create(ASName.k_Helvetica));
		descriptors.put(ASName.k_Helvetica_Oblique, create(ASName.k_Helvetica_Oblique));
		descriptors.put(ASName.k_Helvetica_Bold, create(ASName.k_Helvetica_Bold));
		descriptors.put(ASName.k_Helvetica_BoldOblique, create(ASName.k_Helvetica_BoldOblique));

		descriptors.put(ASName.k_Times_Roman, create(ASName.k_Times_Roman));
		descriptors.put(ASName.k_Times_Italic, create(ASName.k_Times_Italic));
		descriptors.put(ASName.k_Times_Bold, create(ASName.k_Times_Bold));
		descriptors.put(ASName.k_Times_BoldItalic, create(ASName.k_Times_BoldItalic));

		descriptors.put(ASName.k_Courier, create(ASName.k_Courier));
		descriptors.put(ASName.k_Courier_Oblique, create(ASName.k_Courier_Oblique));
		descriptors.put(ASName.k_Courier_Bold, create(ASName.k_Courier_Bold));
		descriptors.put(ASName.k_Courier_BoldOblique, create(ASName.k_Courier_BoldOblique));

		descriptors.put(ASName.k_Symbol, create(ASName.k_Symbol));
		descriptors.put(ASName.k_ZapfDingbats, create(ASName.k_ZapfDingbats));
	}
}
