/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.day.cq.commons.predicates.servlets;

import com.day.cq.replication.ReplicationStatus;
import org.apache.sling.api.SlingHttpServletRequest;

import java.text.Collator;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

public abstract class AbstractReplicableListServlet extends AbstractListServlet {
    /**
     * Value of the sort parameter to indicate to sort the results by
     * replication date.
     */
    public static final String SORT_REPLICATION = "replication";

    /**
     * {@inheritDoc}
     */
    protected List<AbstractListServlet.ListItem> applySorting(SlingHttpServletRequest request,
            List<AbstractListServlet.ListItem> items) {
        String sortKey = request.getParameter(SORT_KEY) != null ?
                request.getParameter(SORT_KEY) : DEFAULT_SORT_KEY;
        String sortDir = request.getParameter(SORT_DIR) != null ?
                request.getParameter(SORT_DIR) : DEFAULT_SORT_DIR;

        /* set collator strength */
        getCollator().setStrength(Collator.PRIMARY);

        // apply sorting
        if (!(DEFAULT_SORT_KEY.equals(sortKey) && DEFAULT_SORT_DIR.equals(sortDir))) {
            items.sort(new ListItemComparator(sortKey));
            if (SORT_DESCENDING.equals(sortDir)) {
                Collections.reverse(items);
            }
        }
        return items;
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings({"UnusedDeclaration"})
    public interface ListItem extends AbstractListServlet.ListItem {

        /**
         * Get the replication status of the item
         * @return ReplicationStatus replication status
         */
        ReplicationStatus getReplicationStatus();
    }

    /**
     * {@inheritDoc}
     */
    public class ListItemComparator implements Comparator<AbstractListServlet.ListItem> {

        private final String compareField;

        /**
         * {@inheritDoc}
         */
        public ListItemComparator(String compareField) {
            this.compareField = compareField;
        }

        /**
         * {@inheritDoc}
         */
        public int compare(AbstractListServlet.ListItem li1, AbstractListServlet.ListItem li2) {
            Object v1, v2;
            try {
                AbstractReplicableListServlet.ListItem o1 = (AbstractReplicableListServlet.ListItem) li1;
                AbstractReplicableListServlet.ListItem o2 = (AbstractReplicableListServlet.ListItem) li2;

                if (SORT_REPLICATION.equals(compareField)) {
                    Calendar c1 = o1.getReplicationStatus().getLastPublished();
                    Calendar c2 = o2.getReplicationStatus().getLastPublished();
                    v1 = c1 == null ? 0 : c1.getTimeInMillis();
                    v2 = c2 == null ? 0 : c2.getTimeInMillis();
                } else {
                    v1 = o1.getClass().getField(compareField).get(o1);
                    v2 = o2.getClass().getField(compareField).get(o2);
                }
                if (v1 == null && v2 == null) {
                    return 0;
                } else if (v1 == null) {
                    return -1;
                } else if (v2 == null) {
                    return 1;
                } else if (v1 instanceof String && v2 instanceof String) {
                    Collator collator = getCollator();
                    return (collator != null) ? collator.compare((String)v1, (String)v2) : ((String)v1).compareTo((String)v2);
                } else if (v1 instanceof Integer && v2 instanceof Integer) {
                    int int1 = (Integer)v1;
                    int int2 = (Integer)v2;
                    return (int1 > int2 ? 1 : int1 != int2 ? -1 : 0);
                } else if (v1 instanceof Long && v2 instanceof Long) {
                    long long1 = (Long)v1;
                    long long2 = (Long)v2;
                    return (long1 > long2 ? 1 : long1 != long2 ? -1 : 0);
                }
            } catch (Exception ignored) {
                // do nothing
            }
            return 0;
        }

    }
}
