/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 * 
 * Copyright 2024 Adobe
 * All Rights Reserved.
 * 
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.day.cq.commons.predicates.servlets;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.request.RequestPathInfo;
import org.apache.sling.api.resource.Resource;
import org.osgi.service.component.annotations.Component;

import javax.servlet.Servlet;
import javax.servlet.ServletException;
import java.io.IOException;
import java.util.function.Predicate;

/**
 * Forwards to the resource considering the resource search path ("/apps", "/libs", etc.).
 * If a suffix is provided a resource of the name of the requested resource will be searched
 * inside the paths provided by the suffix. If the suffix contains a period after the last
 * slash the part behind this period will be ignored.<br>
 * <br>
 * Sample without suffix:<br>
 * Request: /libs/x/y/z.aemOverlay.infinity.json<br>
 * The request will be forwarded to the first resource found in the following order
 * (taking "/apps" and "/libs" as given resource search paths):<br>
 * /apps/x/y/z.infinity.json<br>
 * /libs/x/y/z.infinity.json<br>
 * <br>
 * Sample with suffix:<br>
 * Request: /libs/x/y/z.aemOverlay.infinity.json/a/b.json<br>
 * The request will be forwarded to the first resource found in the following order
 * (taking "/apps" and "/libs" as given resource search paths):<br>
 * /apps/x/y/a/b/z.infinity.json<br>
 * /libs/x/y/a/b/z.infinity.json<br>
 * /apps/x/y/a/z.infinity.json<br>
 * /libs/x/y/a/z.infinity.json<br>
 * /apps/x/y/z.infinity.json<br>
 * /libs/x/y/z.infinity.json<br>
 */
@Component(
        service = { Servlet.class },
        property = {
                "sling.servlet.selectors=" + OverlayServlet.AEM_OVERLAY,
                "sling.servlet.methods=GET",
                "sling.servlet.resourceTypes=sling/servlet/default"
        }
)
public class OverlayServlet extends AbstractPredicateServlet {
    /**
     * selector value
     */
    public static final String AEM_OVERLAY = "aemOverlay";

    /**
     * {@inheritDoc}
     */
    @Override
    protected void doGet(SlingHttpServletRequest req,
            SlingHttpServletResponse resp,
            Predicate predicate)
            throws ServletException, IOException {

        // request: /libs/x/y/z.aemOverlay.json/a/b.json
        String path = req.getResource().getPath();

        // remove resource search path ("/libs" or "/apps")
        // path: x/y/z
        path = path.substring(path.indexOf("/", 1) + 1);

        Resource r = null;
        RequestPathInfo info = req.getRequestPathInfo();
        String suffix = info.getSuffix();

        if (suffix != null) {
            // name: z
            String name = path.substring(path.lastIndexOf("/") + 1);

            // basePath: x/y
            String basePath = path.substring(0, path.lastIndexOf("/"));

            // s: /a/b
            String s = suffix;
            if (s.lastIndexOf(".") > s.lastIndexOf("/")) {
                // assuming suffix has an extension: /a/b.json >> a/b
                s = s.substring(0, s.lastIndexOf("."));
            }
            while(s.indexOf("/") != -1) {
                r = req.getResourceResolver().getResource(basePath + s + "/" + name);
                if (r != null) {
                    break;
                }
                s = s.substring(0, s.lastIndexOf("/"));
            }
        }

        if (r == null) {
            // no suffix or final fallback
            r = req.getResourceResolver().getResource(path);
        }

        if (r != null) {
            String sel = info.getSelectorString();
            // remove "overlay" selector
            if (sel.equals(AEM_OVERLAY)) sel = "";
            else sel = "." + sel.substring(sel.indexOf(".") + 1);

            String ext = info.getExtension();
            if (ext == null) ext = "";
            else ext = "." + ext;

            String url = r.getPath() + sel + ext;
            req.getRequestDispatcher(url).forward(req, resp);
        }
    }

}
