/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.resource.details;

import com.day.cq.commons.LabeledResource;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.AssetReferenceResolver;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.commons.util.DamUtil;
import com.day.cq.dam.commons.util.UIHelper;
import com.day.cq.wcm.commons.ReferenceSearch;
import com.day.text.Text;
import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.commons.json.JSONObject;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * Class which gives access to some details of an asset
 */
public class AssetDetails implements ResourceDetails {

    final String ASSET_PROPERTY_METADATA = "jcr:content/metadata";
    final String ASSET_PROPERTY_COMMENTS = "jcr:content/comments";

    private Resource resource;
    private Asset asset;
    private Node assetNode;

    public AssetDetails(Resource resource) {
        if (resource == null) {
            throw new IllegalArgumentException("Resource may not be null!");
        }
        this.resource = resource;
        this.assetNode = resource.adaptTo(Node.class);
        this.asset = resource.adaptTo(Asset.class);
    }

    public Node getAssetNode() {
        return assetNode;
    }

    public Asset getAsset() {
        return asset;
    }

    /**
     * Get name
     */
    public String getName() throws RepositoryException {
        return asset.getName();
    }

    /**
     * Get last modified
     *
     * @return
     */
    public long getLastModified() throws RepositoryException {
        long assetLastModification = asset.getLastModified();
        ValueMap vm = resource.adaptTo(ValueMap.class);

        if (assetLastModification == 0) {
            Calendar created = vm.get("jcr:created", Calendar.class);
            assetLastModification = (null != created) ? created.getTimeInMillis() : 0;
        }
        return assetLastModification;
    }

    /**
     * Get mime type of asset
     *
     * @return mime type of asset
     */
    public String getMimeType() {
        return (asset.getMimeType() != null)
                ? asset.getMimeType()
                : null;
    }

    /**
     * Count all asset references
     *
     * @param customResolver
     * @return
     * @throws RepositoryException
     */
    public int getReferencesSize(AssetReferenceResolver customResolver) throws RepositoryException {
        int assetReferencesCount = 0;

        // search for references using the reference search
        Collection<ReferenceSearch.Info> resultSet = new ReferenceSearch().search(resource.getResourceResolver(),
                assetNode.getPath()).values();

        if (resultSet != null) {
            //check for references to feeds sling:resourceType = "mac/components/boardpage"
            Iterator<ReferenceSearch.Info> it = resultSet.iterator();
            while (it.hasNext()) {
                ReferenceSearch.Info infoItem = it.next();
                Resource contentRes = infoItem.getPage().getContentResource();
                //check if it is a mac board page
                if (contentRes != null && contentRes.getResourceType().equals("mac/components/boardpage")) {
                    it.remove();
                }

            }
            assetReferencesCount = resultSet.size();
        }

        // complex references
        if (customResolver != null) {
            Map<Resource, String> referencesInfo = new HashMap<Resource, String>();
            assetReferencesCount += customResolver.getReferences(assetNode.getPath(),
                    resource.getResourceResolver()).size();
        }
        return assetReferencesCount;
    }

    /**
     * Get number of comments
     *
     * @return number of comments
     * @throws javax.jcr.RepositoryException
     */
    public int getCommentsSize() throws RepositoryException {
        int commentsCount = 0;
        if (assetNode.hasNode(ASSET_PROPERTY_COMMENTS)) {
            Node commentsNode = assetNode.getNode(ASSET_PROPERTY_COMMENTS);
            for (Iterator<Node> it = commentsNode.getNodes(); it.hasNext(); it.next()) {
                commentsCount++;
            }
        }
        return commentsCount;
    }

    /**
     * Get URL for thumbnail
     *
     * @return
     */
    public String getThumbnailUrl() {
        //thumbnailUrl
        String thumbnailUrl = "";
        Rendition thumbnailRendition = UIHelper.getBestfitRendition(asset, 319);

        //handle case for sets (mixmedia, spinset, etc)
        if (getMimeType() != null && getMimeType().startsWith("Multipart/Related") && !getMimeType().contains("application/x-CarouselSet")) {
            thumbnailUrl = asset.getPath() + ".folderthumbnail.jpg";
        } else if (thumbnailRendition != null) {
            thumbnailUrl = thumbnailRendition.getPath();
        } else {
            //default thumbnail
            thumbnailUrl = asset.getPath() + ".thumb.319.319.png";
        }
        thumbnailUrl = Text.escapePath(thumbnailUrl);
        return thumbnailUrl;
    }

    /**
     * Get width of the asset
     *
     * @return width of asset
     * @throws javax.jcr.RepositoryException
     */
    public long getWidth() throws RepositoryException {
        long width = 0;
        if (assetNode.hasNode(ASSET_PROPERTY_METADATA)) {
            Node metadataNode = assetNode.getNode(ASSET_PROPERTY_METADATA);
            try {
                width = Long.valueOf(
                        DamUtil.getValue(metadataNode, "tiff:ImageWidth",
                                DamUtil.getValue(metadataNode, "exif:PixelXDimension", "")));
            } catch (Exception e) {
                // If this fails it's ok, we return 0 as fallback
            }
        }
        return width;
    }

    /**
     * Get height of the asset
     *
     * @return height of asset
     * @throws javax.jcr.RepositoryException
     */
    public long getHeight() throws RepositoryException {
        long height = 0;
        if (assetNode.hasNode("jcr:content/metadata")) {
            Node metadataNode = assetNode.getNode("jcr:content/metadata");
            try {
                height = Long.valueOf(
                        DamUtil.getValue(metadataNode, "tiff:ImageLength",
                                DamUtil.getValue(metadataNode, "exif:PixelYDimension", "")));
            } catch (Exception e) {
                // If this fails it's ok, we return 0 as fallback
            }
        }
        return height;
    }

    /**
     * Get size of asset
     *
     * @return size of asset
     */
    public String getSize() {
        return (asset.getOriginal() != null)
                ? UIHelper.getSizeLabel(asset.getOriginal().getSize())
                : "0.0 B";
    }

    /**
     * Get resolution of asset
     *
     * @return resolution of asset
     * @throws javax.jcr.RepositoryException
     */
    public String getResolution() throws RepositoryException {
        long width = getWidth();
        long height = getHeight();
        return (width != 0 && height != 0)
                ? width + " x " + height
                : "";
    }

    /**
     * Get description of the asset
     *
     * @return description
     */
    public String getDescription() throws RepositoryException {
        Asset asset = resource.adaptTo(Asset.class);
        if (asset != null) {
            Node metadataNode = assetNode.getNode("jcr:content/metadata");
            return DamUtil.getValue(metadataNode, "dc:description", resource.getName());
        } else {
            LabeledResource lr = resource.adaptTo(LabeledResource.class);
            if (lr != null) {
                return lr.getDescription() != null ? lr.getDescription() : "";
            }
        }

        return resource.getName();
    }

    public String getParamJSON() throws RepositoryException {
        JSONObject param = new JSONObject();

        if (assetNode.hasNode(ASSET_PROPERTY_METADATA)) {
            Node metadataNode = assetNode.getNode(ASSET_PROPERTY_METADATA);
            try {
                String imageMap = DamUtil.getValue(metadataNode, "imageMap", "");
                if (StringUtils.isEmpty(imageMap)) {
                    param.put("./imageMap@Delete", "");
                } else {
                    param.put("./imageMap", imageMap);
                }
                param.put("./imageCrop@Delete", "");
                param.put("./imageRotate@Delete", "");
            } catch (Exception ignored) {
            }
        }

        return param.toString();
    }
}