/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2021 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any.  The
 * intellectual and technical concepts contained herein are
 * proprietary to Adobe and its suppliers and are protected
 * by all applicable intellectual property laws, including
 * trade secret and copyright laws.  Dissemination of this
 * information or reproduction of this material is strictly
 * forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.adobe.aem.wcm.seo;

import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.osgi.annotation.versioning.ConsumerType;
import org.osgi.annotation.versioning.ProviderType;

/**
 * This model interface gives access to SEO specific information of a page.
 * <p>
 * It can be adapted from any {@link com.day.cq.wcm.api.Page}'s content that is of sling:resourceType
 * {@code wcm/foundation/components/basicpage/v1/basicpage}.
 * <p>
 * Usage: <pre>{@code page.getContentResource().adaptTo(SeoTags.class)}</pre>
 */
@ProviderType
public interface SeoTags {

    /**
     * The property name of the robots tags property.
     */
    String PN_ROBOTS_TAGS = "cq:robotsTags";

    /**
     * The property name of the custom canonical url property.
     */
    String PN_CANONICAL_URL = "cq:canonicalUrl";

    /**
     * Returns true when the current Resource belongs to a canonical page. This is the case when the page's canonical url points to itself,
     * meaning when no custom canonical url is set.
     * <p>
     * The default implementation returns true reflecting the assumption that, if not otherwise specified, each page is canonical.
     *
     * @return true when the current resource has no custom canonical url set
     */
    boolean isCanonical();

    /**
     * Returns the canonical url of the current Resource.
     * <p>
     * This uses the {@link org.apache.sling.sitemap.spi.common.SitemapLinkExternalizer} implementation to ensure the same canonical url as
     * it is used in Sitemap(s) is returned. The format of the url is complete, meaning it contains authority, path and extension.
     * <p>
     * This may return {@code null} when the externalization fails.
     *
     * @return The canonical, absolute url of the current page.
     */
    @Nullable
    String getCanonicalUrl();

    /**
     * Returns a {@link Map} of alternate language urls of the current Resource.
     * <p>
     * An alternate language link exists for each language copy of the {@link com.day.cq.wcm.api.Page}, when the
     * {@link com.day.cq.wcm.api.Page} is included in a Sitemap. It uses the
     * {@link org.apache.sling.sitemap.spi.common.SitemapLinkExternalizer} to ensure the same canonical urls as in Sitemap(s) are returned.
     * The format of the url is complete, meaning it contains authority, path and extension.
     *
     * @return The canonical, absolute urls of all alternate language versions of the current page.
     */
    @NotNull
    Map<Locale, String> getAlternateLanguages();

    /**
     * Returns a {@link List} of robots tags to be provided to search engine crawlers.
     * <p>
     * The collection is curated and may only return tags that are allowed to appear together.
     *
     * @return the curated {@link List} of robots tags
     */
    @NotNull
    List<String> getRobotsTags();

}
