/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2021 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.aem.wcm.seo.localization;

import java.util.Collection;
import java.util.Locale;
import java.util.Map;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.wcm.api.Page;

/**
 * A service that provides access to a {@link Page}'s language alternatives.
 */
@ProviderType
public interface LanguageAlternativesService {

    /**
     * Returns the site root as returned by the highest ranked {@link SiteRootSelectionStrategy} for the given {@link Page}.
     *
     * @param page the given {@link Page}
     * @return the site root, or {@code null} if not found
     */
    Page getSiteRoot(Page page);

    /**
     * Returns the language alternative of a given {@link Page} in the given {@link Locale}.
     * <p>
     * This is an optimised implementation of
     * <pre>{@code
     *  languageAlternativesService.getLanguageAlternative(page).get(locale);
     * } </pre>
     *
     * @param page   the {@link Page}
     * @param locale the {@link Locale}
     * @return a {@link Page} with the same relative path to the site root as the given {@link Page} but in the given {@link Locale}, or
     * {@code null} if it does not exist.
     */
    @Nullable
    Page getLanguageAlternative(@NotNull Page page, @NotNull Locale locale);

    /**
     * Traverses the site subtree of the given {@link Page} using depth-first-search up to a configured maximum depth and returns language
     * alternatives of the given page in each of the found language roots in the subtree. Language roots that are ancestors of other
     * language roots are skipped. For example {@code /content/wknd/fr/fr} is taking precedence over {@code /content/wknd/fr} for which
     * {@link Locale} derived from the path would be the same.
     * <p>
     * The {@link Page}'s site root is defined by an implementation of {@link SiteRootSelectionStrategy#getSiteRoot(Page)}.
     *
     * @param page the given {@link Page} to return language alternatives for
     * @return a {@link Collection} of all language alternatives within the {@link Page}'s site root
     */
    @NotNull
    Map<Locale, Page> getLanguageAlternatives(@NotNull Page page);

}
