/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2021 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.aem.wcm.seo.localization;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.osgi.annotation.versioning.ConsumerType;

import com.day.cq.wcm.api.Page;

/**
 * Consumers may implement this interface and register the implementation as an OSGI service to provide an algorithm that defines a site
 * root for a given page.
 * <p>
 * A site root is the {@link Page} that is the common ancestor of all {@link Page}s that belong to a single site. In a single-country
 * setup this is the parent of all language roots. For a multi-country setup it may be different, depending on the content structure.
 * <p>
 * A default implementation is available called DefaultSiteRootSelectionStrategy. To override it an implementation must be registered
 * with a service ranking greater then 0. Implementations may fall back to the DefaultSiteRootSelectionStrategy. If they are not able to
 * return a site root for a given {@link Page} the {@link LanguageAlternativesService} cannot return any language alternatives.
 */
@ConsumerType
public interface SiteRootSelectionStrategy {

    /**
     * Returns the common ancestor of all {@link Page}s with in the site the given {@link Page}.
     *
     * @param page a {@link Page} within a site
     * @return the root {@link Page} of the site
     */
    @Nullable
    Page getSiteRoot(@NotNull Page page);

    /**
     * Returns the depth of the content structure managed under the site root.
     * <p>
     * This defines the number of levels the {@link LanguageAlternativesService} traverses to collect language roots.
     * <p>
     * Examples:
     * <ul>
     * <li>With {@code /content/site} as site root and {@code /content/site/en} as language root the structural depth is {@code 1}
     * <li>With {@code /content/site} as site root, an intermediate level and a language root at {@code /content/site/us/en}, the
     * structural depth is {@code 2}
     * <li>With {@code /content/site} as site root, multiple intermediate levels and a language root at
     * {@code /content/site/emea/europe/germany/de} the structural depth is {@code 4}
     * </ul>
     * <p>
     *
     * @param page a {@link Page} within a site
     * @return an {@code int} greater or equal to one
     */
    int getStructuralDepth(@NotNull Page page);
}
