/*
 *******************************************************************************
 * Copyright (C) 2004-2007, International Business Machines Corporation and    *
 * others. All Rights Reserved.                                                *
 *******************************************************************************
 *
 * Created on Feb 4, 2004
 *
 */

/*
 * File: ICUData.java
 * ************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.agl.impl;

import java.io.InputStream;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.MissingResourceException;

/**
 * Provides access to ICU data files as InputStreams.  Implements security checking.
 */
public final class ICUData {
    /*
     * Return a URL to the ICU resource names resourceName.  The
     * resource name should either be an absolute path, or a path relative to
     * com.adobe.agl.impl (e.g., most likely it is 'data/foo').  If required
     * is true, throw an MissingResourceException instead of returning a null result.
     */
    public static boolean exists(final String resourceName) {
        URL i = null;
        if (System.getSecurityManager() != null) {
            i = (URL)AccessController.doPrivileged(new PrivilegedAction() {
                    public Object run() {
                        return ICUData.class.getResource(resourceName);
                    }
                });
        } else {
            i = ICUData.class.getResource(resourceName);
        }
        return i != null;
    }
        
    private static InputStream getStream(final Class root, final String resourceName, boolean required) {
        InputStream i = null;
        
        if (System.getSecurityManager() != null) {
            i = (InputStream)AccessController.doPrivileged(new PrivilegedAction() {
                    public Object run() {
                        return root.getResourceAsStream(resourceName);
                    }
                });
        } else {
            i = root.getResourceAsStream(resourceName);
        }

        if (i == null && required) {
            throw new MissingResourceException("could not locate data " +resourceName, root.getPackage().getName(), resourceName);
        }
        return i;
    }

    private static InputStream getStream(final ClassLoader loader, final String resourceName, boolean required) {
        InputStream i = null;
        if (System.getSecurityManager() != null) {
            i = (InputStream)AccessController.doPrivileged(new PrivilegedAction() {
                    public Object run() {
                        return loader.getResourceAsStream(resourceName);
                    }
                });
        } else {
            i = loader.getResourceAsStream(resourceName);
        }
        if (i == null && required) {
            throw new MissingResourceException("could not locate data", loader.toString(), resourceName);
        }
        return i;
    }
    
    public static InputStream getStream(ClassLoader loader, String resourceName){
        return getStream(loader,resourceName, false);   
    }

    public static InputStream getRequiredStream(ClassLoader loader, String resourceName){
        return getStream(loader, resourceName, true);
    }

    /*
     * Convenience override that calls getStream(ICUData.class, resourceName, false);
     */
    public static InputStream getStream(String resourceName) {
        return getStream(ICUData.class, resourceName, false);
    }
        
    /*
     * Convenience method that calls getStream(ICUData.class, resourceName, true).
     */
    public static InputStream getRequiredStream(String resourceName) {
        return getStream(ICUData.class, resourceName, true);
    }

    /*
     * Convenience override that calls getStream(root, resourceName, false);
     */
    public static InputStream getStream(Class root, String resourceName) {
        return getStream(root, resourceName, false);
    }
    
    /*
     * Convenience method that calls getStream(root, resourceName, true).
     */
    public static InputStream getRequiredStream(Class root, String resourceName) {
        return getStream(root, resourceName, true);
    }
}

