/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ************************************************************************/

package com.adobe.cq.account.api;

import java.util.Map;

import javax.jcr.RepositoryException;

import aQute.bnd.annotation.ProviderType;
import org.apache.sling.api.request.RequestParameter;

/**
 * The {@code AccountManagementService} provides a way for a visitor (a non logged-in user) to request a new account
 * or to request a password reset.
 * For any other account management task (e.g. modifying a user property) please use the
 * {@code org.apache.jackrabbit.api.security.user.UserManager}, based on the logged-in user session.
 *
 * The default implementation works as follows:
 * - when a new account is requested, the account is created in the repository and the account is disabled. An email
 * is sent to the user together with a link to confirm the operation. When the link is clicked, the request
 * is validated and if valid the account is enabled.
 * - when a password reset is requested, an email is sent to the user together with a link to confirm the operation.
 * When the link is clicked, , the request is validated and if valid the password is reset and an email is sent to
 * the user with the new password.
 *
 * Other implementations could look as follows:
 * - when a new account is requested, an email is sent to the user together with a link to confirm the operation. The
 * link contains an encrypted token with all the information about the account. When the link is clicked, the account
 * is created.
 * - when a new account is requested, a workflow is triggered. The workflow takes care of validating and creating
 * the new account.
 *
 * @since 6.2
 */
@ProviderType
public interface AccountManagementService {

    /**
     * Requests the creation of a new account.
     *
     * @param userId the user id for the new account
     * @param pwd the user password
     * @param properties the user properties
     * @param requestUrl the URL used to request a new account (used to validate the requester)
     * @param configPath the path where the configuration is defined (e.g. the url used to confirm the operation,
     *                          the group of the user, the intermediate path where the user is created)
     * @return  true if the account request succeeds, false otherwise
     * @throws javax.jcr.RepositoryException in case of error when requesting a new account
     */
    public boolean requestAccount(String userId, String pwd, Map<String, RequestParameter[]> properties,
                                  String requestUrl, String configPath) throws RepositoryException;
    /**
     * Requests a password reset for the given user.
     *
     * @param userId the user id
     * @param requestUrl the URL used to request the password reset (used to validate the requester)
     * @param configPath the path where the configuration is defined (e.g. the url to confirm the operation)
     * @return  true if the password reset request succeeds, false otherwise
     * @throws javax.jcr.RepositoryException in case of error when requesting the password reset
     */
    public boolean requestPasswordReset(String userId, String requestUrl, String configPath) throws RepositoryException;
}

