/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2016 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package com.adobe.cq.dam.cfm.converter;

import aQute.bnd.annotation.ConsumerType;
import com.adobe.cq.dam.cfm.ContentFragmentException;

/**
 * <p>Provides a format adapter.</p>
 *
 * <p>Implementations of this interface are used to convert a certain text format from and
 * to HTML.</p>
 *
 * <p>By contract, the conversion to HTML must be as precise as possible, as it is used
 * for render the fragment. I.e., the HTML created by {@link #toHTML(String)} is
 * eventually displayed to the end user.</p>
 *
 * <p>While it is recommended to provide an "as precise as possible" conversion from HTML
 * to the target format as well, it is not absolutely required: {@link #fromHTML(String)}
 * is only used for manual format changes where it is assumed that the author has to tweak
 * the results manually anyway.</p>
 */
@ConsumerType
public interface ContentTypeAdapter {

    /**
     * Returns the MIME type the adapter is responsible for.
     *
     * @return The content type
     */
    String getContentType();

    /**
     * Converts the given HTML text into the represented text format.
     *
     * @param html The HTML text to convert
     * @return The provided HTML, converted into the adapter's text format
     * @throws ContentFragmentException if the conversion fails
     */
    String fromHTML(String html) throws ContentFragmentException;

    /**
     * <p>Converts the given text that is assumed to be in the adapter's format into a
     * valid HTML fragment.</p>
     *
     * <p>The returned HTML fragment must not have a root tag (like &lt;html&gt;,
     * &lt;body&gt; or &lt;div&gt;).</p>
     *
     * <p>Note that "is assumed to be in the transformer's format" means that you can't
     * rely on the input being syntactically correct (unless the editor implementation
     * ensures it in the UI). Hence the implementation should handle syntactical errors
     * gracefully. In a worst case scenario, the original text should be treated as plain
     * text.</p>
     *
     * @param formattedText The text to be converted to HTML; supposed to be in the
     *                      adapter's format
     * @return The converted text
     * @throws ContentFragmentException if the conversion fails
     */
    String toHTML(String formattedText) throws ContentFragmentException;

}
