/*
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */

package com.adobe.cq.dam.download.spi;

import java.util.Collection;
import java.util.Map;

import org.apache.sling.api.resource.ResourceResolver;
import org.osgi.annotation.versioning.ConsumerType;

import com.adobe.cq.dam.download.api.DownloadException;
import com.adobe.cq.dam.download.api.DownloadFile;
import com.adobe.cq.dam.download.api.DownloadService;
import com.adobe.cq.dam.download.api.DownloadTarget;

/**
 * <p>
 * Facilitates the conversion of a {@link DownloadTarget} to one or more binary
 * URIs.
 * </p>
 * <p>
 * This interface should be implemented to create a handler for a given type
 * of {@link DownloadTarget}. Any targets whose type match the processor's
 * type will go through the processor implementation when the time comes to
 * translate the target into the binary URIs to be included in a download
 * provided by the {@link DownloadService}.
 * </p>
 * <p>
 * The concept of the processor provides a place where consumers can customize
 * the types of targets that can be included in the download process. See
 * {@link DownloadService} documentation for examples.
 * </p>
 * @see DownloadService
 * @see DownloadTarget
 * @see DownloadFile
 * @see ResourceResolver
 * @see DownloadException
 */
@ConsumerType
public interface DownloadTargetProcessor {
    /**
     * <p>
     * Does the work of converting a target into a files that can be included in a download.
     * </p>
     * <p>
     * When this method is called, it's assumed that the processors parameters have been validated based on
     * its {@link #getValidParameters()} method.
     * </p>
     * @param target Item whose parameters can be used by the processor to retrieve binary information.
     * @param resourceResolver Can be used by the processor to look up sling-related information for the target.
     * @return List of files for download.
     * @throws DownloadException Thrown if there are issues during the process of processing a target.
     */
    Collection<DownloadFile> processTarget(DownloadTarget target, ResourceResolver resourceResolver) throws DownloadException;

    /**
     * The target type that the processor handles. {@link DownloadTarget} items whose <code>type</code> matches this
     * value will be passed through the processor's {@link #processTarget(DownloadTarget, ResourceResolver)} method.
     * @return The type of targets that will be handled by the processor implementation.
     */
    String getTargetType();

    /**
     * Retrieves a map indicating which parameters the processor supports, and whether or not each
     * each parameter is required.
     * @return A map whose keys are the names of parameters, and whose values indicate if the parameter
     *  is required.
     */
    Map<String, Boolean> getValidParameters();
}
