/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.inbox.ui;

import java.util.Date;
import java.util.Map;
import java.util.Set;

import aQute.bnd.annotation.ProviderType;

/**
 * Represents an inbox item of either type task or workitem.
 */
@ProviderType
public interface InboxItem {

    /**
     * Returns the item title.
     * 
     * @return item title
     */
    String getTitle();

    /**
     * Returns the item description.
     * 
     * @return item description 
     */
    String getDescription();

    /**
     * Returns the item path.
     * 
     * @return item path
     */
    String getPath();

    /**
     * Returns the item resource type.
     * 
     * @return item resource type
     */
    String getResourceType();

    /**
     * Returns a Map containing information about the current assignee.
     * 
     * @return a map of key/value pairs or {@code null}
     */
    Map<String, String> getCurrentAssigneeInfo();

    /**
     * Returns the payload link.
     * 
     * @return payload link or {@code null}
     */
    String getPayloadLink();

    /**
     * Returns the payload path in the repository.
     *
     * @return payload path or {@code null}
     */
    String getPayloadPath();

    /**
     * Returns the payload thumbnail path. If the payload has no payload thumbnail,
     * {@code null} is returned.
     * 
     * @return payload thumbnail path or {@code null}
     */
    String getPayloadThumbnail();

    /**
     * Indicates if referenced payload is of type folder.
     * 
     * @return {@code true} if payload is of type folder, {@code false} otherwise
     */
    boolean isPayloadFolder();

    /**
     * Returns the thumbnail path. If no project is associated with this task {@code null} is returned.
     * 
     * @return thumbnail path or {@code null}
     */
    String getThumbnail();

    /**
     * Returns the task priority. Either 'High', 'Medium' or 'Low'.
     * 
     * @return task priority or {@code null}
     */
    String getPriority();

    /**
     * Returns the task priority icon identifier. Either 'arrowUp' or 'arrowDown'.
     * 
     * @return task priority icon identifier or {@code null}
     */
    String getPriorityIcon();

    /**
     * Returns the task priority style class name. Either
     * 'inbox-item-priority--high', 'inbox-item-priority--normal' or
     * 'inbox-item-priority--low'.
     * 
     * @return task priority style class name
     */
    String getPriorityStyle();

    /**
     * Returns the task status. Either 'Active', 'Complete' or 'Terminated'.
     * 
     * @return task status
     */
    String getStatus();

    /**
     * Returns the task status icon. Either 'clock', 'checkCircle' or 'alert'.
     * 
     * @return task status icon name
     */
    String getStatusIcon();

    /**
     * Returns the task status type. Either 'inbox-status-active' or 'inbox-status-complete'.
     * 
     * @return task status type or an empty String
     */
    String getStatusType();

    /**
     * Returns the task details URL.
     * 
     * @return task details URL
     */
    String getDetailsUrl();

    /**
     * Returns the task project identifier if task is related to a project.
     * 
     * @return project identifier or {@code null}
     */
    String getProjectId();

    /**
     * Returns the task project title if task is related to a project.
     * 
     * @return task project title or {@code null}
     */
    String getProjectTitle();

    /**
     * Returns the task project URL if task is related to a project.
     * 
     * @return task project URL or {@code null}
     */
    String getAssociatedProjectDetailsUrl();

    /**
     * Returns the workflow title if task is related to a workflow.
     * 
     * @return task workflow title or {@code null}
     */
    String getWorkflowTitle();

    /**
     * Returns the workflow initiator identifier if task is related to a workflow.
     * 
     * @return workflow initiator identifier or {@code null}
     */
    String getWorkflowInitiator();

    /**
     * Returns the task start date.
     * 
     * @return task start date
     */
    Date getStartDate();

    /**
     * Returns the task due date.
     * 
     * @return task due date
     */
    Date getDueDate();

    /**
     * Returns the due date color name. Either 'red' or 'orange'.
     * 
     * @return due date color name or {@code null}
     */
    String getDueDateColor();

    /**
     * Returns the quickaction relation identifiers.
     * 
     * @return quickaction relation itendifiers or an empty Set.
     */
    Set<String> getQuickactionsRel();

    /**
     * Returns the task actions as serialized JSONArray in the following form.
     * 
     * <p>
     *  <code>
     *  [{actionName:'Action', actionId:'ID'},...]
     *  </code>
     * </p>
     * 
     * @return task actions
     */
    String getTaskActions();


    /**
     * Returns whether this inbox item is shared or not.
     * Sharing may be explicit or via inbox sharing. i.e. it will return true if
     * - this inbox item has been explicitly shared
     * - or the assignee of the inbox item is sharing his inbox
     * @return whether the inbox item is shared
     */
    boolean isShared();

    /**
     * Returns whether item is locked or not
     * This is applicable for items which are shared explicitly or as inbox items
     * It will return true when a item is locked by original assignee or any other user
     * @return whether the inbox item is locked
     */
    boolean isLocked();

    /**
     * If the inbox item is created as part of a workflow, provides the associated workflow metadata.
     * Returns null otherwise.
     * @return The associated workflow metadata
     */
    Map getWorkflowMetadata();
}
