/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
/**
 * AdobePatentID="3435US01"
 */
package com.adobe.cq.launches.api;

import java.util.Calendar;
import java.util.List;

import org.apache.sling.api.resource.Resource;

/**
 * Represents a Launch (Future versions of resources).
 */
public interface Launch extends Comparable<Launch> {

    /**
     * Returns the launch resource
     *
     * @return Launch resource
     */
    Resource getResource();

    /**
     * Returns the launch root resource (in the launch tree)
     *
     * @return Launch root resource
     */
    Resource getRootResource();

    /**
     * Returns the list of Launch Sources (in the launch tree)
     *
     * @return LaunchSource List
     */
    List<LaunchSource> getLaunchSources();
    
    /**
     * Returns the root resource the launch belongs to (in the production
     * tree)
     *
     * @return Launch source root resource
     */
    @Deprecated
    Resource getSourceRootResource();

    /**
     * Returns the title of the Launch
     *
     * @return Launch title
     */
    String getTitle();

    /**
     * Returns the targeted live date for the launch
     *
     * @return Launch live date
     */
    Calendar getLiveDate();

    /**
     * Returns <code>true</code> if the current <code>Launch</code> is
     * production ready
     * If a target live date is set and this flag is set to <code>true</code>,
     * the launch will automatically be promoted and activated at this date
     *
     * @return <code>true</code> if the launch is production ready
     */
    boolean isProductionReady();

    /**
     * Returns <code>true</code> if the current <code>Launch</code> is a live
     * copy
     *
     * @return <code>true</code> if the launch is a live copy
     */
    boolean isLiveCopy();

    /**
     * Returns <code>true</code> if the current <code>Launch</code> has been
     * created with inclusion of sub pages
     *
     * @return <code>true</code> if the launch is deep
     */
    @Deprecated
    boolean isDeep();

    /**
     * This method returns the date and time this launch was created
     *
     * @return A {@link Calendar} representing the creation date and time
     */
    Calendar getCreated();

    /**
     * Returns the id of the user that added this launch
     *
     * @return The identifier of the user who added this launch
     */
    String getCreatedBy();

    /**
     * Returns the date the definition of this launch was last modified
     *
     * @return The date of last modification or the value of
     *         {@link #getCreated()} if never modified.
     */
    Calendar getModified();

    /**
     * Returns the id of the user that effected the last modification of this
     * launch
     *
     * @return The identifier of the user who last modified this launch or the
     *         value of {@link #getCreatedBy()} if never modified
     */
    String getModifiedBy();

    /**
     * Returns the date the launch was last promoted
     *
     * @return The date of last promotion or <code>null</code> if never
     *         promoted.
     */
    Calendar getLastPromoted();

    /**
     * Returns the id of the user that effected the last promotion of this
     * launch
     *
     * @return The identifier of the user who last promoted this launch or
     *         <code>null</code> if never promoted
     */
    String getLastPromotedBy();

    /**
     * Returns <code>true</code> if the launch contains the provided production
     * resource
     *
     * @param productionResource Production resource
     * @return <code>true</code> if the launch contains the provided production
     *         resource
     */
    boolean containsResource(Resource productionResource);

    /**
     * Compares the current launch with another one for sorting.
     * The current launch is supposed to be listed before the provided launch:
     * <ul>
     * <li>if the latest is null</li>
     * <li>if its live date comparison returns a negative value</li>
     * <li>if its creation date comparison returns a negative value</li>
     * <li>if its path comparison returns a negative value</li>
     * </ul>
     *
     * @param launch The launch which has to be compared with the current one
     * @return a negative integer, zero, or a positive integer as this launch
     *         is less than, equal to, or greater than the specified one.
     * @see Comparable#compareTo(Object)
     */
    public int compareTo(Launch launch);

    /**
     *
     * Returns the <code>LaunchPromotionScope</code> of the launch
     *
     * @return <code>LaunchPromotionScope</code> of the launch if exists. Otherwise null is returned.
     */
    public LaunchPromotionScope getLaunchAutoPromotionScope();

}
