/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.adobe.cq.remote.content.renderer;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.sling.api.SlingHttpServletRequest;

import com.adobe.cq.export.json.ComponentExporter;

/**
 * Interface for OSGi service implementations capable of processing requests sent to a remote content rendering endpoint
 */
public interface RemoteContentRendererRequestHandler {

    /**
     * Should the current {@link RemoteContentRendererRequestHandler} implementation be used to process the request
     * @param servletRequest The AEM internal servlet request
     * @return true if the Handler can process the request
     */
    boolean canHandle(@Nonnull SlingHttpServletRequest servletRequest);

    /**
     * Returns the configuration that will be used to execute the remote request
     * @param servletRequest The AEM internal servlet request
     * @return The {@link RequestConfig} configuration object
     */
    @Nullable
    RequestConfig getRequestConfig(@Nonnull SlingHttpServletRequest servletRequest);

    /**
     * Returns the request object that can be executed by a {@link HttpClient} implementation
     * @param servletRequest The AEM internal servlet request
     * @param componentExporter The data model to be serialized and sent to the remote endpoint
     * @return An implementation of {@link HttpUriRequest}
     * @throws RemoteContentRenderingException exception due to an invalid servlet request
     */
    @Nonnull
    HttpUriRequest getRequest(@Nonnull SlingHttpServletRequest servletRequest, @Nonnull ComponentExporter componentExporter) throws RemoteContentRenderingException;

    /**
     * Returns the potentially processed response from the remote content rendering endpoint
     * @param httpResponse Response to the request sent to the remote endpoint
     * @return The processed response text to be injected in servlets SlingHttpServletResponse
     * @throws IOException exception when reading from the {@link CloseableHttpResponse}
     */
    @Nonnull
    String getResponseMessage(@Nonnull CloseableHttpResponse httpResponse) throws IOException;

    /**
     * Returns the list of classes that the Sling Model Exporter should try to adapt the request to
     * @return The list of classes
     */
    @Nonnull
    default Class<? extends ComponentExporter>[] getSlingModelAdapterClasses() {
        return new Class[0];
    }
}
