/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2017 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/
package com.adobe.cq.sites.ui.designfield.datasources;

import com.adobe.granite.ui.components.Config;
import com.adobe.granite.ui.components.ExpressionHelper;
import com.adobe.granite.ui.components.ExpressionResolver;
import com.adobe.granite.ui.components.PagingIterator;
import com.adobe.granite.ui.components.ds.AbstractDataSource;
import com.adobe.granite.ui.components.ds.DataSource;
import com.adobe.granite.ui.components.ds.EmptyDataSource;
import com.day.cq.wcm.api.NameConstants;
import org.apache.commons.collections.IteratorUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.servlets.SlingSafeMethodsServlet;
import org.apache.sling.jcr.resource.JcrResourceConstants;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

import javax.servlet.Servlet;
import java.util.*;
import java.util.regex.Pattern;

@Component(
        service = {Servlet.class},
        property = {
                "sling.servlet.resourceTypes=wcm/designer/gui/components/designfield/datasources/children",
                "sling.servlet.methods=GET"
        }
)
public class Children extends SlingSafeMethodsServlet{

    @Reference
    ExpressionResolver expressionResolver;

    @Override
    public void doGet(SlingHttpServletRequest request, SlingHttpServletResponse response) {
        final String DEFAULT_ROOT = "/";
        final String LEGACY_DESIGNS_LOCATION = "/etc/designs";
        final String DEFAULT_DESIGNS_LOCATION = "/libs/settings/wcm/designs";
        final String USER_DESIGNS_LOCATIONS = "/apps/settings/wcm/designs";
        final String LEGACY_DESIGNS_ROOT = "/etc";
        final String DEFAULT_DESIGNS_ROOT = "/libs";
        final String USER_DESIGNS_ROOT = "/apps";

        final ResourceResolver resourceResolver = request.getResourceResolver();
        final ExpressionHelper ex = new ExpressionHelper(expressionResolver, request);
        final Config dsCfg = new Config(request.getResource().getChild(Config.DATASOURCE));
        final Map<String, String> rootMapping = new HashMap<String, String>();
        final List<Resource> result = new ArrayList<Resource>();

        DataSource ds;
        String parentPath = null;
        String searchName = null;

        if (resourceResolver.getResource(LEGACY_DESIGNS_LOCATION) != null) {
            rootMapping.put(LEGACY_DESIGNS_ROOT, LEGACY_DESIGNS_LOCATION);
        }
        rootMapping.put(DEFAULT_DESIGNS_ROOT, DEFAULT_DESIGNS_LOCATION);
        rootMapping.put(USER_DESIGNS_ROOT, USER_DESIGNS_LOCATIONS);

        final String query = ex.getString(dsCfg.get("query", String.class));
        if (query != null) {
            final int slashIndex = query.lastIndexOf('/');
            if (slashIndex < 0 && !StringUtils.isBlank(query)) {    // If query is not a path, then run the query for designs under each of the design location.
                rootMapping.values()
                        .stream()
                        .map(s -> resourceResolver.getResource(s))
                        .forEach(r -> result.addAll(IteratorUtils.toList(r.listChildren())));
                searchName = query;
            } else if (rootMapping.values().stream().anyMatch(s -> query.startsWith(s))) { // If query starts with any of the designs location, then show all possible
                parentPath = query.substring(0, slashIndex + 1);                           // values of path starting with query.
                searchName = query.substring(slashIndex + 1);
            } else {                                                // If query doesn't start with any of the design location, then run query only on the path the design locations.
                rootMapping.values()
                        .stream()
                        .filter(s -> s.startsWith(query))
                        .forEach(s -> result.add(resourceResolver.getResource(s)));
            }
        } else {
            parentPath = ex.getString(dsCfg.get("path", String.class));
        }

        if (DEFAULT_ROOT.equals(parentPath)) {
            rootMapping.keySet()
                    .stream()
                    .forEach(path -> result.add(resourceResolver.getResource(path)));
        } else if (parentPath != null) {
            if (rootMapping.containsKey(parentPath)) {
                parentPath = rootMapping.get(parentPath);
            }
            Resource parent = resourceResolver.getResource(parentPath);
            Iterator<Resource> childIt = parent.listChildren();
            while(childIt.hasNext()) {
                result.add(childIt.next());
            };
        }

        if (result.isEmpty()) {
            ds = EmptyDataSource.instance();
        } else {
            final Integer offset = ex.get(dsCfg.get("offset", String.class), Integer.class);
            final Integer limit = ex.get(dsCfg.get("limit", String.class), Integer.class);
            result.removeIf((resource) ->
                    !(resource.isResourceType(NameConstants.NT_PAGE)) &&
                    !(resource.isResourceType(JcrConstants.NT_FOLDER)) &&
                    !(resource.isResourceType(JcrResourceConstants.NT_SLING_FOLDER)) &&
                    !(resource.isResourceType(JcrResourceConstants.NT_SLING_ORDERED_FOLDER))
            );
            if (!StringUtils.isBlank(searchName)) {
                final Pattern searchNamePattern = Pattern.compile(Pattern.quote(searchName), Pattern.CASE_INSENSITIVE);
                result.removeIf( (resource) -> !(searchNamePattern.matcher(resource.getName()).lookingAt()));
            }
            Collections.sort(result,
                    (Resource o1,Resource o2) ->  o1.getName().compareTo(o2.getName()));
            @SuppressWarnings("unchecked")
            DataSource datasource = new AbstractDataSource() {
                @Override
                public Iterator<Resource> iterator() {
                    return new PagingIterator<Resource>(result.iterator(), offset, limit);
                }
            };

            ds = datasource;
        }

        request.setAttribute(DataSource.class.getName(), ds);
    }
}
