/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/
package com.adobe.cq.sites.ui.models.admin.security.permission;

import java.io.StringWriter;
import java.security.Principal;
import java.util.Collection;
import java.util.Collections;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.security.AccessControlEntry;
import javax.jcr.security.AccessControlList;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;

import com.adobe.granite.security.user.util.AuthorizableUtil;
import com.day.cq.security.util.CqActions;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.api.JackrabbitSession;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlEntry;
import org.apache.jackrabbit.api.security.principal.PrincipalManager;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.jackrabbit.api.security.user.UserManager;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.io.JSONWriter;
import org.apache.sling.models.annotations.Model;
import org.apache.sling.models.annotations.injectorspecific.Self;
import org.apache.sling.models.annotations.injectorspecific.SlingObject;
import org.apache.sling.models.annotations.injectorspecific.ValueMapValue;

/**
 * Representation of a {@link java.security.Principal} and its allowed actions. This class is suitable for administering the permissions of {@link java.security.Principal}s
 *
 * The permissions are associated with the resource provided via the item request parameter. We are not displaying the data of the current resource but from a given resource
 *
 * It also provides a Json representation of the current permissions object
 */
@Model(adaptables = SlingHttpServletRequest.class)
public final class Permissions {

    @Self
    private SlingHttpServletRequest slingRequest;

    @ValueMapValue
    private String principalName;

    @ValueMapValue
    private String avatar;

    @Inject
    @SlingObject
    private Resource resource;

    @Inject
    private ResourceResolver resolver;

    private String label;

    private String path;

    private boolean aclEdit;

    private boolean read;

    private boolean modify;

    private boolean delete;

    private boolean replicate;

    private boolean create;

    private CqActions cqActions;

    @PostConstruct
    protected void initModel() throws RepositoryException {
        Session session = resolver.adaptTo(Session.class);
        PrincipalManager principalManager = ((JackrabbitSession) session).getPrincipalManager();
        AccessControlManager accessControlManager = session.getAccessControlManager();
        UserManager userManager = ((JackrabbitSession) session).getUserManager();

        path = resource.getPath();

        if (StringUtils.isEmpty(principalName)) {
            return;
        }

        Principal principal = principalManager.getPrincipal(principalName);
        Authorizable authorizable = userManager.getAuthorizable(principal);
        if (authorizable != null) {
            label = AuthorizableUtil.getFormattedName(resolver, authorizable.getID());
            if (StringUtils.isEmpty(label)) {
                label = principalName;
            }
        } else {
            label = principalName;
        }

        String resourcePath = slingRequest.getParameter("item");
        if (resourcePath == null) resourcePath = slingRequest.getRequestPathInfo().getSuffix();

        if (StringUtils.isNotEmpty(resourcePath)) {
            // Does the user has acl read access
            if(((JackrabbitSession) session).hasPermission(resourcePath, JackrabbitSession.ACTION_READ_ACCESS_CONTROL)) {
                cqActions = new CqActions(session);

                Collection<String> allowedActions = getActions(accessControlManager, resourcePath, principalName, cqActions);

                if (allowedActions != null) {
                    read = allowedActions.contains("read");
                    modify = allowedActions.contains("modify");
                    delete = allowedActions.contains("delete");
                    replicate = allowedActions.contains("replicate");
                    create = allowedActions.contains("create");
                }

                Collection<String> principalAllowedActions = cqActions.getAllowedActions(resourcePath,null);

                aclEdit = principalAllowedActions.contains("acl_edit");
            }
        }
    }

    /**
     * Returns the available actions for the given resource path and name of {@link java.security.Principal}
     *
     * @param accessControlManager      - The Access Control Manager
     * @param resourcePath              - Path of the resource
     * @param principalName             - Name of the {@link java.security.Principal}
     * @param cqActions                 - Accessible actions
     * @return
     * @throws RepositoryException
     */
    private Collection<String> getActions (AccessControlManager accessControlManager, String resourcePath, String principalName, CqActions cqActions) throws RepositoryException {
        AccessControlPolicy[] locals = accessControlManager.getPolicies(resourcePath);

        for (AccessControlPolicy policy : locals) {
            if (!(policy instanceof AccessControlList)) {
                continue;
            }

            for (AccessControlEntry ace : ((AccessControlList) policy).getAccessControlEntries()) {
                if (!(ace instanceof JackrabbitAccessControlEntry)) {
                    continue;
                }

                if (ace.getPrincipal() != null && principalName.equals(ace.getPrincipal().getName())) {
                   return cqActions.getAllowedActions(resourcePath,  Collections.singleton(ace.getPrincipal()));
                }
            }
        }

        return null;
    }

    /**
     * Returns the current acl data as a Json String
     *
     * @return
     * @throws JSONException
     */
    public String getAclData () throws JSONException {
        StringWriter sw = new StringWriter();
        JSONWriter jw = new JSONWriter(sw);

        jw.object();
        jw.key("principalName").value(principalName);
        jw.key("label").value(label);
        jw.key("avatar").value(avatar);
        jw.key("read").value(read);
        jw.key("modify").value(modify);
        jw.key("delete").value(delete);
        jw.key("replicate").value(replicate);
        jw.key("create").value(create);
        jw.endObject();

        return sw.toString();
    }

    /**
     * Returns the path of the component
     *
     * @return
     */
    public String getPath() {
        return path;
    }

    /**
     * Returns the human readable label for the {@link java.security.Principal} via its {@link Authorizable} - if available
     *
     * @return
     */
    public String getLabel() {
        return label;
    }

    /**
     * Does the actual user have rights to edit the given resource
     *
     * @return
     */
    public boolean canEditAcl() {
        return aclEdit;
    }

    /**
     * Can the {@link java.security.Principal} read the given resource
     *
     * @return
     */
    public boolean canRead() {
        return read;
    }

    /**
     * Can the {@link java.security.Principal} modify the given resource
     *
     * @return
     */
    public boolean canModify() {
        return modify;
    }

    /**
     * Can the {@link java.security.Principal} delete the given resource
     *
     * @return
     */
    public boolean canDelete() {
        return delete;
    }

    /**
     * Can the {@link java.security.Principal} replicate the given resource
     *
     * @return
     */
    public boolean canReplicate() {
        return replicate;
    }

    /**
     * Can the {@link java.security.Principal} create child resources of the given resource
     *
     * @return
     */
    public boolean canCreate() {
        return create;
    }
}

