/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2016 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package com.adobe.cq.smartcontent;

import org.osgi.annotation.versioning.ProviderType;

import java.util.Locale;

/**
 * <p>This class implements a service that can be used to apply smart content
 * authoring to a text.</p>
 *
 * <h3>Language/locale handling:</h3>
 *
 * <ol>
 *   <li>An exact match (language + country) is preferred.</li>
 *   <li>If no exact match is found, language data available for the language is chosen.
 *   </li>
 *   <li>If there are neither matches for language and region, nor for language (without
 *   region), a {@link SmartContentException} is thrown.</li>
 * </ol>
 */
@ProviderType
public interface SmartContentAuthoringService {

    /**
     * <p>Determines the language data for the provided {@code Locale}.</p>
     *
     * <p>The method uses the resolution rules stated in the interface's documentation
     * to match over-specific {@code Locale}s to suitable language data.</p>
     *
     * @param locale The locale
     * @return The language data; {@code null} if the specified {@code Locale} is not
     *         supported
     */
    LanguageData getLanguageData(Locale locale);

    /**
     * <p>Creates a summarized version of the provided text.</p>
     *
     * <p>"Summarized" means that the text is reduced to a maximum of the provided number
     * of words. The algorithm will use an algorithm to determine which parts of the
     * original text can be dropped to reach the target length.</p>
     *
     * @param inputString The original text
     * @param outputLen The target length (in words); the actual result may contain less
     *                  words, but not more than specified by this parameter
     * @param language The language of the text, as returned by
     *                 {@link #getLanguageData(Locale)}
     * @return The summarized text
     * @throws SmartContentException if the text could not by summarized, for example if
     *                               required language data (model, dictionaries, ...) were
     *                               missing or an unsupported language was specified
     */
    String summarize(String inputString, int outputLen, LanguageData language) throws SmartContentException;

}