/*
 * ************************************************************************
 *  ADOBE CONFIDENTIAL
 *  ___________________
 *
 *   Copyright 2019 Adobe Systems Incorporated
 *   All Rights Reserved.
 *
 *  NOTICE:  All information contained herein is, and remains
 *  the property of Adobe Systems Incorporated and its suppliers,
 *  if any.  The intellectual and technical concepts contained
 *  herein are proprietary to Adobe Systems Incorporated and its
 *  suppliers and are protected by all applicable intellectual property
 *  laws, including trade secret and copyright laws.
 *  Dissemination of this information or reproduction of this material
 *  is strictly forbidden unless prior written permission is obtained
 *  from Adobe Systems Incorporated.
 * ************************************************************************
 */

package com.adobe.cq.xf;

import org.osgi.annotation.versioning.ConsumerType;
import org.apache.sling.api.resource.Resource;

/**
 * <h1>Experience Fragment Link Rewriter Provider</h1>
 *  <p>
 *  This is an extension point offered to clients in order to allow link rewriting for the Export to Target feature.
 *  <p>
 *  Services implementing this interface will be called during the Export To Target offer content generation for HTML.
 *  <p>
 *  When an Export to Target call is made in the AEM Experience Fragment console, the body of the HTML content is
 *  generated through a request to an Experience Fragment with the selector {@code atoffer.html}. However, before this
 *  HTML content can be sent to Target, we would have to rewrite the links present throughout the content
 *  in such a way so that they point to resources on the publish instance, where they can be accessed publicly.
 *  We already rewrite these links using Externalizer, but if there is the need for more customization,
 *  a service can implement this interface in order to take control of the link rewriting process.
 *  <p>
 *  For example, a call is made to {@code http[s]://aem_instance[:port]/path/to/experience/fragment.nocloudconfigs.atoffer.html}.
 *  During the Sling Rewriter phase, Sling will parse each node of the HTML output (called tag).
 *  At this time, if there is any service implementing this interface, the "shouldRewrite" method is called in order
 *  to establish if the current Experience Fragment should have links rewritten. If the method returns true, then
 *  our system doesn't make any more modifications to the link, but lets the service handle the rewriting through the
 *  {@code rewriteLink} method.
 *  <p>
 *  Currently, only one {@code ExperienceFragmentLinkRewriterProvider} for an Experience Fragment is permitted.
 *  If there are multiple providers for an Experience Fragment, only the one with the highest priority is chosen.
 *  Priority is determined by the providers {@code getPriority()} method. If there are several
 *  {@code ExperienceFragmentLinkRewriterProvider} with the same priority, then only one will be chosen.
 *
 */
@ConsumerType
public interface ExperienceFragmentLinkRewriterProvider {

    /**
     *
     *  This method receives a String which represents the location of the resource on the author instance. These are
     *  the values of attributes present on the HTML tags that are also passed as input.
     *  <p>
     *  The type of attributes that will be processed are the following:
     *  <ul>
     *   <li>{@code src} attributes</li>
     *   <li>{@code href} attributes</li>
     *   <li>{@code action} attributes</li>
     *   <li>attributes that contain {@code -href} in their name</li>
     *   <li>attributes that contain {@code -src} in their name</li>
     *  </ul>
     *  <p>
     *  The {@code link} this method receives is usually an URI in the form of {@code /path/to/resource}. To note that in
     *  some cases, the URI might differ, due to the effects of the {@code html} selector on a component inside
     *  the Experience Fragments. In other words, if there are custom components in your Experience Fragment, then
     *  the link received in this method might be different, depending on its implementation. Our out-of-the-box components
     *  are rendered as a path to the resource, as mentioned previously.
     *  <p>
     *  Besides the link, the method also receives the local attribute name, so that implementations can easily distinguish
     *  what attributes from the above set is currently being processed (e.g. there can be {@code data-src} attributes
     *  that an implementation might not want to change).
     *  The last parameter is the tag name that is currently being processed, that can be helpful in the case some
     *  resources should not be changed, such as {@code link} attributes when the purpose is embedding inside another page.
     *  <p>
     *  The method should return a valid link to the resource, that can be accessed publicly. This link will be
     *  then placed in the corresponding attribute and tag.
     *  An example of an output for a regular case is the mapped public facing URL to a resource:
     *  {@code http[s]://[public-hostname]/path/to/resource}.
     *  How the output should look like is an implementation detail left to the service that implements this interface.
     *  The regular use case is that the Target Offer is embedded inside other HTML pages as a separate {@code div} and
     *  then delivered to clients or used in other places, and as such all the links inside this Target Offer should be
     *  able to be accessed by anyone.
     *  <p>
     *  If the method returns null, then no rewrite will be done and the link will remain the same.
     *
     * @param link - the value of the attribute, which is usually an internal URI (e.g. /path/to/resource)
     * @param tag - the name of the node that is currently modified
     * @param attribute - the name of the attribute that is currently modified
     * @return - a string representing the rewritten link that is public facing (e.g. {@code http[s]://[public_instance]/path/to/resource}), or null
     */
    String rewriteLink(String link, String tag, String attribute);

    /**
     *  This method receives the Resource which represents the Experience Fragment that is currently being processed
     *  by the Sling Rewriter framework.
     *  <p>
     *  Should return true if the service implementing this interface should be used on rewriting links
     *  for the experience fragment variation given.
     *
     * @param experienceFragment - the experience fragment that is currently processed
     * @return - true in case this service should rewrite links, false otherwise
     */
    boolean shouldRewrite(ExperienceFragmentVariation experienceFragment);

    /**
     *
     *  If there are several providers for which the {@code shouldRewrite} returns true for a particular
     *  ExperienceFragmentVariation, then the one with the highest priority is used.
     *
     * @return - an Integer representing the priority of this Provider over others for which {@code shouldRewrite(ExperienceFragmentVariation)} returns true.
     */
    int getPriority();

}
