/*
 * File: Base14.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font;

import com.adobe.fontengine.fontmanagement.Base14Font;
import com.adobe.fontengine.inlineformatting.css20.CSS20FontSet;

/**
 * PDF Base 14 fonts.
 * 
 * <p>
 * PDF documents can refer to fourteen Type 1 fonts with well-known names,
 * without having to embed those fonts, yet achieve predictable results. PDF
 * consumers are supposed to have access to actual fonts that can be used
 * whenever a PDF document refers to one of those fonts. Collectively, those
 * fonts are known as the Base 14 fonts.
 * 
 * <p>
 * PDF consumers are given a little bit of freedom in choosing the actual fonts,
 * but not much: the glyph complement, Type 1 encoding and the advance widths of
 * the glyphs are imposed by the PDF specification.
 * 
 * <p>
 * AFE supports the generation of PDF documents that refers to those 14 fonts
 * in the following way: AFE exposes 14 {@link Font} objects, and guarantees
 * that if layout is performed with one of those objects, then a PDF content
 * stream generated using the usual method (but calling for the corresponding
 * Base 14 font, rather than embedding it) will lead to the appropriate result.
 * 
 * <p>
 * This guarantee is limited to the 14 {@link Font} objects exposed in this
 * class. In particular, using the usual mechanism to load a font which happens
 * to be named "Symbol" does not, by itself, guarantee compabitility with the
 * Base 14 font "Symbol".
 * 
 * <p>
 * The {@link Font} objects exposed by this package are suitable for CSS
 * selection. That is, they can be added to {@link CSS20FontSet}s, and selected
 * by the usual CSS attributes. For each font, we indicate its CSS family name
 * (the other characteristics are obvious).
 * 
 * <p>
 * The {@link Font} objects exposed by this package behave as if they had a
 * Unicode 'cmap', which is necessary for layout of text. The details of these
 * cmaps can be found <a href='doc-files/base14.html'>here </a>.
 */
final public class Base14 {
  private Base14 () {
    // this class is only a name space for static objects
  }

  /** The Courier font, with CSS family name Courier. */
  public static final Font courierRegular 
    = new Base14Font ("FrankenCourier-Regular.otf",  "Courier", "Courier");
  
  /** The Courier-Bold font, with CSS family name Courier. */
  public static final Font courierBold  
    = new Base14Font ("FrankenCourier-Bold.otf",  "Courier", "Courier-Bold");
  
  /** The Courier-Oblique font, with CSS family name Courier. */
  public static final Font courierOblique       
    = new Base14Font ("FrankenCourier-Oblique.otf",  "Courier", "Courier-Oblique");
  
  /** The Courier-BoldOblique font, with CSS family name Courier. */
  public static final Font courierBoldOblique   
    = new Base14Font ("FrankenCourier-BoldOblique.otf", "Courier", "Courier-BoldOblique");
  
  		        
  /** The Helvetica font, with CSS family name Helvetica. */
  public static final Font helveticaRegular
    = new Base14Font ("AdobeSansF1-Regular.otf",  "Helvetica", "Helvetica");
  
  /** The Helvetica-Bold font, with CSS family name Helvetica. */
  public static final Font helveticaBold
    = new Base14Font ("AdobeSansF1-Bold.otf",  "Helvetica", "Helvetica-Bold");
  
  /** The Helvetica-Oblique font, with CSS family name Helvetica. */
  public static final Font helveticaOblique
    = new Base14Font ("AdobeSansF1-Italic.otf",  "Helvetica", "Helvetica-Oblique");
  
  /** The Helvetica-BoldOblique font, with CSS family name Helvetica. */
  public static final Font helveticaBoldOblique
    = new Base14Font ("AdobeSansF1-BoldItalic.otf", "Helvetica", "Helvetica-BoldOblique");
  
  		    
  /** The Times-Roman font, with CSS family name Times. */
  public static final Font timesRegular
    = new Base14Font ("AdobeSerifF1-Regular.otf",  "Times", "Times-Roman");
  
  /** The Times-Bold font, with CSS family name Times. */
  public static final Font timesBold
    = new Base14Font ("AdobeSerifF1-Bold.otf",  "Times", "Times-Bold");
  
  /** The Times-Italic font, with CSS family name Times. */
  public static final Font timesItalic
    = new Base14Font ("AdobeSerifF1-Italic.otf",  "Times", "Times-Italic");
  
  /** The Times-BoldItalic font, with CSS family name Times. */
  public static final Font timesBoldItalic
    = new Base14Font ("AdobeSerifF1-BoldItalic.otf", "Times", "Times-BoldItalic");
  
  		    
  /** The Symbol font, with CSS family name Symbol. */
  public static final Font symbol
    = new Base14Font ("FrankenSymbol.otf",   "Symbol", "Symbol");
  		    
  
  /** The ZapfDingbats font, with CSS family name ZapfDingbats. */
  public static final Font zapfDingbats
    = new Base14Font ("FrankenDingbats.otf",   "ZapfDingbats", "ZapfDingbats");
  
  
  /** Return the Base 14 font for name, or null if name is not
   * the name of a Base 14 font. */
  static public Font fromPSName (String name) {
    if ("Times-Roman".equals (name)) {
      return Base14.timesRegular; }
    else if ("Times-Bold".equals (name)) {
      return Base14.timesBold; }
    else if ("Times-BoldItalic".equals (name)) {
      return Base14.timesBoldItalic; }
    else if ("Times-Italic".equals (name)) {
      return Base14.timesItalic; }
    else if ("Helvetica".equals (name)) {
      return Base14.helveticaRegular; }
    else if ("Helvetica-Bold".equals (name)) {
      return Base14.helveticaBold; }
    else if ("Helvetica-BoldOblique".equals (name)) {
      return Base14.helveticaBoldOblique; }
    else if ("Helvetica-Oblique".equals (name)) {
      return Base14.helveticaOblique; }
    else if ("Courier".equals (name)) {
      return Base14.courierRegular; }
    else if ("Courier-Bold".equals (name)) {
      return Base14.courierBold; }
    else if ("Courier-BoldOblique".equals (name)) {
      return Base14.courierBoldOblique; }
    else if ("Courier-Oblique".equals (name)) {
      return Base14.courierOblique; }
    else if ("Symbol".equals (name)) {
      return Base14.symbol; }
    else if ("ZapfDingbats".equals (name)) {
      return Base14.zapfDingbats; }
    else {
      return null; }
  }
}
