/*
*
*	File: FontInputStream.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font;

/** A source of bytes to read a font.
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>This class is not synchronized.</p>
 */

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.net.URL;
import java.net.URISyntaxException;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

public class FontInputStream {
  private static final long UNKNOWN_LIMIT = -1;

  private MessageDigest digest;
  private PushbackInputStream pbis;
  private long curOffset = 0;
  private static int constructions;	// Unsynchronized and therefore may be wrong
  private long offsetLimit;


  public FontInputStream (FontInputStream in) {
      pbis = in.pbis;
      curOffset = in.curOffset;
      digest = in.digest;
	  offsetLimit = in.offsetLimit;
  }
  
  public FontInputStream (InputStream in) {
	this(in, UNKNOWN_LIMIT);
  }

  public FontInputStream (URL url) throws IOException {
	long size = UNKNOWN_LIMIT;

	try {
		if (url.getProtocol().equalsIgnoreCase("file")) {
			File file = new File(url.toURI());
			if (file.exists()) {
				size = file.length(); }}
	} catch (URISyntaxException e) {
		size = UNKNOWN_LIMIT; }

	init(url.openStream(), size);	
  }

  public FontInputStream (InputStream in, long size) {
    init(in, size);
  }

  private void init(InputStream in, long size) {
    InputStream dis = null;
    MessageDigest d;
    
    try {
      d = MessageDigest.getInstance ("SHA");
      dis = new DigestInputStream (in, d); }
    catch (NoSuchAlgorithmException e) {
      d = null;
      dis = in; }
    
    digest = d;  
    pbis = new PushbackInputStream (dis, 400);
	offsetLimit = size;
    constructions++; 
  }

  public static int numFis()
  {
    return constructions;
  }

  public void close() throws IOException {
    try {
      pbis.close();
    } finally {
      pbis = null;
      digest = null;
    }
  }
  public boolean isValidOffset(long offset, long length) {
	  return isValidOffset(offset+length-1);
  }
  
  public boolean isValidOffset(long offset) {
	if (offset < 0) {
	  return false; }
    
    if (UNKNOWN_LIMIT == offsetLimit) {
	  return true; }

	return offset < offsetLimit;
  } 

  private long clampOffset(long offset, long length) {
    return offset+length-offsetLimit;
  }

  public long getCurrentOffset() {
	  return curOffset;
  }
  
  public long skipTo (long offset) throws IOException, InvalidFontException {
  	if (curOffset > offset) {
  		throw new InvalidFontException ("trying to read the same byte twice (read up to "
  				+ Long.toHexString (curOffset) + ", want to go back to " 
					+ Long.toHexString (offset) + ")"); }
  	
    long initialOffset = curOffset;
    while (curOffset < offset) {
        long bytesSkipped = pbis.skip (offset - curOffset);
        curOffset += bytesSkipped; 
    }
    
    return curOffset - initialOffset;
  }

  public int read (byte[] b, int off, int len) throws IOException {
    long length = len;
    if (! isValidOffset(curOffset, length)) {
		length = clampOffset(curOffset, length);
		if (length == 0) {
		  // Match InputStream behavior for beyond end of stream.
		  return -1; }}

    int n = pbis.read (b, off, (int)length);
    if (n > 0) {
       curOffset += n; }
    return n;
  }

  public int read (byte[] b) throws IOException {
    return read (b, 0, b.length);
  }

  public int read() throws IOException{
      long length = 1;
      if (! isValidOffset(curOffset, length)) {
		length = clampOffset(curOffset, length);
		if (length == 0) {
		  // Match InputStream behavior for beyond end of stream.
		  return -1; }} 

      int retVal = pbis.read();
      if (retVal != -1)
          curOffset += 1;
      return retVal;
  }

  public void unread(int b) throws IOException {
      pbis.unread(b);
      curOffset--;
  }
  
  public void unread (byte[] b, int off, int len) throws IOException {
    pbis.unread (b, off, len);
    curOffset -= len;
  }

  public void unread (byte[] b) throws IOException {
    unread (b, 0, b.length);
  }
  
  public byte[] getDigest () throws IOException {
    if (digest == null) {
      return new byte [0]; }
      
    // make sure that we suck in all the bytes, so that digest includes them
    { byte[] b = new byte [1024];
      int n = 0;
      while (n != -1) {
        n = pbis.read (b); }}
    
    return digest.digest ();
  }
}
