/*
*
*	File: GlyphBBoxOutlineConsumer.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font;

/**
 * A bezier consumer used to calculate the glyph bbox.
 * This consumer can be used to calculate multiple bboxes, but the reset
 * method must be called between each calculation.
 * 
 * <h4>Synchronization</h4>
 * 
 * This class is not synchronized. Multiple instances can safely
 * coexist without threadsafety issues, but each must only be accessed 
 * from one thread (or must be guarded by the client).
 */
final public class GlyphBBoxOutlineConsumer implements OutlineConsumer{
    	private double xmin, xmax, ymin, ymax;
    	private double cpx, cpy;
    	private Matrix matrix;
    	private final Matrix targetMatrix;
    	
    	public GlyphBBoxOutlineConsumer(Matrix finalMatrix)
		{
    		targetMatrix = finalMatrix;
    		matrix = null;
		}
       
    	/**
    	 * resets the state of the consumer so that a new bbox can be calculated
    	 */
    	public void reset()
    	{
    	    xmin = ymin = Double.POSITIVE_INFINITY;
    	    xmax = ymax = Double.NEGATIVE_INFINITY;
    	}
    	
    	public void setMatrix(Matrix newMatrix)
    	{
    		if (newMatrix.isInverse(targetMatrix))
    		{
    		    matrix = Matrix.IDENTITY_MATRIX;
    		}
    		else
    		{
    		    matrix = newMatrix.multiply(targetMatrix);
    		}
    		
    	}
  
        public void moveto(double x, double y)
        {
            Point p = new Point(x,y);
            matrix.applyToPoint(p);
            cpx = p.x;
            cpy = p.y;
        }

        public void lineto (double x, double y)
        {
        	setXMinMax(cpx);
        	setYMinMax(cpy);
        	
        	Point p = new Point(x,y);
            matrix.applyToPoint(p);
            
            cpx = p.x;
            cpy = p.y;
            
            setXMinMax(cpx);
            setYMinMax(cpy);
            
        }
        
        private void setXMinMax(double limit)
        {
            if (limit < xmin)
                xmin = limit;
            if (limit > xmax)
                xmax = limit;
        }
        
        private void setYMinMax(double limit)
        {
            if (limit < ymin)
                ymin = limit;
            if (limit > ymax)
                ymax = limit;
        }
        
        /** 
         * given a point along a bezier, see if it is a min or max.
         * The formula for a bezier is ax^3 + 3bx^2 + 3cx + d, where
         * a,b,c,d are the parameters to this function. xDir tells whether
         * we are updating xmin/xmax or ymin/ymax
         * 
         * @param s the point on the bezier
         */
        private void checkCurveMinMax(double s, double a, double b, double c, double d, boolean xDir)
        {
            // check to see if our solution is a new min/max
        	
        	// the bezier is only valid in [0,1], but s = 0 and s = 1
        	// are covered by the endpoints.
            if (s <= 0 || s > 1)
            	return;
            
            double limit = s * (s * (s * a + 3*b) + 3*c) + d;
            if (xDir)
            {
                setXMinMax(limit);
            }
            else
            {
                setYMinMax(limit);
            } 
        }
        
        /**
         * Given control points for a bezier, calculate the derivative and see where
         * it equals 0. At that point, we have a min or max. isXDir tells whether we are
         * looking at the x or y coordinates of the bezier.
         */
        private void setCurveMinMax(double p0, double p1, double p2, double p3, boolean isXDir)
        {
            // the formula for a bezier is ax^3 + 3bx^2 + 3cx + d
            // where a, b, c are given below and where d = p0. where the
            // derivative of this curve is 0, you have a max/min. 
            
            double a = p3 - 3*(p2 - p1) - p0;
            double b = p2 - 2*p1 + p0;
            double c = p1 - p0;
            
            if (a == 0)
            {
                // we have a quadratic (at most)
                
                if (b != 0)
                {
                    checkCurveMinMax(-c/(2*b), a, b, c, p0, isXDir);
                }
                
                return;
            }
            else
            {
                double r = b*b - a*c;
                if (r < 0)
                    return; // no solution
                
                r = Math.sqrt(r);
                checkCurveMinMax((-b + r)/a, a, b, c, p0, isXDir);
                checkCurveMinMax((-b - r)/a, a, b, c, p0, isXDir);
            }
 
        }
        
        public void curveto (double x1, double y1, double x2, double y2)
        {
          curveto (Math.round ((cpx + 2*x1)/3.0), Math.round ((cpy + 2*y1)/3.0), 
              Math.round ((2*x1 + x2)/3.0), Math.round ((2*y1 + y2)/3.0),
                   x2, y2);
        }
                   

        public void curveto (double x1, double y1,
                             double x2, double y2, double x3, double y3)
        {
            double endp_left, endp_bottom, endp_right, endp_top;
            double curve_left, curve_bottom, curve_right, curve_top;
            double orig_cpx, orig_cpy;
            
            orig_cpx = cpx;
            orig_cpy = cpy;
            
            // figure out if the curve might extend the bbox.
            
            Point p = new Point(x3,y3);
            matrix.applyToPoint(p);
            x3 = p.x;
            y3 = p.y;
            
            p = new Point(x1,y1);
            matrix.applyToPoint(p);
            x1 = p.x;
            y1 = p.y;
            
            p = new Point(x2,y2);
            matrix.applyToPoint(p);
            x2 = p.x;
            y2 = p.y;
            
            endp_left = Math.min(cpx, x3);
            endp_right = Math.max(cpx,x3);
            
            endp_bottom = Math.min(cpy, y3);
            endp_top = Math.max(cpy, y3);
           
            curve_left = Math.min(x1,x2);
            curve_right = Math.max(x1,x2);
            
            curve_bottom = Math.min(y1, y2);
            curve_top = Math.max(y1, y2);
            
            cpx = x3;
            cpy = y3;
            
            if (endp_left >= xmin && curve_left >= xmin 
                    && endp_bottom >= ymin && curve_bottom >= ymin
                    && endp_right <= xmax && curve_right <= xmax
                    && endp_top <= ymax && curve_top <= ymax)
            {
                // we cannot extend the bbox
                return;
            }
            
            if (curve_left < endp_left || curve_right > endp_right)
            {
                // in the x direction, the curve extends beyond the endpoints
                setCurveMinMax(orig_cpx, x1, x2, x3, true);
            } 
            
            // check if the curve is bounded by the endpoints
            setXMinMax(endp_left);
            setXMinMax(endp_right);
            
            if (curve_bottom < endp_bottom || curve_top > endp_top)
            {
                // in the y direction, the curve extends beyond the endpoints
                setCurveMinMax(orig_cpy, y1, y2, y3, false);
            } 
            
            // check if the curve is bounded by the endpoints
            setYMinMax(endp_bottom);
            setYMinMax(endp_top);
            
        }

        /**
         * Returns the glyph bbox. This function can be called any time after
         * the outlines have been run through this consumer.
         */
        public Rect getBBox()
        {
        	// if a glyph has no outlines, min > max.
        	if (xmin > xmax)
        		xmax = xmin = 0;
        	if (ymin > ymax)
        		ymin = ymax = 0;
        	
            return new Rect(xmin, ymin, xmax, ymax);
        }

		public void endchar() {}
}
