/*
*
*	File: SWFFont3Description.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2007 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font;

/**
 * This interface contains methods that allow for the construction of a SWF
 * DefineFont2 or DefineFont3 tag. 
 * 
 * When the mapping from character to glyph occurs in this interface, it is
 * always a simple mapping. This means no OpenType processing occurs. Only BMP
 * characters are handled, since that is all DefineFont2/3 can handle.
 * 
 * <h4>Synchronization</h4>
 * 
 * Implementations of this interface are immutable after contruction and contains no mutable
 * static data. It is therefore threadsafe.
 */

public interface SWFFontDescription {
	/**
	 * Get the outline associated with the given character.
	 */
	void getOutline(char ccode, OutlineConsumer consumer) throws UnsupportedFontException, InvalidFontException;
	
	/**
	 * Get the advance width associated with the given character.
	 */
	double getHorizontalAdvance(char ccode) throws UnsupportedFontException, InvalidFontException;
	
	/**
     * Return true iff the font has a non-notdef glyph associated with it.
     */
    boolean canDisplay(char c) throws UnsupportedFontException, InvalidFontException;
    	
	/**
	 * Determine the permissions associated with this font.
	 */
	Permission getPermissions() throws InvalidFontException, UnsupportedFontException;
	
	/**
	 * Get the family name for the font. Return null if none exists.
	 */
    String getFamily() throws InvalidFontException,UnsupportedFontException;
    
    /**
     * Get the subfamily name for the font. Return null if none exists.
     */
    String getSubFamily() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the ascent value for the font.
     * 
     * The returned value is in emscale units. 0 is returned if no ascent value can be found, even through heuristics.
     */
    double getAscent() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the descent for the font.
     * 
     * The returned value is in emscale units. 0 is returned if no descent value can be found, even through heuristics.
     */
    double getDescent() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the line gap for the font.
     * 
     * The returned value is in emscale units. 0 is returned if no line gap value can be found, even through heuristics.
     */
    double getLineGap() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the lowest unicode scalar value that has a non-notdef glyph associated with it.
     */
    int getFirstChar() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the largest BMP value that has a non-notdef glyph associated with it.
     */
    int getLastChar() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the number of glyphs in the font.
     */
    int getNumGlyphs() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Return true iff the font is the bold member of a family.
     */
    boolean isBold() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Return true iff the font is the italic member of a family.
     */
    boolean isItalic() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the native scaling units associated with the font. All metrics returned 
     * directly from this interface must be divided by this value to get to em. 
     */
    double getEmScale() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the copyright string associated with the font. Return null if none exists.
     */
    String getCopyright() throws InvalidFontException, UnsupportedFontException;

    /** 
     * Get the trademark string associated with the font. Return null if none exists.
     */
    String getTrademark() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the postscript name associated with the font. Return null if none exists.
     */
    String getPostscriptName() throws InvalidFontException, UnsupportedFontException;
    
    /**
     * Get the Mac/English full name associated with the font, as required by the
     * DefineFontName tag. Return null if none exists.
     */
    String getFullName() throws InvalidFontException, UnsupportedFontException;
}
