/*
*
*	File: Subset.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font;

/**
 * Represent a subset of a font.
 * 
 * <p>
 * Implementations of this interface keeps track of the correspondance between
 * gids in the original font and gids in the subsetted font, which can be
 * accessed via <code>getFullGID</code> and <code>getSubsetGID</code>. The
 * number of glyphs in the subset is accessed by <code>getNumGlyphs</code>.
 * New glyphs are added to the subset as needed when <code>getSubsetGID</code>
 * is called: if the glyph that is passed in is not yet in the subset, it is
 * added.
 * 
 * <p>
 * Initially, the subset contains only the glyphs needed for the
 * integrity of the subset font (e.g. it contains the .notdef glyph if that is
 * required by the font format). For TrueType fonts, the subset 
 * always includes the glyphs with glyphID 0, 1, 2, 3; the purpose of this is
 * to preserve the recommendation on the first four glyphs, if it has been 
 * followed by the original font (and the cost is typically very small,
 * even if the recommendation was not followed in the first place).
 * 
 * <p>
 * Some fonts do not allow subsetting. The subset behaves as if it had been
 * built with all the glyphs of the original font.
 * 
 * <p>
 * Some font technologies have the notion of building glyphs using other glyphs.
 * For example, a TrueType glyph outline can invoke other glyph outlines.
 * Implementations of this inteface maintain the subset so that it is closed,
 * i.e. if such a composite glyph is in the subset, then the component glyphs
 * are also in the subset.
 * 
 * <p>
 * There is no guarantee that the glyph ID in a subset be equal to the glyph ID
 * in the full font, even that is possible. In particular, adding all the
 * glyphs in a font to a subset may result in an arbitrary permutation of
 * the glyph ids (except for glyphs with well-known gids, such as .notdef, if
 * any).
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>
 * A Subset object implements enough synchronization to be called safely from
 * multiple threads.
 * 
 * <p>
 * As methods are called, glyphs are only added to the subset. In particular,
 * <code>getNumGlyphs</code> never decreases over time; if
 * <code>getSubsetGid</code> returns some value at some point it always
 * returns that value afterward; if <code>getFullGid</code> returns some value
 * at some point for a glyph in the subset, then it always returns that value
 * afterwards. Thus, code of the form:
 * 
 * <pre>
 * if (k &lt; getNumGlyphs ()) {
 *   getFullGid (k);
 * }
 * </pre>
 * 
 * does not need to be synchronized; the precondition on getFullGid is
 * guaranteed by the earlier test (since getNumGlyph cannot decrease).
 */
public interface Subset {

  /** Return the number of glyphs in the subset. */
  public int getNumGlyphs ();

  /** Return the subset gid corresponding to a full gid.
   * @param fullGid the full gid, must be a legal gid for the full font.
   * */
  public int getSubsetGid (int fullGid) 
      throws UnsupportedFontException, InvalidFontException;

  /** Return the currently esisting subset gid corresponding to a full gid,
   * if there is one, without generating a new one. Return -1 if the full gid is not in the subset.
   * @param fullGid the full gid, must be a legal gid for the full font.
   * */
  public int getExistingSubsetGid (int fullGid) 
      throws UnsupportedFontException, InvalidFontException;

  /** Return the full gid corresponding to a subset gid.
   * @param subsetGid the subset gid, must be in the range 
   * [0..getNumGlyphs () [
   */
  public int getFullGid (int subsetGid);
}  
