/*
*
*	File: FontSet.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.cff;

import java.io.IOException;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;

/** Represents a CFF font set. 
 */
final class FontSet {
  
  /** The underlying bytes. */
  protected final CFFByteArray data;
  
  /** The header. */
  protected final Header header;
  
  /** The nameINDEX. */
  protected final NameIndex nameIndex;
  
  /** The TopDICT INDEX. */
  protected final Index topDictIndex;
  
  /** The String INDEX. */
  protected final StringIndex stringIndex;
  
  /** The global subroutines. */
  protected final CharStrings globalSubrs;
  
  /** The fonts in the font set. */
  public final CFFFont[] fonts;
 
  /** Construct FontSet from a stream input. */
  public FontSet (FontByteArray buffer) 
  throws IOException, InvalidFontException, UnsupportedFontException {
    
    data = new CFFByteArray (buffer);
    
    if (buffer.getSize() < 4) {
      throw new InvalidFontException ("CFFFontSet too small"); }
    
    int offset = 0; 
    header = new Header (data, offset);
    offset += header.size ();
    
    { int majorVersion = header.getMajorVersion ();  
      if (majorVersion != 1) {
        throw new UnsupportedFontException ("CFFFontSet major version " +
          majorVersion + " not supported"); }}


    nameIndex = new NameIndex (data, offset);
    offset += nameIndex.size ();
    
    topDictIndex = new Index (data, offset);
    offset += topDictIndex.size ();
    
    stringIndex = new StringIndex (data, offset);
    offset += stringIndex.size ();
    
    globalSubrs = new CharStrings (data, offset);
    
    fonts = new CFFFont [nameIndex.getCount ()];
    Dict[] syntheticTopDicts = new Dict [nameIndex.getCount ()];
    int[] syntheticBases = new int [nameIndex.getCount ()];
    
    // Do a first pass to create non-synthetic fonts
    
    for (int i = 0; i < fonts.length; i++) {
      // The first character of the font name indicate whether the font
      // is really part of the fontset.
      
      String name = nameIndex.getNthName (i);

      if (name.length() == 0 || name.charAt (0) != 0) {
        Dict topDict = new Dict (data, topDictIndex.offsetOf (i), topDictIndex.sizeOf (i), stringIndex);

        Dict.ROSValue rosOp = topDict.get (Dict.Key.ROS, false);
        if (rosOp != null) {
          fonts [i] = new CIDKeyedFont (stringIndex, globalSubrs, name, topDict, data, null); }
        else {
          Dict.IntegerValue syntheticBaseOp = topDict.get (Dict.Key.SyntheticBase, false);
          if (syntheticBaseOp != null) {
            syntheticTopDicts [i] = topDict;
            syntheticBases [i] = syntheticBaseOp.value; }
          else {
            fonts [i] = new NameKeyedFont (stringIndex, globalSubrs, name, topDict, data, null); }}}}
      
    for (int i = 0; i < fonts.length; i++) {
      if (syntheticTopDicts [i] != null) {
        fonts [i] = new SyntheticFont (stringIndex, fonts [syntheticBases [i]], nameIndex.getNthName (i), syntheticTopDicts [i], data, null); }}
  }
}
