/*
 *
 *	File: Type2CStringGenerator.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.cff;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Collections;
import java.util.Comparator;
import java.util.ListIterator;
import java.util.Iterator;

import com.adobe.fontengine.font.HintedOutlineConsumer;
import com.adobe.fontengine.font.cff.CFFByteArray.CFFByteArrayBuilder;
import com.adobe.fontengine.font.postscript.SubArrays;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.Matrix;

/**
 * Generate a type2 charstring index from hinted outlines.
 * 
 * An instance of this class cannot be reused with multiple fonts.
 * 
 * newGlyph must be called before each glyph that is played. After all glyphs
 * have been played, calculateDefaultWidth and calculateNominalWidth may be used
 * to calculate optimal values for the DefaultWidth and NominalWidth cff keys. If
 * one of calculateDefaultWidth or calculateNominalWidth is called, the other must also
 * be called and the returned values must be put in the cff dictionary associated with
 * the call unless default values are returned. (Default values are 
 * Type2CStringGenerator.DEFAULT_NOMINAL_WIDTH and Type2CStringGenerator.DEFAULT_DEFAULT_WIDTH).
 * Finally, getCharstringIndex can be called to fetch an index formatted for use as a
 * cff charstring index.
 * 
 * <h4> Synchronization</h4>
 * This class is not synchronized. If a single instance is to be used in multiple threads, 
 * the client is responsible for its threadsafety. However, multiple instances of 
 * Type2CStringGenerator can be used concurrently in separate threads.
 */
final public class Type2CStringGenerator implements HintedOutlineConsumer {
  private static final int MAX_STACK=48;
  private static final int MAX_STEMS = 96;
  private static final int STANDARD_FLEX_DEPTH = 50;
  public static final double DEFAULT_NOMINAL_WIDTH = 0;
  public static final double DEFAULT_DEFAULT_WIDTH = 0;
  private static final double BOTTOM_EDGE = -21;
  private static final double TOP_EDGE = -20;
  private static final boolean debug = false;

  /** the glyph id of the charstring we are working on */
  private int currCStringIndex = 0;

  /** the info about the glyphs seen thus far */
  GlyphInfo[] glyphs;

  /** default and nominal widths are analyzed on a fontdescriptor basis. widthsAnalyzed
   * says whether a given fd has had its widths analyzed yet.
   */
  private boolean[] widthsAnalyzed;
  private double[] nominalWidth;
  private double[] defaultWidth;

  // intermediate charstring data. outlines can't be converted to type2 until
  // the end of all of the charstrings since hint declarations have to 
  // all be at the beginning of the charstring and
  // since the width isn't known until nominalwidth is calculated.


  /** the current X position in the current subpath. */
  private double currentX;
  /** the current Y position in the current subpath. */
  private double currentY;

  /** in the case of HV or VH alternating ops (eg, HVLine) startOfSeqOp tells
   * the operator that begins the sequence. For other ops, it is unused.
   */
  private int startOfSeqOp;

  /** ensure a moveto before all subpaths. */
  private boolean moveToSeen;

  /** in the case of HV or VH alternating ops (eg, HVLine) lastOpCode tells
   * the current alternate in the sequence. For other ops, it tells the op
   * to be written out.
   */
  private int lastOpCode;
  private int stackCount;
  private double stack[] = new double[MAX_STACK];

  /** the index in the charstring where the current hintmask is located. */
  private int hintMaskIndex;

  private boolean hintMaskSeen;
  private boolean cntrMaskSeen;

  /** the hintmask being worked on */
  private byte[] hintmask = new byte[(MAX_STEMS + 7)/8];

  /** the cntrmask being worked on */
  private byte[] cntrmask = new byte[(MAX_STEMS + 7)/8];

  /** add extra mask op byte count for subrizer if true */
  private boolean subrize;

  private static class WidthFrequency {
    double width;
    int count;

    WidthFrequency (double width) {
      this.width = width;
      this.count = 1; }

    WidthFrequency () {
      count = 0;}
  }

  private static class WidthComparator implements Comparator{
    public int compare(Object o1, Object o2) {
      WidthFrequency f1 = (WidthFrequency)o1;
      WidthFrequency f2 = (WidthFrequency)o2;

      return (int)(f1.width - f2.width);
    }
  }

  private static class HintInfo {
    final double top; // the top/right edge of the hint
    final double bottom; // the bottom/left edge of the hint.
    final int flags; 
    int id; // the id of the hint.

    // possible flags
    final static int VERTICAL_STEM = 1<<0;
    final static int CNTR_STEM = 1<<1;

    HintInfo(double top, double bottom, int flags) {
      this.top = top;
      this.bottom = bottom;
      this.flags = flags;
    }


    boolean fuzzyMatch(HintInfo oldHint) {            
      return ((oldHint.flags & HintInfo.CNTR_STEM) != 0
          && ((flags & HintInfo.VERTICAL_STEM) == (oldHint.flags & HintInfo.VERTICAL_STEM))
          && Math.abs(bottom - oldHint.bottom) < 2
          && Math.abs(top - oldHint.top) < 2);
    }
  }

  private static class HintComparator implements Comparator {
    public int compare(Object o1, Object o2) {
      HintInfo h1 = (HintInfo) o1;
      HintInfo h2 = (HintInfo) o2;
      int diff;

      diff = (h1.flags & HintInfo.VERTICAL_STEM) - (h2.flags & HintInfo.VERTICAL_STEM);
      if (diff != 0) {
        return diff; }

      if (h1.bottom < h2.bottom) {
        return -1; }
      if (h1.bottom > h2.bottom) {
        return 1; }

      if (h1.top < h2.top) {
        return -1; }
      if (h1.top > h2.top) {
        return 1; }

      return 0;
    }
  }

  private static class HintMask {
    /** the index into GlyphInfo.tmpcstr where this hintmask should start */
    final int indexIntoCStr; 
    final byte[] mask = new byte[(MAX_STEMS + 7)/8];

    HintMask(int indexIntoCStr, byte[] mask) {
      this.indexIntoCStr = indexIntoCStr;
      System.arraycopy(mask, 0, this.mask, 0, mask.length);
    }
  }

  private static class CntrMask {
    byte[] mask = new byte[(MAX_STEMS + 7)/8];

    CntrMask(byte[] mask) {
      System.arraycopy(mask, 0, this.mask, 0, mask.length);
    }
  }

  /** a byte array that can grow as needed. */
  final static class GrowableBuffer {
    byte[] cstr = new byte[200];
    int cstrLen = 0;

    void ensureCStrIsBigEnough(int numBytesNeeded) {
      if (cstrLen + numBytesNeeded > cstr.length) {
        byte[] tmp = new byte[cstr.length + Math.max(numBytesNeeded, 50)];
        System.arraycopy(cstr, 0, tmp, 0, cstrLen);
        cstr = tmp; }
    }

    void addByte (int value) {
      ensureCStrIsBigEnough(1);
      cstr[cstrLen++] = (byte)(value & 0xff);
    }
    
    /** Convert an integer to their cff encoded value and write them to this. */
    private void intToCString(int iVal)  {
      short value = (short)iVal;
      if (-107 <= value && value <= 107) {
        addByte(value + 139); }
      else if (108 <= value && value <= 1131) {
        value -= 108;
        addByte((value>>8) + 247);
        addByte(value); }
      else if (-1131 <= value && value <= -108) {
        value += 108;
        addByte((-value>>8) + 251);
        addByte(-value); }
      else if (-32768 <= value && value <= 32767) {
        addByte(Type2CStringOps.SHORTINT);
        addByte(value>>8);
        addByte(value); }
    }

    /** Convert a number to its cff encoded value and write it to this */
    void numToCString(double value)  {
      int intVal = (int)value;
      if (intVal == value)  {
        intToCString(intVal);
        return; }

      int i = (int)(value*65536.0 + ((value < 0)? -0.5: 0.5));
      addByte(Type2CStringOps.FIVE_BYTE);
      addByte(i>>24);
      addByte(i>>16);
      addByte(i>>8);
      addByte(i);
    }
  }


  final static class GlyphInfo {
    /** the advance width of the glyph */
    double width;
    /** the fd that the glyph is associated with */
    final int fd;
    /** the hints associated with the glyph */
    List hints = new ArrayList();
    /** the hintmasks associated with the glyph */
    List /* HintMask */ hintMasks = new ArrayList();
    /** the cntrmasks associated with the glyph */
    List /* CntrMask */ cntrMasks = new ArrayList();
    /** the hintmask that comes before the first subpath */
    byte[] initMask = new byte[(MAX_STEMS + 7)/8];
    /** the path portion of the charstring encoded as a type2 cstr */
    GrowableBuffer tmpcstr = new GrowableBuffer(); 
    /** the final complete type 2 charstring */
    GrowableBuffer finalcstr = new GrowableBuffer();

    GlyphInfo(int fd, double defaultWidth) {
      this.fd = fd;
      width = defaultWidth;
    }

    /** look for the specified hint. If it already exists, return its id. Otherwise, create
     * a new one and return its id.
     * @param edge1 The lower/left edge of the hint.
     * @param edge2 The upper/right edge of the hint.
     * @param vertical true iff the hint is a vertical hint.
     * @param counter true iff the hint controls counters.
     */
    private int findHint(double edge1, double edge2, boolean vertical, boolean counter) {
      double e1, e2, delta;
      HintInfo thisHint;
      Comparator c = new HintComparator();

      delta = edge2 - edge1;

      if (delta < 0 
          && Double.compare(delta, BOTTOM_EDGE) != 0 
          && Double.compare(delta,TOP_EDGE) != 0)  {
        e1 = edge2;
        e2 = edge1; }
      else {
        e1 = edge1;
        e2 = edge2; }
      thisHint = new HintInfo(e2, e1, 
          (vertical ? HintInfo.VERTICAL_STEM: 0) | (counter ? HintInfo.CNTR_STEM: 0));

      // look in array for this stem
      int index = Collections.binarySearch(hints, thisHint, c);


      // if not found, create a new one and add it to the array
      if (index < 0) {
        if (cntrMasks.size() > 0 && !counter) {
          // counter groups are the first hints to be specified in type1 fonts. They always
          // use integerized stem values, while the hints do not. We need to match those
          // counter values rather than create a new hint or we will have overlapping hints.
          // If we have seen counters, but this is not a counter, do a fuzzy match. This will
          // not be the case with type2 -> type2 since counters can never come before hints.
          if (-index - 1 < hints.size() && thisHint.fuzzyMatch((HintInfo)hints.get(-index-1))) {
            return ((HintInfo)hints.get(-index-1)).id; }
          else if (-index - 1 > 0 && thisHint.fuzzyMatch((HintInfo)hints.get(-index-2))) {
            return ((HintInfo)hints.get(-index-2)).id; } }
        thisHint.id = hints.size();
        if (thisHint.id < MAX_STEMS) {
          hints.add(-index - 1, thisHint);
        } else {
          if (debug) System.out.println("Too many stems at findHint");
        }
        return thisHint.id; }

      // return the id
      return ((HintInfo)hints.get(index)).id;
    }


    private static boolean maskByteSet(int id, byte[]mask) {
      return (mask[id/8] & (1<<(id%8))) != 0;
    }


    private static void addHintToMask(int index, byte[] mask)  {
      mask[index / 8] |= (1 << (7 - (index % 8)));
    }


    private void writeStems(int start, int limit, int opCode, boolean writeOpCode) {
      // 2 args per stem.  -1 because the width may be on the stack.
      final int stems_per_op = (MAX_STACK-1)/2; 

      // number of ops needed to write all the stems
      int ops = (limit - start + stems_per_op - 1)/stems_per_op; 

      // start with the "short" op.
      int hintsForThisOp = limit - start - (ops - 1)*stems_per_op;

      ListIterator iter = hints.listIterator(start);
      while (ops-- > 0) {
        double last = 0;
        while (hintsForThisOp-- > 0) {
          HintInfo hint = (HintInfo)iter.next();
          finalcstr.numToCString(hint.bottom - last);
          finalcstr.numToCString(hint.top - hint.bottom);
          last = hint.top; }

        if (writeOpCode || ops > 0) {
          finalcstr.addByte(opCode); }

        hintsForThisOp = stems_per_op; }
    }

    private void saveStartOfCString(byte[] hintMap, double[] defaultWidth, double[] nominalWidth, boolean willSubrize) {
      int maskBytes = (hints.size() + 7) / 8;
      int numHStems;
      int i;
      byte[] tmpMask = new byte[maskBytes];
      boolean haveMask = false, initMask = false;
      int numHints = hints.size();

      // write out width
      if (width != defaultWidth[fd]) {
        finalcstr.numToCString(width - nominalWidth[fd]); }

      // if we have no stems, we can't have masks either, so we are done.
      if (numHints == 0) {
        return; }

      Iterator iter = hints.iterator();

      // how many of our stems are horizontal?
      numHStems = 0;
      while (iter.hasNext()) {
        HintInfo hint = (HintInfo)iter.next();
        if ((hint.flags & HintInfo.VERTICAL_STEM) != 0) {
          break; } // stop counting when we hit the first vertical stem.

        numHStems++; }

      // do we have any hintmasks? 

      Arrays.fill(tmpMask, (byte)0xff);

      if ((numHints %8) != 0) {
        tmpMask[tmpMask.length-1] = (byte)(0xff>>(8 - (numHints%8))); }

      if (!SubArrays.arrayCompare(tmpMask, 0, this.initMask, 0, tmpMask.length)) {
        haveMask = true;
        initMask = true; }
      else {
        // XXX_lmb optimization: check for to see if all hintmasks are dups. if so,
        // get rid of all but the first.
        if (!hintMasks.isEmpty()) {
          HintMask hintMask = (HintMask)hintMasks.get(0);
          if (hintMask.indexIntoCStr == 0)
            initMask = true;
          haveMask = true; }}

      byte[] curMaskBytes = initMask ? this.initMask : tmpMask;
      iter = hintMasks.iterator();
      while (iter.hasNext()) {
        HintMask hintMask = (HintMask)iter.next();
        if (hintMask.indexIntoCStr != 0 && SubArrays.arrayCompare(curMaskBytes, 0, hintMask.mask, 0, curMaskBytes.length))
          iter.remove();
        else
          curMaskBytes = hintMask.mask;
      }
      if (hintMasks.isEmpty() && !initMask)
        haveMask = false;

      // write out hstems
      writeStems(0, numHStems, 
          haveMask? Type2CStringOps.HSTEMHM: Type2CStringOps.HSTEM, true);        

      // write out vstems
      writeStems(numHStems, numHints, 
          haveMask? Type2CStringOps.VSTEMHM: Type2CStringOps.VSTEM, !initMask && cntrMasks.isEmpty()); 

      // write out countermask(s)
      iter = cntrMasks.iterator();
      while (iter.hasNext()) {
        CntrMask mask = (CntrMask)iter.next();
        Arrays.fill(tmpMask, (byte)0);
        finalcstr.addByte(Type2CStringOps.CNTRMASK);

        for (i = 0; i < numHints; i++) {
          if (maskByteSet(i, mask.mask)) {
            addHintToMask(hintMap[i], tmpMask); } }

        if (willSubrize)
          finalcstr.addByte(tmpMask.length + 2);

        for (i = 0; i < tmpMask.length; i++) {
          finalcstr.addByte(tmpMask[i]); } }

      // write out initial hint mask
      if (initMask) {
        Arrays.fill(tmpMask, (byte)0);
        finalcstr.addByte(Type2CStringOps.HINTMASK);

        for (i = 0; i < numHints; i++) {
          if (maskByteSet(i, this.initMask)) {
            addHintToMask(hintMap[i], tmpMask); } }

        if (willSubrize)
          finalcstr.addByte(tmpMask.length + 2);

        for (i = 0; i < tmpMask.length; i++) {
          finalcstr.addByte(tmpMask[i]); }}
    }

    private void savePathsToCString(byte[] hintMap, boolean willSubrize) {
      int maskBytes = (hints.size() + 7) / 8;
      Iterator iter = hintMasks.iterator();
      int cstrIndex = 0, i;
      byte[] tmpMask = new byte[maskBytes];
      int numHints = hints.size();

      while (iter.hasNext()) {
        HintMask mask = (HintMask)iter.next();
        int stopAt = mask.indexIntoCStr;

        while (cstrIndex < stopAt) {
          finalcstr.addByte(tmpcstr.cstr[cstrIndex]);
          cstrIndex++; }

        // write out hintmask
        Arrays.fill(tmpMask, (byte)0);
        finalcstr.addByte(Type2CStringOps.HINTMASK);

        for (i = 0; i < numHints; i++) {
          if (maskByteSet(i, mask.mask)) {
            addHintToMask(hintMap[i], tmpMask); } }

        if (willSubrize)
          finalcstr.addByte(tmpMask.length + 2);

        for (i = 0; i < tmpMask.length; i++) {
          finalcstr.addByte(tmpMask[i]); } }

      // write out final portion of path
      while (cstrIndex < tmpcstr.cstrLen) {
        finalcstr.addByte(tmpcstr.cstr[cstrIndex]);
        cstrIndex++; }
    }

    void generateFullCharstring(double[] defaultWidth, double[] nominalWidth, boolean willSubrize) {
      byte[] hintMap = new byte[hints.size()];
      byte i;
      Iterator iter = hints.iterator();

      for (i = 0; iter.hasNext(); i++) {
        HintInfo info = (HintInfo)iter.next();
        hintMap[info.id] = i; }

      saveStartOfCString(hintMap, defaultWidth, nominalWidth, willSubrize); // width, stems, initial masks
      savePathsToCString(hintMap, willSubrize); // paths and subsequent hintmasks
    }

    void setWidth(double width, double[] nominalWidth) {
      this.width = width + nominalWidth[fd];
    }
  }

  /**
   * Creates an instance of a Type2CStringGenerator.
   * @param numGlyphs The number of charstrings that will be generated.
   * @param numFDs The number of font dictionaries that will be associated with those charstrings.
   * numFDs must be at least 1.
   */
  public Type2CStringGenerator(int numGlyphs, int numFDs) {
    init(numGlyphs, numFDs, false);
  }

  public Type2CStringGenerator(int numGlyphs, int numFDs, boolean willSubrize) {
    init(numGlyphs, numFDs, willSubrize);
  }

  private void init(int numGlyphs, int numFDs, boolean willSubrize) {
    int i;
    glyphs = new GlyphInfo[numGlyphs];
    widthsAnalyzed = new boolean[numFDs];
    nominalWidth = new double[numFDs];
    this.defaultWidth = new double[numFDs];
    this.subrize = willSubrize;

    for (i = 0; i < numFDs; i++) {
      nominalWidth[i] = DEFAULT_NOMINAL_WIDTH;
      this.defaultWidth[i] = DEFAULT_DEFAULT_WIDTH; }

    resetState();
  }

  /** reset state between glyphs. */
  private void resetState()  {
    moveToSeen = false;
    lastOpCode = 0;
    stackCount = 0;
    currentX = 0;
    currentY = 0;
    startOfSeqOp = 0;
    hintMaskSeen = false;
    cntrMaskSeen = false;
    hintMaskIndex = -1;
    Arrays.fill(hintmask, (byte)0);
    Arrays.fill(cntrmask, (byte)0);
  }



  /** Write the operator in lastOpCode and its associated stack values to tmpcstr */
  private void stackToCString () {
    if (stackCount != 0) {
      for (int i = 0; i < stackCount; i++) {
        glyphs[currCStringIndex].tmpcstr.numToCString(stack[i]); }
      stackCount = 0; }

    switch (lastOpCode) {
      case Type2CStringOps.HLINETO:
      case Type2CStringOps.VLINETO:
      case Type2CStringOps.HVCURVETO:
      case Type2CStringOps.VHCURVETO: {
        glyphs[currCStringIndex].tmpcstr.addByte(startOfSeqOp);
        break; }
      default: {
        glyphs[currCStringIndex].tmpcstr.addByte(lastOpCode); } }

    lastOpCode = 0;
  }

  /** Ensure that there is room from numArgs 
   * (where numArgs must be < MAXSTACK) on the stack. 
   */
  private void ensureStackSpace(int numArgs) {
    if (stackCount + numArgs + (this.subrize ? 1 : 0) > MAX_STACK) {
      stackToCString(); }
  }

  /** Called before playing the outlines for a new glyph.
   * This function must be called exactly one time for each glyph that is to be
   * played.
   * @param glyphID The glyphID for the new glyph. glyphID must be less than the numGlyphs
   * value passed to the constructor.
   * @param fontDictionary The font dictionary to which the glyph belongs. fontDictionary must
   * be at least 0 and must be less than the numFDs value passed to the constructor.
   * @param defaultWidth The default width associated with this glyph. DEFAULT_DEFAULT_WIDTH should
   * be passed if there isn't one.
   */
  public void newGlyph(int glyphID, int fontDictionary, double defaultWidth, double nominalWidth) {
    resetState();
    currCStringIndex = glyphID;
    glyphs[glyphID] = new GlyphInfo(fontDictionary, defaultWidth);
    this.defaultWidth[fontDictionary] = defaultWidth;
    this.nominalWidth[fontDictionary] = nominalWidth;   
  }

  /** start a new hintmask. Save out the current hintmask if there is one */
  private void newHintMask() {
    if (lastOpCode == 0 && hintMaskIndex == glyphs[currCStringIndex].tmpcstr.cstrLen)  {
      // if there are 2 hintmasks in a row, cancel out the first one.
      Arrays.fill(hintmask, (byte)0); }
    else {
      if (lastOpCode != 0) {
        stackToCString(); }

      if (hintMaskSeen)  {
        HintMask mask = new HintMask(hintMaskIndex, hintmask);
        glyphs[currCStringIndex].hintMasks.add(glyphs[currCStringIndex].hintMasks.size(), mask);
        Arrays.fill(hintmask, (byte)0); } 
      else {
        hintMaskSeen = true; }

      // hold onto the current location in the cstring.
      hintMaskIndex = glyphs[currCStringIndex].tmpcstr.cstrLen; }
  }

  /** start a new cntrmask. Save out the current cntrmask if there is one */
  private void newCntrMask() {
    if (lastOpCode != 0) {
      stackToCString(); }

    if (cntrMaskSeen) {
      CntrMask mask = new CntrMask(cntrmask);
      glyphs[currCStringIndex].cntrMasks.add(glyphs[currCStringIndex].cntrMasks.size(), mask);
      Arrays.fill(cntrmask, (byte)0); }
    else {
      cntrMaskSeen = true; }
  }

  private void addIdToMask(int id, byte[] mask) {
    if (id < MAX_STEMS) {
      mask[id / 8] |= (1 << (id % 8));
    } else {
      if (debug) System.out.println("Too many stems at addIdToMask");
    }
  }

  /**
   * If a new hintmask is needed, create one.
   */
  private void checkForNewHintMask(boolean doHintSubstitution) {

    // we starting a new hint group if 1) we have had path operations already happen and
    // either the caller says this is a new group or if we have never seen any hints or 
    // 2) we have not seen path operations but the caller says it is a new group and we
    // have seen previous hints. 
    if ((moveToSeen && (doHintSubstitution || glyphs[currCStringIndex].hints.size() == 0))
        || (!moveToSeen && doHintSubstitution && glyphs[currCStringIndex].hints.size() > 0))
      newHintMask();
  }

  /**
   * If a new cntrmask is needed, create one.
   * @param newGroup true iff the client says that we are starting a new counter group.
   */
  private void checkForNewCntrMask(boolean newGroup) {
    // if we've never seen a cntr mask before, create one.
    if (!cntrMaskSeen || newGroup) {
      newCntrMask(); }
  }

  private void cntrHint(double edge1, double edge2, boolean vertical) {
    int id = glyphs[currCStringIndex].findHint(edge1, edge2, vertical, true);

    addIdToMask(id, cntrmask);

    if (hintMaskSeen) {
      addIdToMask(id, hintmask); }
    else {
      addIdToMask(id, glyphs[currCStringIndex].initMask); }
  }

  public void setMatrix(Matrix m) {
    // No implementation is needed given that the font dictionaries (containing the font matrix being
    // passed here) are copied to the new font.
  }


  public void stem(double edge1, double edge2, 
      boolean doHintSubstitution, boolean vertical, 
      boolean isCounter) {

    if (!isCounter) {
      checkForNewHintMask(doHintSubstitution); }

    int id = glyphs[currCStringIndex].findHint(edge1, edge2, vertical, isCounter);

    if (isCounter) {
      checkForNewCntrMask(doHintSubstitution);
      addIdToMask(id, cntrmask); }
    else {
      // add the new hint to the current hintmask.
      if (hintMaskSeen) {
        addIdToMask(id, hintmask); }
      else {
        addIdToMask(id, glyphs[currCStringIndex].initMask); }}
  }

  public void stem3(double edge1, double edge2, double edge3, double edge4,
      double edge5, double edge6, boolean doHintSubstitution,
      boolean isVertical) {

    checkForNewHintMask(doHintSubstitution);
    checkForNewCntrMask(false);

    cntrHint(edge1, edge2, isVertical);
    cntrHint(edge3, edge4, isVertical);
    cntrHint(edge5, edge6, isVertical);
  }

  public boolean width(double wx) {
    glyphs[currCStringIndex].setWidth(wx, nominalWidth);
    return true;
  }

  public void globalColorOn() {
    if (lastOpCode != 0) {
      stackToCString(); }

    glyphs[currCStringIndex].tmpcstr.addByte(Type2CStringOps.ESCAPE); 
    glyphs[currCStringIndex].tmpcstr.addByte(Type2CStringOps.ESC_GLOBALCOLORME);
  }

  public void noCounters() {
    // the rendering of a glyph with an empty cntrmask but global coloring on
    // is different than the rendering of a glyph with no cntrmask but global coloring on.
    // The former turns off 2-pass global coloring and says there are no counters. The
    // latter runs 2-pass global coloring.
    newCntrMask();
  }

  public void noHints() {
    newHintMask();
  }

  public void lineto(double x, double y) {
    if (!moveToSeen) {
      moveto(0,0); }

    double dx, dy;

    dx = x - currentX;
    dy = y - currentY;
    currentX = x;
    currentY = y;

    if (dx == 0) { // a vertical line.
      ensureStackSpace(1);
      switch (lastOpCode) {
        default:
          stackToCString();
        /* fall through */

        case 0:
          startOfSeqOp = Type2CStringOps.VLINETO;
          /* fall through */

        case Type2CStringOps.HLINETO:
          stack[stackCount++] = dy;
          lastOpCode = Type2CStringOps.VLINETO;
          break; } }
    else if (dy == 0) { // a horizontal line.
      ensureStackSpace(1);
      switch (lastOpCode)  {
        default:
          stackToCString();
        /* fall through */

        case 0:
          startOfSeqOp = Type2CStringOps.HLINETO;
          /* fall through */

        case Type2CStringOps.VLINETO:
          stack[stackCount++] = dx;
          lastOpCode = Type2CStringOps.HLINETO;
          break; }}
    else { // a diagonal line.
      ensureStackSpace(2);
      switch (lastOpCode) {
        case Type2CStringOps.RLINETO:
          stack[stackCount++] = dx;
          stack[stackCount++] = dy;
          break;
        case Type2CStringOps.RRCURVETO:
          stack[stackCount++] = dx;
          stack[stackCount++] = dy;
          lastOpCode = Type2CStringOps.RCURVELINE;
          stackToCString(); // this must be the last args for this command. flush the stack.
          break;
        default:
          stackToCString();
        /* fall through */
        case 0:
          stack[stackCount++] = dx;
          stack[stackCount++] = dy;
          lastOpCode = Type2CStringOps.RLINETO;
          break; }}
  }

  private void addVHCurveTo(double dy1, double dx2, double dy2, double dx3)  {
    ensureStackSpace(4);
    switch (lastOpCode) {
      default:
        stackToCString();
      /* Fall through */
      case 0:
        startOfSeqOp = Type2CStringOps.VHCURVETO;
        /* fall through */

      case Type2CStringOps.HVCURVETO:
        stack[stackCount++] = dy1;
        stack[stackCount++] = dx2;
        stack[stackCount++] = dy2;
        stack[stackCount++] = dx3;
        lastOpCode = Type2CStringOps.VHCURVETO;
        break; }
  }

  private void addVHCurveTo(double dy1, double dx2, double dy2, double dx3, double dy3) {
    ensureStackSpace(5);
    addVHCurveTo(dy1, dx2, dy2, dx3);
    stack[stackCount++] = dy3;
    stackToCString(); // with 5 args, we have to end this sequence
  }

  private void addVVCurveTo(double dy1,double dx2,double dy2,double dy3) {
    ensureStackSpace(4);
    switch (lastOpCode) {
      default:
        stackToCString();
      /* Fall through */
      case Type2CStringOps.VVCURVETO:
      case 0:
        stack[stackCount++] = dy1;
        stack[stackCount++] = dx2;
        stack[stackCount++] = dy2;
        stack[stackCount++] = dy3;
        lastOpCode = Type2CStringOps.VVCURVETO;
        break; }
  }

  private void addVVCurveTo(double dx1, double dy1, double dx2, double dy2, double dy3)  {
    ensureStackSpace(5);

    // the 5 arg version of VVcurveTo must begin a sequence (unlike HV/VH where the
    // 5 arg version must end a sequence).
    switch (lastOpCode) {
      default:
        stackToCString();
      /* Fall through */
      case 0:
        stack[stackCount++] = dx1;
        stack[stackCount++] = dy1;
        stack[stackCount++] = dx2;
        stack[stackCount++] = dy2;
        stack[stackCount++] = dy3;
        lastOpCode = Type2CStringOps.VVCURVETO;
        break; }
  }

  private void addHVCurveTo(double dx1, double dx2, double dy2, double dy3) {
    ensureStackSpace(4);
    switch (lastOpCode) {
      default:
        stackToCString();
      /* Fall through */
      case 0:
        startOfSeqOp = Type2CStringOps.HVCURVETO;
        /* fall through */

      case Type2CStringOps.VHCURVETO:
        stack[stackCount++] = dx1;
        stack[stackCount++] = dx2;
        stack[stackCount++] = dy2;
        stack[stackCount++] = dy3;
        lastOpCode = Type2CStringOps.HVCURVETO;
        break; }
  }

  private void addHVCurveTo(double dx1, double dx2, double dy2, double dx3, double dy3) {
    ensureStackSpace(5);
    addHVCurveTo(dx1, dx2, dy2, dy3);
    stack[stackCount++] = dx3;
    stackToCString(); // with 5 ops, we have to stop the sequence.
  }


  private void addHHCurveTo(double dx1, double dx2, double dy2, double dx3) {
    ensureStackSpace(4);
    switch (lastOpCode) {
      default:
        stackToCString();
      /* Fall through */
      case Type2CStringOps.HHCURVETO:
      case 0:
        stack[stackCount++] = dx1;
        stack[stackCount++] = dx2;
        stack[stackCount++] = dy2;
        stack[stackCount++] = dx3;
        lastOpCode = Type2CStringOps.HHCURVETO;
        break; }
  }

  private void addHHCurveTo(double dx1, double dy1, double dx2, double dy2, double dx3) {
    ensureStackSpace(5);

    // the 5 arg version of HHcurveTo must begin a sequence (unlike HV/VH where the
    // 5 arg version must end a sequence).
    switch (lastOpCode) {
      default:
        stackToCString();
      /* Fall through */
      case 0:
        stack[stackCount++] = dy1; // y before x in the 5 arg version.
        stack[stackCount++] = dx1;
        stack[stackCount++] = dx2;
        stack[stackCount++] = dy2;
        stack[stackCount++] = dx3;
        lastOpCode = Type2CStringOps.HHCURVETO;
        break; }
  }

  private void addRRCurveTo(double dx1, double dy1, double dx2, double dy2, double dx3, double dy3) {
    ensureStackSpace(6);
    switch (lastOpCode) {
      case Type2CStringOps.RLINETO:
        stack[stackCount++] = dx1; 
        stack[stackCount++] = dy1;
        stack[stackCount++] = dx2;
        stack[stackCount++] = dy2;
        stack[stackCount++] = dx3;
        stack[stackCount++] = dy3;
        lastOpCode = Type2CStringOps.RLINECURVE;
        stackToCString();
        break;

      default:
        stackToCString();
      /* Fall through */
      case Type2CStringOps.RRCURVETO:
      case 0:
        stack[stackCount++] = dx1;
        stack[stackCount++] = dy1;
        stack[stackCount++] = dx2;
        stack[stackCount++] = dy2;
        stack[stackCount++] = dx3;
        stack[stackCount++] = dy3;
        lastOpCode = Type2CStringOps.RRCURVETO;
        break; }
  }

  public void curveto (double x1, double y1, double x2, double y2) {
    if (! moveToSeen)  {
      moveto(0,0); }

    curveto (Math.round ((currentX + 2*x1)/3.0), Math.round ((currentY + 2*y1)/3.0), 
        Math.round ((2*x1 + x2)/3.0), Math.round ((2*y1 + y2)/3.0),
        x2, y2);
  }

  public void curveto(double x1, double y1, double x2, double y2, double x3, double y3) {
    if (!moveToSeen) {
      moveto(0,0); }

    double dx1 = x1 - currentX;
    double dx2 = x2 - x1;
    double dx3 = x3 - x2;
    double dy1 = y1 - currentY;
    double dy2 = y2 - y1;
    double dy3 = y3 - y2;

    currentX = x3;
    currentY = y3;

    if (dx1 == 0) {
      if (dy3 == 0) {
        addVHCurveTo(dy1, dx2, dy2, dx3); }
      else if (dx3 == 0) {
        addVVCurveTo(dy1, dx2, dy2, dy3); }
      else {
        addVHCurveTo(dy1, dx2, dy2, dx3, dy3); }}
    else if (dy1 == 0) {
      if (dx3 == 0) {
        addHVCurveTo(dx1, dx2, dy2, dy3); }
      else if (dy3 == 0) {
        addHHCurveTo(dx1, dx2, dy2, dx3); }
      else {
        addHVCurveTo(dx1, dx2, dy2, dx3, dy3); }}
    else if (dx3 == 0) {
      addVVCurveTo(dx1, dy1, dx2, dy2, dy3); }
    else if (dy3 == 0) {
      addHHCurveTo(dx1, dy1, dx2, dy2, dx3); }
    else {
      addRRCurveTo(dx1, dy1, dx2, dy2, dx3, dy3); }
  }

  private void clearMoveTo(double dx, double dy) {
    this.currentX -= dx;
    this.currentY -= dy;
    this.stackCount = 0;
    this.lastOpCode = 0;
  }

   public void moveto(double x, double y) {
    double dx, dy;

    switch (lastOpCode) {
      default:
        stackToCString();
      break;
      case 0:
        break;
      case Type2CStringOps.VMOVETO:
        clearMoveTo(0, stack[0]);
        break;
      case Type2CStringOps.HMOVETO:
        clearMoveTo(stack[0], 0);
        break;
      case Type2CStringOps.RMOVETO:
        clearMoveTo(stack[0], stack[1]);
        break; }

    dx = x - currentX;
    dy = y - currentY;
    currentX = x;
    currentY = y;

    if (dx == 0) {
      stack[stackCount++] = dy;
      lastOpCode = Type2CStringOps.VMOVETO; }
    else if (dy == 0) {
      stack[stackCount++] = dx;
      lastOpCode = Type2CStringOps.HMOVETO; }
    else {
      stack[stackCount++] = dx;
      stack[stackCount++] = dy;
      lastOpCode = Type2CStringOps.RMOVETO; }

    moveToSeen = true;
   }

   public void closepath() {
     // noop. a moveto is expected next. the moveto is from the currentpoint
     // prior to this closepath.
   }

   private void addGeneralFlex(double depth, double x1, double y1, double x2, double y2,
       double x3, double y3, double x4, double y4, double x5, double y5,
       double x6, double y6)  {
     stack[stackCount++] = x1 - currentX; 
     stack[stackCount++] = y1 - currentY;
     stack[stackCount++] = x2 - x1; 
     stack[stackCount++] = y2 - y1;
     stack[stackCount++] = x3 - x2; 
     stack[stackCount++] = y3 - y2;
     stack[stackCount++] = x4 - x3; 
     stack[stackCount++] = y4 - y3;
     stack[stackCount++] = x5 - x4; 
     stack[stackCount++] = y5 - y4;
     stack[stackCount++] = x6 - x5; 
     stack[stackCount++] = y6 - y5;
     stack[stackCount++] = depth;
     lastOpCode = Type2CStringOps.ESCAPE;
     stackToCString();
     lastOpCode = Type2CStringOps.ESC_FLEX;
     stackToCString();

     currentX = x6;
     currentY = y6;
   }

   public void flex(double depth, double x1, double y1, double x2, double y2,
       double x3, double y3, double x4, double y4, double x5, double y5,
       double x6, double y6) {

     boolean horizontalFlex = true;

     if (lastOpCode != 0) {
       stackToCString(); }

     if (depth != STANDARD_FLEX_DEPTH) {
       addGeneralFlex(depth, x1, y1, x2, y2, x3, y3, x4, y4, x5, y5, x6, y6);
       return; }
     else if (currentY == y6)  {
       if (y2 == y3 && y3 == y4) { // can we use hflex/hflex1?
         if (currentY == y1 && y5 == y6) { // can we use hflex?
           // we can use hflex
           stack[stackCount++] = x1 - currentX; 
           stack[stackCount++] = x2 - x1; 
           stack[stackCount++] = y2 - y1;
           stack[stackCount++] = x3 - x2; 
           stack[stackCount++] = x4 - x3; 
           stack[stackCount++] = x5 - x4; 
           stack[stackCount++] = x6 - x5; 
           lastOpCode = Type2CStringOps.ESCAPE;
           stackToCString();
           lastOpCode = Type2CStringOps.ESC_HFLEX;
           stackToCString(); }
         else {
           // we have to fall back to hflex1
           stack[stackCount++] = x1 - currentX; 
           stack[stackCount++] = y1 - currentY;
           stack[stackCount++] = x2 - x1; 
           stack[stackCount++] = y2 - y1;
           stack[stackCount++] = x3 - x2; 
           stack[stackCount++] = x4 - x3; 
           stack[stackCount++] = x5 - x4; 
           stack[stackCount++] = y5 - y4;
           stack[stackCount++] = x6 - x5; 
           lastOpCode = Type2CStringOps.ESCAPE;
           stackToCString();
           lastOpCode = Type2CStringOps.ESC_HFLEX1;
           stackToCString(); }

         currentX = x6;
         currentY = y6;
         return; }}
     else if (currentX == x6) {
       horizontalFlex = false; }
     else {
       addGeneralFlex(depth, x1, y1, x2, y2, x3, y3, x4, y4, x5, y5, x6, y6);
       return; }

     // ensure that the parser can correctly determine if the last arg is dx or dy
     if ((Math.abs(x5 - currentX) > Math.abs(y5 - currentY)) != horizontalFlex) {
       addGeneralFlex(depth, x1, y1, x2, y2, x3, y3, x4, y4, x5, y5, x6, y6);
       return; }

     // we can use flex1
     stack[stackCount++] = x1 - currentX; 
     stack[stackCount++] = y1 - currentY;
     stack[stackCount++] = x2 - x1; 
     stack[stackCount++] = y2 - y1;
     stack[stackCount++] = x3 - x2; 
     stack[stackCount++] = y3 - y2;
     stack[stackCount++] = x4 - x3; 
     stack[stackCount++] = y4 - y3;
     stack[stackCount++] = x5 - x4; 
     stack[stackCount++] = y5 - y4;
     stack[stackCount++] = horizontalFlex? x6 - x5: y6 - y5;
     lastOpCode = Type2CStringOps.ESCAPE;
     stackToCString();
     lastOpCode = Type2CStringOps.ESC_FLEX1;
     stackToCString();

     currentX = x6;
     currentY = y6;
   }

   public void endchar() {
     switch (lastOpCode) {
       default:
         stackToCString();
       break;

       case 0:
       case Type2CStringOps.VMOVETO:
       case Type2CStringOps.HMOVETO:
       case Type2CStringOps.RMOVETO:
         stackCount = 0;
         lastOpCode = 0;
         break; }

     lastOpCode = Type2CStringOps.ENDCHAR;
     stackToCString();
     newHintMask();
     newCntrMask();
   }

   private CharStrings createIndexFromCharstrings()
   throws InvalidFontException, UnsupportedFontException {
     int i;
     int totalSize = 1; // charstring offsets are from 1 before the data.
     int offsetSize;

     // TODO - reorder so that sizes can be calculated and the underlying
     // arrays can be built more efficiently
     CFFByteArrayBuilder builder = CFFByteArray.getCFFByteArrayBuilderInstance();

     builder.addCard16(glyphs.length); // number of charstrings

     // calculate offset size
     for (i = 0; i < glyphs.length; i++) {
       totalSize += glyphs[i].finalcstr.cstrLen; }

     if (totalSize > 0x00ffffff) {
       offsetSize = 4; }
     else if (totalSize > 0x0000ffff) {
       offsetSize = 3; }
     else if (totalSize > 0x000000ff) {
       offsetSize = 2; }
     else {
       offsetSize = 1; }

     // put in the offset size
     builder.addCard8(offsetSize);

     // put in the offsets
     totalSize = 1;
     for (i = 0; i < glyphs.length; i++) {
       builder.addOffset(offsetSize, totalSize);
       totalSize += glyphs[i].finalcstr.cstrLen; }

     // add the extra offset so the last cstring len can be computed        
     builder.addOffset(offsetSize, totalSize); 

     // put in the charstrings
     for (i = 0; i < glyphs.length; i++) {
       builder.addBytes(glyphs[i].finalcstr.cstr, 0, glyphs[i].finalcstr.cstrLen); }

     // get CFFByteArray
     return new CharStrings(builder.toCFFByteArray(), 0);
   }

   /** 
    * All of the charstrings have been run. Generate and return the resulting index. 
    */
   public CharStrings getCharstringIndex()
   throws InvalidFontException, UnsupportedFontException {
     int i;
     for (i = 0; i < glyphs.length; i++) {
       glyphs[i].generateFullCharstring(defaultWidth, nominalWidth, subrize); }

     return createIndexFromCharstrings();
   }

   private int numsize(double r) {
     int i = (int)r;
     if (i == r) {
       /* Integer */
       if (-107 <= i && i <= 107)
         return 1;
       else if (-1131 <= i && i <= 1131)
         return 2;
       else
         return 3; }
     else
       /* Fractional */
       return 5;
   }

   private void analyzeWidths(int fontDictionary) {
     List frequencies = new ArrayList();
     WidthFrequency thisWidth = new WidthFrequency();
     WidthComparator c = new WidthComparator();

     // find all of the widths used.
     for (int i = 0; i < glyphs.length; i++) {
       if (glyphs[i].fd == fontDictionary) {
         thisWidth.width = glyphs[i].width;
         int index = Collections.binarySearch(frequencies, thisWidth, c);

         if (index >= 0)
           ((WidthFrequency)frequencies.get(index)).count++;
         else
           frequencies.add(-index - 1, new WidthFrequency(glyphs[i].width)); }}

     if (frequencies.size() > 0) {
       if (frequencies.size() == 1) {
         defaultWidth[fontDictionary] = ((WidthFrequency)frequencies.get(0)).width; }
       else {
         int j;
         int defaultSize = 0;
         int minSize;
         int indexOfNW = 0;
         int indexOfDW = 0;
         int dictSize;
         double dflt;
         double nominal;

         // what is the size using the default nominal/default widths?
         for (j = 0; j < frequencies.size(); j++) {
           WidthFrequency width = (WidthFrequency)frequencies.get(j);
           if (width.width != DEFAULT_DEFAULT_WIDTH) {
             defaultSize += numsize(width.width - DEFAULT_NOMINAL_WIDTH) * width.count; }}

         minSize = defaultSize;

         // look for something better than the default..
         for (j = 0; j < frequencies.size(); j++) {
           int k;
           int nomsize;

           // make charstrings with this width have a width containing the
           // smallest possible single byte number. All widths larger will be pulled
           // as much as possible toward being a single byte.
           double nomwidth = ((WidthFrequency)frequencies.get(j)).width + 107;

           /* Compute total size for this nominal width with the default defaultwidth */
           nomsize = 0;
           for (k = 0; k < frequencies.size(); k++) {
             WidthFrequency rec = (WidthFrequency)frequencies.get(k);
             nomsize += numsize(rec.width - nomwidth) * rec.count; }

           /* Try this nominal width against all reasonable non-default defaultwidths */
           for (k = 0; k < frequencies.size(); k++) {
             WidthFrequency rec = (WidthFrequency)frequencies.get(k);
             int totsize = nomsize - numsize(rec.width - nomwidth) * rec.count;
             if (totsize < minSize) {
               minSize = totsize;
               indexOfNW = j;
               indexOfDW = k; }}}

         dflt = ((WidthFrequency)frequencies.get(indexOfDW)).width;
         nominal = ((WidthFrequency)frequencies.get(indexOfNW)).width + 107;

         /* Compute size of dictionary entries */
         dictSize = 0;
         if (dflt != 0) {
           dictSize += numsize(dflt) + Dict.Key.defaultWidthX.opCodeLength(); }
         if (nominal != 0) {
           dictSize += numsize(nominal) + Dict.Key.nominalWidthX.opCodeLength(); }

         if (minSize + dictSize < defaultSize) {
           /* We save something by using these values. So hang onto them. */
           defaultWidth[fontDictionary] = dflt;
           nominalWidth[fontDictionary] = nominal; }}}

     widthsAnalyzed[fontDictionary] = true;
   }

   /** 
    * All of the charstrings have been run. Calculate and return the optimal nominal width 
    */
   public double calculateNominalWidth(int fontDictionary) {
     if (!widthsAnalyzed[fontDictionary])
       analyzeWidths(fontDictionary);
     return nominalWidth[fontDictionary];
   }

   /** 
    * All of the charstrings have been run. Calculate and return the optimal default width 
    */
   public double calculateDefaultWidth(int fontDictionary) {
     if (!widthsAnalyzed[fontDictionary])
       analyzeWidths(fontDictionary);
     return defaultWidth[fontDictionary];
   }

}
