/*
 * File: Base.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

final public class Base extends Table {
  /** Gives access to a 'BASE' table. 
   *
   * <h4>Version handling</h4>
   * 
   * <p>'BASE' tables have a major/minor version number.
   * This implementation:
   * <ul> 
   * <li>fully supports version 1.0 tables,</li>
   * <li>interprets 1.x (x>1) tables as 1.0 tables,</li>
   * <li>rejects other versions with an <code>UnsupportedFontException</code> 
   * at construction time.</li>
   * </ul>
   * 
   * <h4>Synchronization</h4>
   * 
   * <p>Like all tables, these objects are immutable.</p> 
  */
  protected Base (FontByteArray buffer) 
      throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);
    
    int majorVersion = getTableMajorVersion ();
    if (majorVersion != 1) {
      throw new UnsupportedFontException 
          ("'BASE' tables with major version " 
           + majorVersion + "are not supported"); }
  }
  
  /** Get the major version of the table. */
  public int getTableMajorVersion () throws InvalidFontException {
    return this.data.getuint16 (0);
  }
  
  /** Get the minor version of the table. */
  public int getTableMinorVersion () throws InvalidFontException {
    return this.data.getuint16 (2);
  }

  /** Get the position of a baseline.
   * @return Integer.MAX_VALUE if the baseline is not present.
   */
  public int getBaselinePosition (Orientation o, int scriptTag, int baselineTag)
      throws InvalidFontException {
    int axisTableOffset = this.data.getOffset (0, o == Orientation.HORIZONTAL ? 4 : 6);
    if (axisTableOffset == 0) {
      return Integer.MAX_VALUE; }
    
    int baseTagListOffset = this.data.getOffset (axisTableOffset, 0);
    if (baseTagListOffset == 0) {
      return Integer.MAX_VALUE; }
    
    int baselineIndex = -1;
    int baseTagCount = this.data.getuint16 (baseTagListOffset);
    for (int b = 0; b < baseTagCount; b++) {
      if (this.data.getuint32 (baseTagListOffset + 2 + 4*b) == baselineTag) {
        baselineIndex = b;
        break; }}
    
    if (baselineIndex == -1) {
      return Integer.MAX_VALUE; }
        
    int baseScriptListOffset = this.data.getOffset (axisTableOffset, 2);
    
    int baseScriptCount = this.data.getuint16 (baseScriptListOffset);
    int baseScriptIndex = -1;
    for (int s = 0; s < baseScriptCount; s++) {
      if (Tag.script_DFLT == scriptTag || this.data.getuint32 (baseScriptListOffset + 2 + 6*s) == scriptTag) {
        baseScriptIndex = s; 
        break; }
      else if (baseScriptIndex == -1 && this.data.getuint32 (baseScriptListOffset + 2 + 6*s) == Tag.script_DFLT) {
        baseScriptIndex = s;
        break; }}
    
    if (baseScriptIndex == -1) {
      return Integer.MAX_VALUE; }
    
    int baseScriptOffset = this.data.getOffset (baseScriptListOffset, 2 + 6*baseScriptIndex + 4); 
    
    int baseValuesOffset = this.data.getOffset (baseScriptOffset, 0);
    if (baseValuesOffset == 0) {
      return Integer.MAX_VALUE; }
    
    int baseCoordOffset = this.data.getOffset (baseValuesOffset, 4 + 2*baselineIndex);
    return this.data.getint16 (baseCoordOffset + 2);
  }

  private boolean baseCoordNeedsRegenerated(int bcOffset) throws InvalidFontException
  {
	  return bcOffset != 0 && this.data.getuint16(bcOffset) == 2;
  }
  
  private boolean minMaxNeedsRegenerated(int minMaxOffset) throws InvalidFontException
  {
	  if (minMaxOffset == 0)
		  return false;
	  
	  if (baseCoordNeedsRegenerated(this.data.getOffset(minMaxOffset, 0)))
		  return true;
	  
	  if (baseCoordNeedsRegenerated(this.data.getOffset(minMaxOffset, 2)))
		  return true;
	  
	  int featMinMaxCount = this.data.getuint16(minMaxOffset + 4);
	  for (int i = 0; i < featMinMaxCount; i++) 
	  {
		  if (baseCoordNeedsRegenerated(this.data.getOffset(minMaxOffset, 6 + 8*i + 4)))
			  return true;
		  if (baseCoordNeedsRegenerated(this.data.getOffset(minMaxOffset, 6 + 8*i + 6)))
			  return true;
	  }
	  
	  return false;
  }
  
  private boolean axisNeedsRegenerated(int axisOffset) throws InvalidFontException
  {
	  if (axisOffset == 0) 
		  return false;
	  
	  int baseScriptListOffset = this.data.getOffset(axisOffset, 2);
	  int baseScriptCount = this.data.getuint16(baseScriptListOffset);
	  for (int i = 0; i < baseScriptCount; i++)
	  {
		  int baseScriptOffset = this.data.getOffset(baseScriptListOffset, 2 + 6*i + 4);
		  int baseValuesOffset = this.data.getOffset(baseScriptOffset, 0);
		  if (baseValuesOffset != 0)
		  {
			  int baseCoordCount = this.data.getuint16(baseValuesOffset + 2);
			  for (int j = 0; j < baseCoordCount; j++)
			  {
				  int baseCoordOffset = this.data.getOffset(baseValuesOffset, 4 + 2*j);
				  if (baseCoordNeedsRegenerated(baseCoordOffset))
					  return true;
			  }
		  }
		  
		  int defaultMinMaxOffset = this.data.getOffset(baseScriptOffset, 2);
		  if (minMaxNeedsRegenerated(defaultMinMaxOffset)) 
		  {
			  return true;
		  }
		  
		  int baseLangSysCount = this.data.getuint16(baseScriptOffset + 4);
		  for (int j = 0; j < baseLangSysCount; j++)
		  {
			  int minMaxOffset = this.data.getOffset(baseScriptOffset, 6 + 6*j + 4);
			  if (minMaxNeedsRegenerated(minMaxOffset))
				  return true;
		  }
	  }
	  
	  return false;
  }
  
  boolean baseNeedsRegenerated() throws InvalidFontException 
  {
	  return (axisNeedsRegenerated(this.data.getOffset(0, 4)) || axisNeedsRegenerated(this.data.getOffset(0, 6)));
  }
  
  private int generateBaseCoord(OTByteArrayBuilder newData, int origBCOffset, int newBCOffset) throws InvalidFontException
  {
	  newData.ensureCapacity(newBCOffset+4);
	  newData.setuint16(newBCOffset, 1); // we always only write out format 1.
	  newData.setint16(newBCOffset+2, this.data.getint16(origBCOffset+2));
	  return 4;
  }
  
  private int generateMinMax(OTByteArrayBuilder newData, int origMinMaxOffset, int newMinMaxOffset) throws InvalidFontException
  {
	  int mmCount = this.data.getuint16(origMinMaxOffset + 4);
	  int mmSize = 6 + 8*mmCount;
	  newData.ensureCapacity(newMinMaxOffset + mmSize);
	  
	  int minOffset = this.data.getOffset(origMinMaxOffset, 0);
	  if (minOffset != 0)
	  {
		  newData.setuint16(newMinMaxOffset, mmSize);
		  mmSize += generateBaseCoord(newData, minOffset, newMinMaxOffset + mmSize);
	  } else
	  {
		  newData.setuint16(newMinMaxOffset, 0);
	  }
	  

	  int maxOffset = this.data.getOffset(origMinMaxOffset, 2);
	  if (maxOffset != 0)
	  {
		  newData.setuint16(newMinMaxOffset+2, mmSize);
		  mmSize += generateBaseCoord(newData, maxOffset, newMinMaxOffset + mmSize);
	  } else
	  {
		  newData.setuint16(newMinMaxOffset + 2, 0);
	  }
	  
	  for (int i = 0; i < mmCount; i++)
	  {
		  for (int j = 0; j < 4; j++)
		  {
			  newData.setuint8(newMinMaxOffset + 6 + 8*i + j, this.data.getuint8(origMinMaxOffset + 6 + i*8 + j));
		  }
		  
		  minOffset = this.data.getOffset(origMinMaxOffset, 6 + 8*i + 4);
		  if (minOffset != 0)
		  {
			  newData.setuint16(newMinMaxOffset+ 6 + 8*i + 4, mmSize);
			  mmSize += generateBaseCoord(newData, minOffset, newMinMaxOffset + mmSize);
		  } else
		  {
			  newData.setuint16(newMinMaxOffset + 6 + 8*i + 4, 0);
		  }
		  
		  maxOffset = this.data.getOffset(origMinMaxOffset, 6 + 8*i + 6);
		  if (maxOffset != 0)
		  {
			  newData.setuint16(newMinMaxOffset+ 6 + 8*i + 6, mmSize);
			  mmSize += generateBaseCoord(newData, maxOffset, newMinMaxOffset + mmSize);
		  } else
		  {
			  newData.setuint16(newMinMaxOffset + 6 + 8*i + 6, 0);
		  }
	  }
	  
	  return mmSize;
  }
  
  private int generateBaseValues(OTByteArrayBuilder newData, int origBaseValuesOffset, int newBaseValuesOffset) throws InvalidFontException
  {
	  int bcCount = this.data.getuint16(origBaseValuesOffset + 2);
	  int bvSize = 4 + 2*bcCount;
	  newData.ensureCapacity(newBaseValuesOffset + bvSize);
	  
	  newData.setuint16(newBaseValuesOffset, this.data.getuint16(origBaseValuesOffset)); // Default index
	  newData.setuint16(newBaseValuesOffset+2, bcCount); // BaseCoordCount
	  for (int i = 0; i < bcCount; i++)
	  {
		  newData.setuint16(newBaseValuesOffset+4 + 2*i, bvSize);
		  bvSize += generateBaseCoord(newData, this.data.getOffset(origBaseValuesOffset, 4 + 2*i), newBaseValuesOffset + bvSize);
	  }
	  
	  return bvSize;
  }
  
  private int generateBaseScript(OTByteArrayBuilder newData, int newBaseScriptOffset, int origBaseScriptOffset) throws InvalidFontException
  {
	  int baseLangSysCount = this.data.getuint16(origBaseScriptOffset + 4);
	  int baseScriptSize = 6 + 6*baseLangSysCount;
	  
	  // Make sure we can hold this BaseScript table.
	  newData.ensureCapacity(newBaseScriptOffset + baseScriptSize);
	  
	  // Regenerate the BaseValues table.
	  int origBaseValuesOffset = this.data.getOffset(origBaseScriptOffset, 0);
	  if (origBaseValuesOffset != 0)
	  {
		  newData.setuint16(newBaseScriptOffset, baseScriptSize);
	  	  baseScriptSize += generateBaseValues(newData, origBaseValuesOffset, newBaseScriptOffset+baseScriptSize);
	  }  else
		  newData.setuint16(newBaseScriptOffset, 0);
	  
	  // Regenerate the DefaultMinMax table.
	  int origDefaultMinMaxOffset = this.data.getOffset(origBaseScriptOffset, 2);
	  if (origDefaultMinMaxOffset != 0)
	  {
		  newData.setuint16(newBaseScriptOffset + 2, baseScriptSize);
		  baseScriptSize += generateMinMax(newData, origDefaultMinMaxOffset, newBaseScriptOffset + baseScriptSize);
	  } else
		  newData.setuint16(newBaseScriptOffset+2, 0);
	  
	  newData.setuint16(newBaseScriptOffset+4, baseLangSysCount);
	  
	  // Regenerate the BaseLangSys records.
	  for (int i = 0; i < baseLangSysCount; i++)
	  {
		  // Copy the tag.
		  for (int j = 0; j < 4; j++)
		  {
			  newData.setuint8(newBaseScriptOffset + 6 + 6*i + j, this.data.getuint8(origBaseScriptOffset + 6 + i*6 + j));
		  }
		  
		  newData.setuint16(newBaseScriptOffset + 6 + 6*i + 4, baseScriptSize);
		  baseScriptSize += generateMinMax(newData, this.data.getOffset(origBaseScriptOffset, 6 + 6*i + 4), newBaseScriptOffset + baseScriptSize);
	  }
	  
	  return baseScriptSize;
  }
  
  private int generateAxis(OTByteArrayBuilder newData, int originalAxisOffset, int newAxisOffset) throws InvalidFontException
  {
	  int origBaseTagListOffset = this.data.getOffset(originalAxisOffset, 0);
	  int origBaseScriptListOffset = this.data.getOffset(originalAxisOffset, 2);
	  int axisSize = 4;
	  
	  newData.ensureCapacity(newAxisOffset + axisSize);
	  if (origBaseTagListOffset != 0)
	  {
		  newData.setuint16(newAxisOffset, 4); // new BaseTagList offset
		  int baseTagCount = this.data.getuint16(origBaseTagListOffset);
		  int baseTagListSize = 2 + baseTagCount*4;
		  axisSize += baseTagListSize;
		  newData.ensureCapacity(newAxisOffset + axisSize);
		  for (int i = 0; i < baseTagListSize; i++)
		  {
			  newData.setuint8(newAxisOffset + 4 + i, this.data.getuint8(origBaseTagListOffset + i));
		  }
	  } else
		  newData.setuint16(newAxisOffset, 0);
	  
	  if (origBaseScriptListOffset != 0)
	  {
		  int baseScriptCount = this.data.getuint16(origBaseScriptListOffset);
		  int newBaseScriptListOffset = newAxisOffset + axisSize;
		  int lastBaseScriptSize = 0;
		  int lastBaseScriptOffset = 0;
		  newData.ensureCapacity(newBaseScriptListOffset + baseScriptCount*6 + 2);
		  axisSize += baseScriptCount*6 + 2;

		  newData.setuint16(newAxisOffset + 2, newBaseScriptListOffset-newAxisOffset);
		  newData.setuint16(newBaseScriptListOffset, baseScriptCount);
		  
		  for (int i = 0; i < baseScriptCount; i++)
		  {
			  int j;
			  
			  for (j = 0; j < 4; j++) // copy the tags.
			  {
				  newData.setuint8(newBaseScriptListOffset + 2 + i*6 + j, this.data.getuint8(origBaseScriptListOffset + 2 + i*6 + j));
			  }
			  
			  // does this share the offset of a script that we've already regenerated? If so, share it in the output.

			  int currOffset = this.data.getOffset(origBaseScriptListOffset, 2 + i*6 + 4);
			  for (j = 0; j < i; j++)
			  {
				  int prevOffset = this.data.getOffset(origBaseScriptListOffset, 2 + j*6 + 4);
				  if (prevOffset == currOffset)
				  {
					  break;
				  }
			  }
			  
			  if (j == i)
			  {
				  newData.setuint16(newBaseScriptListOffset + 2 + i*6 + 4, 2 + baseScriptCount*6 + lastBaseScriptOffset + lastBaseScriptSize);
				  
				  lastBaseScriptOffset += lastBaseScriptSize;
				  lastBaseScriptSize = generateBaseScript(newData, newAxisOffset + axisSize, this.data.getOffset(origBaseScriptListOffset, 2 + i*6 + 4));
				  
				  axisSize += lastBaseScriptSize;
			  } else {
				  newData.setuint16(newBaseScriptListOffset + 2 + i*6 + 4, newData.getuint16(newBaseScriptListOffset + 2 + j*6 + 4));
			  }
		  }
	  } else
		  newData.setuint16(newAxisOffset + 2, 0);
	  
	  return axisSize;

  }
  
  OTByteArrayBuilder regenerateBaseTable() throws InvalidFontException
  {
	  OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance();
	  newData.ensureCapacity(8);
	  newData.setuint32(0, 0x00010000); // version
	  
	  int originalHorizontalOffset = this.data.getOffset(0, 4);
	  int horizontalOffset = originalHorizontalOffset == 0 ? 0: 8;
	  newData.setuint16(4, horizontalOffset); // start the horizontal axis at offset 8 if there is one.
	  int horizontalAxisSize = 0;
	  if (horizontalOffset != 0)
	  {
		  horizontalAxisSize = generateAxis(newData, originalHorizontalOffset, horizontalOffset);
	  }
	  
	  // write the offset to the vertical axis.
	  int originalVerticalOffset = this.data.getOffset(0, 6);
	  if (originalVerticalOffset == 0)
		  newData.setuint16(6, 0);
	  else {
		  newData.setuint16(6, horizontalAxisSize + 8);
		  generateAxis(newData, originalVerticalOffset, horizontalAxisSize + 8);
	  }
	  
	  return newData;
  }
  
  public void subsetAndStream(Subset subset, Map tables) throws InvalidFontException 
  {
	  OTByteArrayBuilder newData;
	  if (baseNeedsRegenerated())
	  { 
		  newData = regenerateBaseTable();
	  } else
	  {
		  // just copy the original table.
		  newData = this.getDataAsByteArray();
	  }

	  tables.put (new Integer (Tag.table_BASE), newData); 
  }
  
  public void stream(Map tables) {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_BASE), newData);
  }
}
