package com.adobe.fontengine.font.opentype;

import java.util.Arrays;
import java.util.Map;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;
import com.adobe.fontengine.font.opentype.LayoutTableSubsetter.ClassCoveredBySubset;
import com.adobe.fontengine.font.opentype.LayoutTableSubsetter.ClassDefGenerator;
import com.adobe.fontengine.font.opentype.LookupTableSubsetter.LookupSubset;
import com.adobe.fontengine.font.opentype.LookupTableSubsetter.LookupSubtableGenerator;
import com.adobe.fontengine.font.opentype.LookupTableSubsetter.NewCoverage;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

class ContextualGenerator {
	private static final boolean warnAboutDroppedSublookups = false;
	
	static void gatherCoveragesForSubtable(LayoutTable origTable, int stOffset, int lookupType, int lookupIndex, Integer subtableIndex, 
			Map coverageMapToUse, Subset subset) throws InvalidFontException, UnsupportedFontException 
	{
	  int stFormat = origTable.data.getuint16(stOffset);
	  switch (stFormat) {
	  case 1:
	  case 2:
	  {
		  int coverageOffset = origTable.data.getOffset (stOffset, 2);
		  LookupTableSubsetter.addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		  break;
	  }
	  case 3:
	  {
		  int coverageCount = origTable.data.getuint16(stOffset + 2);
		  for (int j = 0; j < coverageCount; j++)
		  {
			  int coverageOffset = origTable.data.getOffset (stOffset, 6 + 2*j);
			  LookupTableSubsetter.addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		  }
		  break;
	  }
	  default:
		  throw new InvalidFontException ("Invalid contextual subtable format (" + stFormat + ")"); 
	  }
	}
	
	static void patchSubtableCoverage(LayoutTable origTable, OTByteArrayBuilder builder, int origSTOffset, 
			int newSTOffset, Map coverageInfo, int lookupType) throws InvalidFontException
	{
		int format = origTable.data.getuint16(origSTOffset);
		switch (format)
		{
		case 1:
		case 2:
			LookupTableSubsetter.patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 2, coverageInfo);
			break;
		case 3:
			int numCoverages = origTable.data.getuint16(origSTOffset + 2);
			for (int i = 0; i < numCoverages; i++)
			{
				LookupTableSubsetter.patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 6 + 2*i, coverageInfo);
			}
			break;
		default:
				throw new InvalidFontException("Unrecognized lookup type 5 format: " + format);	
		}
	}
	  
	  
	static LookupSubtableGenerator newContextualGenerator(LayoutTable origTable, OTByteArrayBuilder builder,
			int origSTOffset, int newSTOffset, Subset subset, Map newCoverages, LookupSubset lookupSubset, int origNumGlyphs)
	throws InvalidFontException, UnsupportedFontException
	{
  	  int format = origTable.data.getuint16(origSTOffset);
	  switch (format)
	  {
	  case 1: {
		  return ContextualFormat1Generator.newInstance(origSTOffset, newSTOffset, origTable, subset, builder, newCoverages,lookupSubset);
	  }
		  
	  case 2: {
		  return ContextualFormat2Generator.newInstance(origTable, subset, builder, newSTOffset, origSTOffset, lookupSubset, origNumGlyphs);
	  }
	  case 3: {
		  return ContextualFormat3Generator.newInstance(origTable, builder, origSTOffset, newSTOffset, newCoverages, lookupSubset);
	  }
	  default:
		  throw new InvalidFontException("Invalid contextual lookup format: " + format);
	  }
	}
	
	
	static private class ContextualFormat1Generator extends SetGenerator implements CoverageConsumer
	{
		private final LookupSubset lookupSubset;

		private ContextualFormat1Generator(LayoutTable origTable, Subset subset, OTByteArrayBuilder newData, int newSTOffset, int origSTOffset, int subRuleSetCount, LookupSubset lookupSubset) throws InvalidFontException
		{
			super(origTable, subset, newData, newSTOffset, origSTOffset, subRuleSetCount, 4, false, true);
			writeHeader(newSTOffset, subRuleSetCount);
			
			this.lookupSubset = lookupSubset;
		}
		

		private void writeHeader(int newSTOffset, int glyphCount)
		{
			builder.ensureCapacity(newSTOffset + 6 + 2*glyphCount);
			builder.setuint16(newSTOffset, 1);
			builder.setuint16(newSTOffset + 4, glyphCount);
		}
		
		static ContextualFormat1Generator newInstance(int origSTOffset, int newSTOffset, LayoutTable origTable, Subset subset, OTByteArrayBuilder builder, Map newCoverages, LookupSubset lookupSubset) 
		throws InvalidFontException, UnsupportedFontException
		{
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			NewCoverage newCoverage = (NewCoverage)newCoverages.get(new Integer(coverageOffset));
			ContextualFormat1Generator consumer = new ContextualFormat1Generator(origTable, subset, builder, newSTOffset, origSTOffset, newCoverage.glyphCount,lookupSubset);
			
			origTable.iterateCoverage(coverageOffset, subset, consumer);
			return consumer;
		}
		
		boolean[] computeMembersToKeep(int newRuleSetOffset, int origRuleSetOffset) throws InvalidFontException, UnsupportedFontException
		{
			int origRuleCount = origTable.data.getuint16(origRuleSetOffset);
			boolean[] keep = new boolean[origRuleCount];
			Arrays.fill(keep, true);
			for (int i = 0; i < origRuleCount; i++)
			{
				int ruleOffset = origTable.data.getOffset(origRuleSetOffset, 2 + 2*i);
				int glyphCount = origTable.data.getuint16(ruleOffset);
				for (int j = 0; j < glyphCount -1; j++)
				{
					// if a subrule requires a glyph that isn't in our subset, drop the subrule.
					if (subset.getExistingSubsetGid(origTable.data.getuint16(ruleOffset + 4 + 2*j)) == -1)
					{
						keep[i] = false;
						break;
					}
				}
				
			}
			
			return keep;
		}
		
		int writeMember(int newSubRuleOffset, int origSubRuleOffset) throws InvalidFontException, UnsupportedFontException
		{
			int glyphCount = origTable.data.getuint16(origSubRuleOffset);
			int substCount = origTable.data.getuint16(origSubRuleOffset + 2);
			int lSize = 4 + 2 * (glyphCount - 1);
			builder.ensureCapacity(newSubRuleOffset + lSize);
			
			// write out input
			builder.setuint16(newSubRuleOffset, glyphCount);
			for (int i = 0; i < glyphCount - 1; i++)
			{
				int gid = subset.getExistingSubsetGid(origTable.data.getuint16(origSubRuleOffset + 4 + i*2));
				builder.setuint16(newSubRuleOffset + 4 + 2*i, gid);
			}
			
			// write out new substitutions
			int newSubstCount = 0;
			for (int i = 0; i < substCount; i++)
			{
				int newLookup = lookupSubset.getExistingSubsetGid(origTable.data.getuint16(origSubRuleOffset + 4 + 2*(glyphCount-1) + i*4 + 2));
				if (newLookup != -1)
				{
					lSize += 4;
					builder.ensureCapacity(newSubRuleOffset + lSize);
					int origValue = origTable.data.getuint16(origSubRuleOffset + 4 + 2*(glyphCount-1) + i*4);
					builder.setuint16(newSubRuleOffset + 4 + 2*(glyphCount-1) + 4*newSubstCount, origValue);
					builder.setuint16(newSubRuleOffset + 4 + 2*(glyphCount-1) + 4*newSubstCount + 2, newLookup);
					newSubstCount++;
				} else if (warnAboutDroppedSublookups) {
					System.err.println("Sublookup dropped for contextual format 1");
				}
			}

			builder.setuint16(newSubRuleOffset + 2, newSubstCount);
			return lSize;
		}

		int whichRuleSetIndexApplies(int gid, int coverageIndex)
		{
			return coverageIndex;
		}

		int getOrigRecordSize() {
			return 0;
		}
	}
	
	static private class ContextualFormat2Generator extends SetGenerator implements CoverageConsumer
	{
		private final LookupSubset lookupSubset;
		private final ClassCoveredBySubset classCovered;
		private final int origClassDefOffset;
		private final int numGlyphs;
		private final int classCount;

		private ContextualFormat2Generator(LayoutTable origTable, Subset subset, OTByteArrayBuilder newData, int newSTOffset, 
				int origSTOffset, int classCount, LookupSubset lookupSubset, int numGlyphs) 
		throws InvalidFontException
		{
			super(origTable, subset, newData, newSTOffset, origSTOffset, classCount, 6, true, true);
			this.lookupSubset = lookupSubset;
			this.classCovered = new ClassCoveredBySubset(subset);
			this.origClassDefOffset = origTable.data.getOffset(origSTOffset,4);
			this.numGlyphs = numGlyphs;
			this.classCount = classCount;
		}
		
		static ContextualFormat2Generator newInstance(LayoutTable origTable, Subset subset, OTByteArrayBuilder builder, int newSTOffset, 
				int origSTOffset, LookupSubset lookupSubset, int numGlyphs) throws InvalidFontException
		{
			int classCount = origTable.data.getuint16(origSTOffset+6);
			ContextualFormat2Generator consumer = new ContextualFormat2Generator(origTable, subset, builder, newSTOffset, origSTOffset, classCount, lookupSubset, numGlyphs);

			return consumer;
		}
		
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException
		{
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			builder.ensureCapacity(newSTOffset + 8 + 2*classCount);
			builder.setuint16(newSTOffset, 2);
			builder.setuint16(newSTOffset + 6, classCount);
			
			origTable.iterateCoverage(coverageOffset, subset, this);
			writeClassDef();
			return this.subtableSize;
		}
		
		boolean[] computeMembersToKeep(int newRuleSetOffset, int origRuleSetOffset) throws InvalidFontException, UnsupportedFontException
		{			
			int origRuleCount = origTable.data.getuint16(origRuleSetOffset);
			boolean[] keep = new boolean[origRuleCount];
			Arrays.fill(keep, true);
			for (int i = 0; i < origRuleCount; i++)
			{
				int ruleOffset = origTable.data.getOffset(origRuleSetOffset, 2 + 2*i);
				int glyphCount = origTable.data.getuint16(ruleOffset);
				for (int j = 0; j < glyphCount -1; j++)
				{
					// if a subrule requires a class that isn't covered by our subset, the subrule isn't kept.
					if (!classCovered.classCoveredBySubset(origTable, origClassDefOffset, numGlyphs, origTable.data.getuint16(ruleOffset + 4 + 2*j)))
					{
						keep[i] = false;
						break;
					}
				}
				
			}
			
			return keep;
		}
		
		int writeMember(int newSubRuleOffset, int origSubRuleOffset) throws InvalidFontException, UnsupportedFontException
		{
			int glyphCount = origTable.data.getuint16(origSubRuleOffset);
			int substCount = origTable.data.getuint16(origSubRuleOffset + 2);
			int lSize = 4 + 2 * (glyphCount - 1);
			builder.ensureCapacity(newSubRuleOffset + lSize);
			
			// write classes
			builder.setuint16(newSubRuleOffset, glyphCount);
			for (int i = 0; i < glyphCount - 1; i++)
			{
				builder.setuint16(newSubRuleOffset + 4 + 2*i, origTable.data.getuint16(origSubRuleOffset + 4 + i*2));
			}
			
			// write substitutions
			int newSubstCount = 0;
			for (int i = 0; i < substCount; i++)
			{
				int lookup = lookupSubset.getExistingSubsetGid(origTable.data.getuint16(origSubRuleOffset + 4 + 2*(glyphCount-1) + i*4 + 2));
				if (lookup != -1)
				{
					lSize += 4;
					builder.ensureCapacity(newSubRuleOffset + lSize);
					
					int origValue = origTable.data.getuint16(origSubRuleOffset + 4 + 2*(glyphCount-1) + i*4);
					builder.setuint16(newSubRuleOffset + 4 + 2*(glyphCount-1) + 4*newSubstCount, origValue);
					builder.setuint16(newSubRuleOffset + 4 + 2*(glyphCount-1) + 4*newSubstCount + 2, lookup);
					newSubstCount++;
				}else if (warnAboutDroppedSublookups) {
					System.err.println("Sublookup dropped for contextual format 2");
				}
			}

			builder.setuint16(newSubRuleOffset + 2, newSubstCount);
			return lSize;
		}
		
		void writeClassDef() throws InvalidFontException, UnsupportedFontException
		{
			// write the class definition table
			ClassDefGenerator generator = ClassDefGenerator.newInstance(origTable, origClassDefOffset, subset, numGlyphs);
			builder.setuint16(this.newSTOffset + 4, subtableSize);
			OTByteArray array = generator.generateClass().toOTByteArray();
			subtableSize += writeByteArrayAtOffset(newSTOffset + subtableSize, array);
		}

		int whichRuleSetIndexApplies(int gid, int coverageIndex) throws InvalidFontException
		{
			return origTable.getClassIndex(gid, origClassDefOffset);
		}

		int getOrigRecordSize() {
			return 0;
		}
	}
	
	static private class ContextualFormat3Generator implements LookupSubtableGenerator
	{
		private final int origSTOffset;
		private final int newSTOffset;
		private final LookupSubset lookupSubset;
		private final LayoutTable origTable;
		private final OTByteArrayBuilder builder;
		
		private ContextualFormat3Generator(LayoutTable origTable, OTByteArrayBuilder builder, int origSTOffset, int newSTOffset, Map newCoverages, LookupSubset lookupSubset)
		{
			this.lookupSubset = lookupSubset;
			this.newSTOffset = newSTOffset;
			this.origSTOffset = origSTOffset;
			this.origTable = origTable;
			this.builder = builder;
		}
		
		static ContextualFormat3Generator newInstance(LayoutTable origTable, OTByteArrayBuilder builder, int origSTOffset, int newSTOffset, Map newCoverages, LookupSubset lookupSubset)
		{
			return new ContextualFormat3Generator(origTable, builder, origSTOffset, newSTOffset, newCoverages, lookupSubset);
		}
		
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException {
			int coverageCount = origTable.data.getuint16(origSTOffset + 2);
			int substCount = origTable.data.getuint16(origSTOffset + 4);
			int size = 6 + 2*coverageCount;
			builder.ensureCapacity(newSTOffset + size);
			builder.setuint16(newSTOffset, 3); // write the format
			builder.setuint16(newSTOffset + 2, coverageCount); // coverage count
			
			//write the new substitutions.
			int newSubstCount = 0;
			for (int i = 0; i < substCount; i++)
			{
				int origValue = origTable.data.getuint16(origSTOffset + 6 + 2*coverageCount + 4*i + 2);
				int newValue = lookupSubset.getExistingSubsetGid(origValue);
				if (newValue != -1)
				{
					size += 4;
					builder.ensureCapacity(newSTOffset + size);
					builder.setuint16(newSTOffset + 6 + 2*coverageCount + 4*newSubstCount, origTable.data.getuint16(origSTOffset + 6 + 2*coverageCount + 4*i));
					builder.setuint16(newSTOffset + 6 + 2*coverageCount + 4*newSubstCount + 2, newValue);
					newSubstCount++;
				}else if (warnAboutDroppedSublookups) {
					System.err.println("Sublookup dropped for contextual format 3");
				}
			}
			builder.setuint16(newSTOffset + 4, newSubstCount); // substitution count
			
			return size;
		}
	}
}
