package com.adobe.fontengine.font.opentype;

import java.util.Arrays;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;


class GdefSubsetter extends LayoutTableSubsetter implements CoverageConsumer {
	private Subset subset;
	private int numLigGlyphs;
	private int origLCLOffset;
	private int[] keepLigGlyph;
	
	GdefSubsetter(Gdef origTable) {
		super(origTable, OTByteArray.getOTByteArrayBuilderInstance(12));
	}
	
	OTByteArrayBuilder subsetAndStream(Subset subset, int origNumGlyphs) 
	throws InvalidFontException, UnsupportedFontException {
		int tableSize = 12;
		
		builder.setuint32(0, 0x00010000);
		
		//		 Glyph Class Def offset
		int origOffset = origTable.data.getOffset(0, 4);
		if (origOffset == 0)
			builder.setuint16(4, 0); 
		else
		{
			builder.setuint16(4, tableSize);
			tableSize += writeClassDef(origOffset, tableSize, subset, origNumGlyphs);
		}
		
		// Attach glyph class offset. This can't work for tt -> cff unless contours are preserved (which they aren't now)
		// CTS doesn't use them anyway, so drop them.
		builder.setuint16(6, 0); 
		
		//		 lig caret list offset
		origOffset = origTable.data.getOffset(0, 8);
		if (origOffset == 0)
			builder.setuint16(8, 0); 
		else
		{
			builder.setuint16(8, tableSize);
			tableSize += writeLigCaretList(origOffset, tableSize, subset);
		}
		
		//		mark attach class def offset
		origOffset = origTable.data.getOffset(0, 10);

		// A bunch of fonts have bad offsets (eg, microsoftFonts\OfficeXP_Fonts\ARIAL.TTF)
		// If the data is obviously bad, just drop it.
		if (origOffset <= 10) 
			builder.setuint16(10, 0); 
		else
		{
			builder.setuint16(10, tableSize);
			writeClassDef(origOffset, tableSize, subset, origNumGlyphs);
		}
 
		return builder;
	}
	
	public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
		if (subset.getExistingSubsetGid(gid) != -1)
		{
			int origLigGlyphOffset = origTable.data.getOffset(origLCLOffset, 4 + 2*coverageIndex);
			int caretCount = origTable.data.getuint16(origLigGlyphOffset);
			for (int i = 0; i < caretCount; i++)
			{
				int caretValueOffset = origTable.data.getOffset(origLigGlyphOffset, 2 + 2*i);
				int format = origTable.data.getuint16(caretValueOffset);
				
				// if we have a format 2 caret value table, we can't keep it so throw out this ligature from the table.
				if (format == 2)
					return true;
				
			}
			this.keepLigGlyph[subset.getExistingSubsetGid(gid)] = coverageIndex;
			numLigGlyphs++;
		}
		return true;
	}
	
	int writeLigCaretList(int origLCLOffset, int newLCLOffset, Subset subset) throws InvalidFontException, UnsupportedFontException
	{
		
		this.numLigGlyphs = 0;
		this.subset = subset;
		this.origLCLOffset = origLCLOffset;
		this.keepLigGlyph = new int[subset.getNumGlyphs()];
		Arrays.fill(keepLigGlyph, -1);
		
		int origCoverageOffset = origTable.data.getOffset(origLCLOffset, 0);
		origTable.iterateCoverage(origCoverageOffset, null, this);

		int lCLSize = 4 + 2*numLigGlyphs;
		builder.ensureCapacity(newLCLOffset + lCLSize);
		
		builder.setuint16(newLCLOffset, lCLSize);
		lCLSize += writeCoverage(origTable.data.getOffset(origLCLOffset, 0), newLCLOffset + lCLSize);
		
		builder.setuint16(newLCLOffset+2, numLigGlyphs);
		
		for (int i = 0, j = 0; i < keepLigGlyph.length; i++)
		{
			if (keepLigGlyph[i] != -1)
			{
				int origCoverageIndex = keepLigGlyph[i];
				builder.setuint16(newLCLOffset + 4 + 2*j, lCLSize);
				lCLSize += writeLigGlyph(origTable.data.getOffset(origLCLOffset, 4 + 2*origCoverageIndex), newLCLOffset + lCLSize);
				j++;
			}
		}

		return lCLSize;
	}
	
	int writeCoverage(int origCoverageOffset, int newCoverageOffset) throws InvalidFontException, UnsupportedFontException
	{
		CoverageGenerator generator = CoverageGenerator.newInstance(origTable, origCoverageOffset, subset);
		OTByteArray newCoverage = generator.generateCoverage().toOTByteArray();
		return writeByteArrayAtOffset(newCoverageOffset, newCoverage);
	}
	
	int writeLigGlyph(int origLGOffset, int newLGOffset) throws InvalidFontException
	{
		int caretCount = origTable.data.getuint16(origLGOffset);
		int lgSize = 2 + 2*caretCount;
		builder.ensureCapacity(newLGOffset + lgSize);
		
		builder.setuint16(newLGOffset, caretCount);
		for (int i = 0; i < caretCount; i++)
		{
			builder.setuint16(newLGOffset + 2 + 2*i, lgSize);
			lgSize += writeCaretValue(origTable.data.getOffset(origLGOffset, 2 + 2*i), newLGOffset + lgSize);
		}
		
		return lgSize;
	}
	
	int writeCaretValue(int origCaretOffset, int newCaretOffset) throws InvalidFontException
	{
		builder.ensureCapacity(newCaretOffset + 4);
		builder.setuint16(newCaretOffset, 1);
		builder.setint16(newCaretOffset + 2, origTable.data.getint16(origCaretOffset + 2));
		return 4;
	}
	
	int writeClassDef(int origClassDefOffset, int newClassDefOffset, Subset subset, int numGlyphs) throws InvalidFontException, UnsupportedFontException
	{
		// write the class definition table
		ClassDefGenerator generator = ClassDefGenerator.newInstance(origTable, origClassDefOffset, subset, numGlyphs);
		OTByteArray array = generator.generateClass().toOTByteArray();
		return writeByteArrayAtOffset(newClassDefOffset, array);
	}
}
