/*
 * File: GlyphNames.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.adobe.fontengine.SparseArray;

/** This class supports PostScript glyph names.*/
final public class GlyphNames {

  /** In memory representation of the AGL. */
  static Object mutex = new Object ();
  static final Map aglNameList;
  static final Map aglNameListMulti;
  static final Map dingbatNameList;
  static final SparseArray charTable;
  static final SparseArray dingbatCharTable;

  /** Load the AGL resource into our in memory representation. */
  static {
    try {
      ObjectInputStream is = new ObjectInputStream(GlyphNames.class.getResourceAsStream ("agnc"));
      aglNameList = (Map) is.readObject();
      aglNameListMulti = (Map) is.readObject();
      charTable = (SparseArray) is.readObject();
      dingbatNameList = (Map) is.readObject();
      dingbatCharTable = (SparseArray) is.readObject (); }
    catch (IOException e) {
      throw new RuntimeException ("cannot load AFE AGNC resources", e); }
    catch (ClassNotFoundException e) {
      throw new RuntimeException ("cannot load AFE AGNC resources", e); }
  }    

  public static String resolveUSVWithAGNCNameTable (int usv) {
    List glyphNameList = (List) charTable.get (usv);
    if (glyphNameList == null) {
      return null; }

    // let's get the first one
    GlyphNameEntry entry = (GlyphNameEntry) glyphNameList.get (0);
    return entry.getName ();
  }

  public static String resolveUSVToAGNCName (int usv) {
    String name = resolveUSVWithAGNCNameTable (usv);
    if (name == null) {
      name = generateUPlusName (usv); }
    return name;
  }

  /**
   * @param usv
   * @return the "name" of the Unicode scalar value in the form 'u' followed by the hex representation (see PDF 1.6 specs)
   */
  public static String generateUPlusName (int usv) {
    if (usv >= 0 && usv <= 0x10FFFF) {
      String number = Integer.toHexString (usv).toUpperCase ();

      switch (number.length ()) {
        case 1: {
          return "u" + "000" + number; }
        case 2: {
          return "u" + "00" + number; }
        case 3: {
          return "u" + "0" + number; }
        default: {
          return "u" + number; }}}
    return null;
  }

  /** Find the mapping of a name via the Adobe Glyph Naming convention. 
   * 
   * @param name the String containing the name to resolve
   * @param isDingbats whether the name is that of the Dingbats font
   * @return List containing the USVs described by the name
   */
  public static List/*<Integer>*/ resolveAGNCName (String name, boolean isDingbats) {    
    List/*<Integer>*/ usvs = new ArrayList/*<Integer>*/ ();

    // Step 1: drop all the characters from the glyph name starting 
    // with the first occurrence of a period (U+002E FULL STOP), if any.
    int firstPeriod = name.indexOf('.');
    if (firstPeriod != -1) {
      name = name.substring(0, firstPeriod); }

    // Step 2: split the remaining string into a sequence of components,
    // using underscore (U+005F LOW LINE) as the delimiter.
    String[] components = name.split("_");

    // Step 3: map each component to a character string according to the
    // procedure below, and concatenate those strings; the result is the
    // character string to which the glyph name is mapped.
    for (int i = 0; i < components.length; i++) {

      // If the font is Zapf Dingbats (Postscript FontName ZapfDingbats),
      // and the component is in the ZapfDingbats list, then map it ot the
      // corresponding character in that list.
      if (isDingbats) {
        if (resolveDingbatsList(components[i], usvs)) {
          continue; }}
      else {
        // otherwise, if the component is in the Adobe Glyph List, then map
        // it to the corresponding character from that list.
        if (resolveGlyphList (components[i], usvs)) {
          continue; } }

      if (!resolveUniName(components[i], usvs)) {
        resolveUPlusName(components[i], usvs); }}

    return usvs;
  }

  /** A convenience method wrapping resolveAGNCName for code that wants
   * arrays and not collection objects. 
   * 
   * @param name the String containing the name to resolve
   * @param isDingbats whether the name is that of the Dingbats font
   * @return array containing the USVs described by the name
   */
  public static int[] resolveAGNCNameIntoArray (String name, boolean isDingbats) {    
    List usvs = resolveAGNCName(name, isDingbats);
    int[] array = new int[usvs.size()];
    for (int i = 0; i < usvs.size(); i++) {
      array[i] = ((Integer) usvs.get(i)).intValue(); }
    return array;
  }

  /** The suffix of the name as per the Adobe Glyph Naming Convention. 
   * 
   * @param name the String containing the name to resolve
   */
  public static String getAGNCNameSuffix (String name) {
    int firstPeriod = name.indexOf('.');
    // no period or period at end of the string
    if (firstPeriod == -1 || firstPeriod == name.length() - 1) {
      return null; }
    return name.substring(firstPeriod + 1, name.length());
  }

  private static boolean resolveUPlusName(String component, List/*<Integer>*/ usvs) {
    if (component.matches("^u(([\\dA-F]{4})|([1-9A-F][\\dA-F]{4})|(1[\\dA-F]{5}))")) {
      usvs.add(new Integer(Integer.parseInt(component.substring(1), 16)));
      return true; }
    return false;
  }

  private static boolean resolveUniName(String component, List/*<Integer>*/ usvs) {
    int added = 0;
    // This component must match the text "uni" followed by multiples of four
    // hexadecimal digits made up only of uppercase letters
    if (component.matches("^uni([\\dA-F]{4})+")) {
      for (int start = 3; start < component.length(); start += 4) {
        int value = Integer.parseInt(component.substring(start, start + 4), 16);
        if ((value >= 0x0000 && value <= 0xD7FF)
            || (value >= 0xE000 && value <=0xFFFF)) {
          usvs.add(new Integer(value));
          added++; }
        else {
          // if we find one out of range then all are invalid - see the spec
          for (int i = 0; i < added; i++) {
            usvs.remove(usvs.size() - 1); }
          break; }}}
    
    return added != 0;
  }

  private static boolean resolveGlyphList(String component, List/*<Integer>*/ usvs) {
    GlyphNameEntry entry = (GlyphNameEntry) aglNameList.get (component);
    if (entry == null) {
      return false; }
    usvs.add (entry.getUSV ());
    return true;
  }

  private static boolean resolveDingbatsList(String component, List/*<Integer>*/ usvs) {
    GlyphNameEntry entry = (GlyphNameEntry) dingbatNameList.get (component);
    if (entry == null) {
      return false; }
    usvs.add (entry.getUSV ());
    return true;
  }

  static protected class GlyphNameEntry implements Serializable {
    /* Serialization signature is explicitly set and should be 
     * incremented on each release to prevent compatibility.
     */
    static final long serialVersionUID = 1;

    protected Integer usv;
    protected String name;

    protected GlyphNameEntry (Integer usv, String name)  {
      this.usv = usv;
      this.name = name;
    }

    protected GlyphNameEntry (int usv, String name) {
      this.usv = new Integer (usv);
      this.name = name;
    }

    protected Integer getUSV () {
      return usv;
    }

    protected String getName () {
      return name;
    }

    public String toString () {
      return "[ " + usv + ", " + name + " ]";
    }
  }

  static protected class GlyphNameMultiEntry implements Serializable {
    /* Serialization signature is explicitly set and should be 
     * incremented on each release to prevent compatibility.
     */
    static final long serialVersionUID = 1;

    protected List usvs;
    protected String name;

    protected GlyphNameMultiEntry (List usvs, String name)  {
      this.usvs = new ArrayList(usvs);
      this.name = name;
    }

    protected List getUSVS () {
      return usvs;
    }

    protected String getName () {
      return name;
    }

    public String toString () {
      return "[ " + usvs + ", " + name + " ]";
    }
  }

	/**
	 * Return the Adobe glyph name list as a HashMap with the glyph name
	 * as its key and the Unicode for that name as its value.
	 */
	public static Map getAdobeGlyphNameList()
	{
		HashMap retVal = new HashMap();
		for (Object key : aglNameList.keySet()) {
			GlyphNameEntry entry = (GlyphNameEntry)aglNameList.get(key);
			retVal.put(key, entry.getUSV());
		}
		return retVal;
	}

	/**
	 * Return the Adobe glyph name list as a HashMap with the glyph name
	 * as its key and the Unicode for that name as its value.
	 */
	public static Map getAdobeGlyphNameListMulti()
	{
		HashMap retVal = new HashMap();
		for (Object key : aglNameListMulti.keySet()) {
			GlyphNameMultiEntry entry = (GlyphNameMultiEntry)aglNameListMulti.get(key);
			retVal.put(key, entry.getUSVS());
		}
		return retVal;
	}

	/**
	 * Return the Dingbat name list as a HashMap with the glyph name
	 * as its key and the Unicode for that name as its value.
	 */
	public static Map getDingbatNameList()
	{
		HashMap retVal = new HashMap();
		for (Object key : dingbatNameList.keySet()) {
			GlyphNameEntry entry = (GlyphNameEntry)dingbatNameList.get(key);
			retVal.put(key, entry.getUSV());
		}
		return retVal;
	}
}

