/*
*
*	File: GposSubsetter.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2008 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font.opentype;

import java.util.Arrays;
import java.util.Map;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.SubsetDefaultImpl;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.ClassConsumer;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

class GposSubsetter extends LookupTableSubsetter {
	private final boolean preserveNonEmptyKernFeature;
	
	GposSubsetter(Gpos origTable, int numGlyphs, boolean preserveNonEmptyKernFeature)
	{
		super(origTable, OTByteArray.getOTByteArrayBuilderInstance(), numGlyphs);
		this.preserveNonEmptyKernFeature = preserveNonEmptyKernFeature;
	}
	
	void gatherCoveragesForSubtable(int stOffset, int lookupType, int lookupIndex, Integer subtableIndex, 
			Map coverages, Map extensionCoverages, Subset subset) 
	throws InvalidFontException, UnsupportedFontException
	{

	      Map coverageMapToUse = coverages;
	      
		  if (lookupType == 9)
		  {
			  lookupType = origTable.data.getuint16(stOffset + 2);
			  stOffset += origTable.data.getuint32asint(stOffset + 4, "Unhandled extension offset");
			  coverageMapToUse = extensionCoverages;
		  }
		  
		  
		  switch (lookupType) {
		      case 1: 
		      case 2: 
		      case 3: { 
		    	  int coverageOffset = origTable.data.getOffset (stOffset, 2);
		    	  addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		    	  break; 
		      }
		      
		      case 4: 
		      case 5: 
		      case 6: { 
		    	  int coverageOffset = origTable.data.getOffset (stOffset, 2);
		    	  addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);

		    	  coverageOffset = origTable.data.getOffset (stOffset, 4);
		    	  addToCoveragesMap(origTable, coverageOffset, coverageMapToUse, lookupIndex, subtableIndex, subset);
		    	  break; 
		      }
		      
		      case 7: 
		    	  ContextualGenerator.gatherCoveragesForSubtable(origTable, stOffset, lookupType, 
		    			  lookupIndex, subtableIndex, coverageMapToUse, subset);
		    	  break;

		      case 8: 
		    	  ChainingGenerator.gatherCoveragesForSubtable(origTable, stOffset, lookupType, 
	    			  lookupIndex, subtableIndex, coverageMapToUse, subset);
		      
		    	  break;

		      default: {
		        throw new InvalidFontException ("Invalid GSUB lookup type (" + lookupType + ")"); }}	    	
	}

	int getExtensionSubtableSize()
	{
		return 8;
	}

	void patchSubtableCoverage(int origSTOffset, int newSTOffset, Map coverageInfo, int lookupType) throws InvalidFontException
	{
		switch (lookupType)
		{
		case 1:
		case 2:
		case 3:
			patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 2, coverageInfo);
			break;
		case 4:
		case 5:
		case 6:
			patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 2, coverageInfo);
			patchCoverageAtOffset(origTable, builder, origSTOffset, newSTOffset, 4, coverageInfo);
			break;
	
		case 7:
			ContextualGenerator.patchSubtableCoverage(origTable, builder, origSTOffset, newSTOffset, coverageInfo, lookupType);
			break;
			
		case 8: 
			ChainingGenerator.patchSubtableCoverage(origTable, builder, origSTOffset, newSTOffset, coverageInfo, lookupType);
			break;
			
		case 9:
			break;
			
		default:
			throw new InvalidFontException("Unrecognized lookup type: " + lookupType);	
		}
	}
	
	/**
	 * Determines whether lookup 1 format 2 subtables can become format 1. 
	 */
	private class L1F2TargetFormatDeterminer implements CoverageConsumer
	{
		private boolean canUseFormat1 = true;
		private final int origSTOffset;
		private final int valueFormatSize;
		int bytes[] = null;
		
		public L1F2TargetFormatDeterminer(int origSTOffset) throws InvalidFontException {
			this.origSTOffset = origSTOffset;
			this.valueFormatSize = ((Gpos)origTable).getValueRecordSize(origTable.data.getuint16(origSTOffset+4))/2;
		}
		
		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			if (bytes == null)
			{
				bytes = new int[valueFormatSize];
				for (int i = 0; i < valueFormatSize; i++)
				{
					bytes[i] = origTable.data.getint16(origSTOffset + 8 + coverageIndex*valueFormatSize*2 + i);
				}
			} else
			{
				for (int i = 0; i < valueFormatSize; i++)
				{
					int value = origTable.data.getint16(origSTOffset + 8 + coverageIndex*valueFormatSize*2 + i);
					if (value != bytes[i])
					{
						canUseFormat1 = false;
						return false;
					}
				}
			}
			
			return true;
		}
		
		boolean canUseFormat1(Subset subset) throws InvalidFontException, UnsupportedFontException {
  		    origTable.iterateCoverage(origTable.data.getOffset(origSTOffset, 2), subset, this);
			return canUseFormat1;
		}
	}
	
	
	private int targetValueFormat(int originalValueFormat)
	{
		return originalValueFormat & 0xf; // if the original format had offsets to device tables, drop them.
	}
	
	// Both the reading and writing of the value records assumes that we will drop all of the unsigned
	// offsets and keep the signed int16s. So all of the reading and writing consistently uses signed...
	private void writeValueFormat(int startingOffset, int origFormat, int targetFormat, int[] value)
	{
		for (int i = 0, j = 0, mask = 0x1; mask <= 0x8 && i < value.length; mask = mask << 1)
		{
			if ((origFormat & mask) != 0)
			{
				if ((targetFormat & mask) != 0)
				{
					builder.setint16(startingOffset + 2*j, value[i]);
					j++;
				}
				i++;
			}
		}
	}
	
	private void writeValueFormatFromOriginal(int origOffset, int startingOffset, int origFormat, int targetFormat) throws InvalidFontException
	{
		for (int i = 0, j = 0, mask = 0x1; mask <= 0x8; mask = mask << 1)
		{
			if ((origFormat & mask) != 0)
			{
				if ((targetFormat & mask) != 0)
				{
					int orig = origTable.data.getint16(origOffset + 2*i);
					builder.setint16(startingOffset + 2*j, orig);
					j++;
				}
				i++;
			}
		}
	}
	
	private int writeAnchorFromOriginal(int origOffset, int newOffset) throws InvalidFontException
	{
		builder.ensureCapacity(newOffset + 6);
		builder.setuint16(newOffset, 1);
		builder.setint16(newOffset + 2, origTable.data.getint16(origOffset+2));
		builder.setint16(newOffset + 4, origTable.data.getint16(origOffset+4));
		return 6;
	}
	
	private int writeLookup1Format1(int origSTOffset, int newSTOffset, int[] value) throws InvalidFontException
	{
		int origFormat = origTable.data.getuint16(origSTOffset + 4);
		int targetFormat = targetValueFormat(origFormat);
		int targetFormatLength = ((Gpos)origTable).getValueRecordSize(targetFormat);
		
		int size = 6 + targetFormatLength;
		builder.ensureCapacity(newSTOffset + size);
		builder.setuint16(newSTOffset, 1); // format
		builder.setuint16(newSTOffset + 4, targetFormat); // value format
	
		writeValueFormat(newSTOffset + 6, origFormat, targetFormat, value);
		
		return size;
	}
	
	private class Type1Format2Generator implements CoverageConsumer, LookupSubtableGenerator
	{
		private final Subset subset;
		private final int newSTOffset;
		private final int origSTOffset;
		int glyphCount = 0;
		private int[] subsetToOrigCoverageIndex;
		
		Type1Format2Generator(int origSTOffset, int newSTOffset, Subset subset) throws InvalidFontException, UnsupportedFontException
		{
			this.subset = subset;
			this.newSTOffset = newSTOffset;
			this.origSTOffset = origSTOffset;
			this.subsetToOrigCoverageIndex = new int[subset.getNumGlyphs()];
			Arrays.fill(this.subsetToOrigCoverageIndex, -1);
			origTable.iterateCoverage(origTable.data.getOffset(origSTOffset, 2), subset, this);

		}
		
		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			int subsetGID = subset.getExistingSubsetGid(gid);
			subsetToOrigCoverageIndex[subsetGID] = coverageIndex;
			glyphCount++;
			return true;
		}
		
		public int writeSubtable() throws InvalidFontException
		{
			int origFormat = origTable.data.getuint16(origSTOffset + 4);
			int targetFormat = targetValueFormat(origFormat);
			int targetFormatLength = ((Gpos)origTable).getValueRecordSize(targetFormat);
			int origFormatLength = ((Gpos)origTable).getValueRecordSize(origFormat);
			
			builder.ensureCapacity(newSTOffset + 8 + targetFormatLength*glyphCount);
			
			builder.setuint16(newSTOffset, 2); // format
			builder.setuint16(newSTOffset + 4, targetFormat); // valueFormat
			builder.setuint16(newSTOffset + 6, glyphCount); //valueCount
			
			for (int i = 0, j = 0; i < subset.getNumGlyphs() && j < glyphCount; i++)
			{
				if (subsetToOrigCoverageIndex[i] != -1)
				{
					writeValueFormatFromOriginal(origSTOffset + 8 + origFormatLength*subsetToOrigCoverageIndex[i],
							newSTOffset + 8 + targetFormatLength*j, origFormat, targetFormat);
					j++;
				}
			}
			
			return 8 + targetFormatLength*glyphCount;
		}

	}

	private class Type2Format1Generator extends SetGenerator implements LookupSubtableGenerator {
		private final int value1Size;
		private final int value2Size;
		private final int value1Format;
		private final int value2Format;
		private final int targetValue1Format;
		private final int targetValue2Format;
		private final int targetValue1Size;
		private final int targetValue2Size;
		
		
		Type2Format1Generator(LayoutTable origTable, Subset subset, OTByteArrayBuilder newData, int newSTOffset,
				int origSTOffset, Map newCoverages) throws InvalidFontException, UnsupportedFontException 
		{
			super(origTable, subset, newData, newSTOffset, origSTOffset, 
					((NewCoverage)newCoverages.get(new Integer(origTable.data.getOffset(origSTOffset, 2)))).glyphCount, 
					8, false, false);
			
			value1Format = origTable.data.getuint16(origSTOffset + 4);
			value1Size = ((Gpos)origTable).getValueRecordSize(value1Format);
			value2Format = origTable.data.getuint16(origSTOffset + 6);
			value2Size = ((Gpos)origTable).getValueRecordSize(value2Format);
			
			targetValue1Format = targetValueFormat(value1Format);
			targetValue2Format = targetValueFormat(value2Format);
			targetValue1Size = ((Gpos)origTable).getValueRecordSize(targetValue1Format);
			targetValue2Size = ((Gpos)origTable).getValueRecordSize(targetValue2Format);

			writeHeader(((NewCoverage)newCoverages.get(new Integer(origTable.data.getOffset(origSTOffset, 2)))).glyphCount);
			origTable.iterateCoverage(origTable.data.getOffset(origSTOffset, 2), subset, this);
		}
		
		void writeHeader(int setCount) 
		{
			builder.ensureCapacity(newSTOffset + 8);
			builder.setuint16(newSTOffset, 1); // format.
			builder.setuint16(newSTOffset + 4, targetValue1Format); // ValueFormat1
			builder.setuint16(newSTOffset + 6, targetValue2Format); // ValueFormat2
			builder.setuint16(newSTOffset + 8, setCount); // PairCnt
		}
		
		boolean[] computeMembersToKeep(int newSetOffset, int origSetOffset) throws InvalidFontException, UnsupportedFontException {
			int pairValueCount = origTable.data.getuint16(origSetOffset);
			boolean keep[] = new boolean[pairValueCount];
			Arrays.fill(keep, true);
			
			for (int i = 0; i < pairValueCount; i++)
			{
				int secondGlyph = origTable.data.getuint16(origSetOffset + 2 + (2 + value1Size + value2Size)*i);
				if (subset.getExistingSubsetGid(secondGlyph) == -1)
					keep[i] = false;
			}
			
			return keep;
		}
		
		int whichRuleSetIndexApplies(int gid, int coverageIndex) throws InvalidFontException {
			return coverageIndex;
		}
		
		int writeMember(int newMemberOffset, int origMemberOffset) throws InvalidFontException, UnsupportedFontException {
			builder.ensureCapacity(newMemberOffset + 2 + targetValue1Size + targetValue2Size);
			int secondGlyph = origTable.data.getuint16(origMemberOffset);
			builder.setuint16(newMemberOffset, subset.getExistingSubsetGid(secondGlyph)); // SecondGlyph
			writeValueFormatFromOriginal(origMemberOffset + 2, newMemberOffset + 2, value1Format, targetValue1Format); // Value1
			writeValueFormatFromOriginal(origMemberOffset + 2 + value1Size,  // Value2
					newMemberOffset + 2 + targetValue1Size, value2Format, targetValue2Format);
			
			return 2 + targetValue1Size + targetValue2Size;
		}
		
		int getOrigRecordSize()
		{
			return 2 + value1Size + value2Size;
		}
	}
	
	private class Type2Format2Generator implements ClassConsumer, LookupSubtableGenerator {
		private int subtableSize = 0; 
		private int class1CountInSubset;
		private int class2CountInSubset;
		private final int origClass2Count;
		private int currentMax = 0;
		private final Subset subset;
		private final int value1Format;
		private final int value2Format;
		private final int value1Size;
		private final int value2Size;
		private final int target1Format;
		private final int target2Format;
		private final int target1Size;
		private final int target2Size;
		private final int newSTOffset;
		private final int origSTOffset;
		
		
		public Type2Format2Generator(int origSTOffset, int newSTOffset, Subset subset) 
		throws InvalidFontException, UnsupportedFontException 
		{
			this.subset = subset;
			this.newSTOffset = newSTOffset;
			this.origSTOffset = origSTOffset;
			
			int class1Offset = origTable.data.getOffset(origSTOffset, 8);
			int class2Offset = origTable.data.getOffset(origSTOffset, 10);
			
			origClass2Count = origTable.data.getuint16(origSTOffset + 14);
			
			origTable.iterateClass(class1Offset, origNumGlyphs, this, -1);
			class1CountInSubset = currentMax + 1;
			
			if (class1Offset == class2Offset)
				class2CountInSubset = currentMax;
			else
			{
				currentMax = 0;
				origTable.iterateClass(class2Offset, origNumGlyphs, this, -1);
				class2CountInSubset = currentMax + 1;
			}
			
			value1Format = origTable.data.getuint16(origSTOffset + 4);
			value2Format = origTable.data.getuint16(origSTOffset + 6);
			value1Size = ((Gpos)origTable).getValueRecordSize(value1Format);
			value2Size = ((Gpos)origTable).getValueRecordSize(value2Format);
			
			target1Format = targetValueFormat(value1Format);
			target2Format = targetValueFormat(value2Format);
			target1Size = ((Gpos)origTable).getValueRecordSize(target1Format);
			target2Size = ((Gpos)origTable).getValueRecordSize(target2Format);
		}
		
		private void writeHeader() throws InvalidFontException, UnsupportedFontException {
			subtableSize = 16 + (target1Size + target2Size)*class2CountInSubset*class1CountInSubset;
			builder.ensureCapacity(newSTOffset + subtableSize );
			builder.setuint16(newSTOffset, 2); //format
			builder.setuint16(newSTOffset+4, target1Format); //valueFormat1
			builder.setuint16(newSTOffset+ 6, target2Format); // valueFormat2
			subtableSize += writeClassDef(origSTOffset, 8, newSTOffset, subtableSize, subset, origNumGlyphs);
			
			int class1Offset = origTable.data.getOffset(origSTOffset, 8);
			int class2Offset = origTable.data.getOffset(origSTOffset, 10);
			
			if (class1Offset == class2Offset)
			{
				// They are still the same offset...
				builder.setuint16(newSTOffset+10, builder.getuint16(newSTOffset+8));
			} else
			{
				subtableSize += writeClassDef(origSTOffset, 10, newSTOffset, subtableSize, subset, origNumGlyphs);
			}
			
			builder.setuint16(newSTOffset+12, class1CountInSubset);
			builder.setuint16(newSTOffset + 14, class2CountInSubset);
		}
		
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException {
			writeHeader();
			
			for (int i = 0; i < class1CountInSubset; i++)
			{
				for (int j = 0; j < class2CountInSubset; j++)
				{
					writeValueFormatFromOriginal(origSTOffset+16+(value1Size+value2Size)*(origClass2Count*i + j), 
							newSTOffset + 16 + (target1Size + target2Size)*(class2CountInSubset*i + j), value1Format, target1Format);
					writeValueFormatFromOriginal(origSTOffset+16+(value1Size+value2Size)*(origClass2Count*i + j) + value1Size, 
							newSTOffset + 16 + (target1Size + target2Size)*(class2CountInSubset*i + j) + target1Size, value2Format, target2Format);
				}
			}
			
			return subtableSize;
		}
		
		public boolean glyph(int glyphID, int classID) throws UnsupportedFontException, InvalidFontException {
			if (subset.getExistingSubsetGid(glyphID) != -1 && classID > currentMax)
			{
				currentMax = classID;
			}
			return true;
		}
		
	}
	
	private class Type3Generator implements CoverageConsumer, LookupSubtableGenerator
	{
		private int subtableSize = 0;
		private final int origSTOffset;
		private final int newSTOffset;
		private final Subset subset;
		private int entryExitCount = 0;
		private int[] subsetGidToCoverageIndex;
		
		Type3Generator(int origSTOffset, int newSTOffset, Subset subset) throws InvalidFontException, UnsupportedFontException
		{
			this.origSTOffset = origSTOffset;
			this.newSTOffset = newSTOffset;
			this.subset = subset;
			
			subsetGidToCoverageIndex = new int[subset.getNumGlyphs()];
			Arrays.fill(subsetGidToCoverageIndex, -1);

			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			origTable.iterateCoverage(coverageOffset, subset, this);
		}
		
		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			subsetGidToCoverageIndex[subset.getExistingSubsetGid(gid)] = coverageIndex;
			entryExitCount++;
			return true;
		}
		
		public int writeSubtable() throws InvalidFontException
		{
			subtableSize = 6 + 4*entryExitCount;
			builder.ensureCapacity(newSTOffset + subtableSize);
			builder.setuint16(newSTOffset, 1);
			builder.setuint16(newSTOffset+4, entryExitCount);
			for (int i = 0, j = 0; i < entryExitCount*2 && j < subset.getNumGlyphs(); j++)
			{
				if (subsetGidToCoverageIndex[j] != -1)
				{
					for (int k = 0; k < 2; k++)
					{
						int origOffset = origTable.data.getOffset(origSTOffset, 6 + 4*subsetGidToCoverageIndex[j] + 2*k);
						if (origOffset == 0)
						{
							builder.setuint16(newSTOffset + 6 + 2*i, 0);
						} else
						{
							builder.setuint16(newSTOffset + 6 + 2*i, subtableSize);
							subtableSize += writeAnchorFromOriginal(origOffset, newSTOffset + subtableSize);
						}
						i++;
					}
					
				}
			}
			return subtableSize;
		}
		
	}
	
	/**
	 * The subset of classes to be added to a subset subtable. 
	 */
	private class ClassSubset extends SubsetDefaultImpl
	{
		ClassSubset(int numGlyphs) throws InvalidFontException, UnsupportedFontException
		{
			super(numGlyphs, true);
		}
	}

	private class MarkArrayGenerator implements CoverageConsumer
	{
		private final Subset subset;
		private int[] subsetToCoverageIndex;
		private int entryCount = 0;
		int subtableSize;
		ClassSubset classSubset;
		
		MarkArrayGenerator(Subset subset) throws InvalidFontException, UnsupportedFontException
		{
			this.subset = subset;
			this.subsetToCoverageIndex = new int[subset.getNumGlyphs()];
			Arrays.fill(this.subsetToCoverageIndex, -1);
		}
		
		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			subsetToCoverageIndex[subset.getExistingSubsetGid(gid)] = coverageIndex;
			entryCount++;
			return true;
		}
		
		int writeMarkArray(int coverageOffset, int origMAOffset, int newMAOffset, int origClassCount) throws InvalidFontException, UnsupportedFontException
		{
			origTable.iterateCoverage(coverageOffset, subset, this);
			
			classSubset = new ClassSubset(origClassCount);
			subtableSize = 2 + 4*entryCount;
			builder.ensureCapacity(newMAOffset + subtableSize);
			builder.setuint16(newMAOffset, entryCount);
			for (int i = 0, j = 0; i < entryCount && j < subset.getNumGlyphs(); j++)
			{
				if (subsetToCoverageIndex[j] != -1)
				{
					int origClass = origTable.data.getuint16(origMAOffset + 2 + 4*subsetToCoverageIndex[j]);
					builder.setuint16(newMAOffset + 2 + 4*i, classSubset.getSubsetGid(origClass)); // Class
					builder.setuint16(newMAOffset + 4 + 4*i, subtableSize); // MarkAnchor
					subtableSize += writeAnchorFromOriginal(origTable.data.getOffset(origMAOffset, 4 + 4*subsetToCoverageIndex[j]), 
							newMAOffset + subtableSize);
					i++;
				}
			}
			
			return subtableSize;
		}
	}
	
	// Base class for Mark-to-something generators (eg, MarkToBase).
	abstract private class MarkGenerator implements CoverageConsumer, LookupSubtableGenerator
	{
		protected final Subset subset;
		private final int origSTOffset;
		private final int newSTOffset;
		protected int entryCount = 0;
		protected int[] subsetToCoverage;
		
		MarkGenerator(int origSTOffset, int newSTOffset, Subset subset)
		{
			this.origSTOffset = origSTOffset;
			this.newSTOffset = newSTOffset;
			this.subset = subset;
			this.subsetToCoverage = new int[subset.getNumGlyphs()];
			Arrays.fill(subsetToCoverage, -1);
		}
		
		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			subsetToCoverage[subset.getExistingSubsetGid(gid)] = coverageIndex;
			entryCount++;
			return true;
		}		
		
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException
		{
			int subtableSize = 12;
			MarkArrayGenerator markGenerator = new MarkArrayGenerator(subset);
			int coverageOffset = origTable.data.getOffset(origSTOffset, 2);
			int origMAOffset = origTable.data.getOffset(origSTOffset, 8);
			int origClassCount = origTable.data.getuint16(origSTOffset + 6);
			builder.ensureCapacity(newSTOffset + subtableSize);
			builder.setuint16(newSTOffset, 1); // PosFormat
			builder.setuint16(newSTOffset + 8, subtableSize); // MarkArray offset
			
			subtableSize += markGenerator.writeMarkArray(coverageOffset, origMAOffset, newSTOffset + subtableSize, origClassCount);
			builder.setuint16(newSTOffset + 6, markGenerator.classSubset.getNumGlyphs()); // ClassCount
			

			origTable.iterateCoverage(origTable.data.getOffset(origSTOffset, 4), subset, this); // iterate the target coverage
			
			builder.setuint16(newSTOffset + 10, subtableSize); // Target Array offset (where the Target is the thing the Mark is attaching to)
			subtableSize += writeTargetArray(origTable.data.getOffset(origSTOffset, 10), 
					newSTOffset + subtableSize, markGenerator.classSubset, origClassCount);
			
			return subtableSize;
		}
		
		abstract int writeTargetArray(int origArrayOffset, int newArrayOffset, ClassSubset classSubset, int origClassCount) throws InvalidFontException;
	}
	
	private class Type4Or6Generator extends MarkGenerator
	{
		Type4Or6Generator(int origSTOffset, int newSTOffset, Subset subset)
		{
			super(origSTOffset, newSTOffset, subset);
		}

		int writeTargetArray(int origBAOffset, int newBAOffset, ClassSubset classSubset, int origClassCount) throws InvalidFontException
		{
			int bASize;
			bASize = 2 + 2*classSubset.getNumGlyphs()*entryCount;
			builder.ensureCapacity(newBAOffset + bASize);
			builder.setuint16(newBAOffset, entryCount);
			
			for (int i = 0, j = 0; i < entryCount && j < subset.getNumGlyphs(); j++)
			{
				if (subsetToCoverage[j] != -1)
				{
					for (int k = 0; k < classSubset.getNumGlyphs(); k++)
					{
						builder.setuint16(newBAOffset + 2 + 2*(i*classSubset.getNumGlyphs()+k), bASize); // BaseAnchor
						bASize += writeAnchorFromOriginal(origTable.data.getOffset(origBAOffset,  2 + 2*(subsetToCoverage[j]*origClassCount + classSubset.getFullGid(k))), newBAOffset + bASize);
					}
					i++;
				}
			}
			
			return bASize;
		}
		

	}
	
	private class Type5Generator extends MarkGenerator
	{
		Type5Generator(int origSTOffset, int newSTOffset, Subset subset)
		{
			super(origSTOffset, newSTOffset, subset);
		}

		int writeTargetArray(int origLAOffset, int newLAOffset, ClassSubset classSubset, int origClassCount) throws InvalidFontException
		{
			int lASize = 2 + 2*entryCount;
			builder.ensureCapacity(newLAOffset + lASize);
			builder.setuint16(newLAOffset, entryCount); // LigatureCount
			
			for (int i = 0, j = 0; i < entryCount && j < subset.getNumGlyphs(); j++)
			{
				if (subsetToCoverage[j] != -1)
				{
					builder.setuint16(newLAOffset + 2 + 2*i, lASize); // LigatureAttach
					lASize += writeArrayEntry(origLAOffset, 2 + 2*subsetToCoverage[j], newLAOffset + lASize, classSubset, origClassCount, subsetToCoverage[j]);
					i++;
				}
			}
			
			return lASize;
		}
		
		int writeArrayEntry(int origLAOffset, int origLAEntryDelta, int newLAEntryOffset, ClassSubset classSubset, int origClassCount, int origCoverageIndex) throws InvalidFontException
		{
			int ligAttachOffset = origTable.data.getOffset(origLAOffset, origLAEntryDelta);
			int componentCount = origTable.data.getuint16(ligAttachOffset);
			int entrySize = 2 + 2*componentCount*classSubset.getNumGlyphs();
			builder.ensureCapacity(newLAEntryOffset + entrySize);
			builder.setuint16(newLAEntryOffset, componentCount);
			for (int i = 0; i < componentCount; i++)
			{					
				for (int k = 0; k < classSubset.getNumGlyphs(); k++)
				{
					int origDelta = 2 + 2*(origClassCount*i + classSubset.getFullGid(k));
					int origOffset = origTable.data.getOffset(ligAttachOffset, origDelta);
					if (origOffset == 0)
					{
						builder.setuint16(newLAEntryOffset + 2 + 2*(i*classSubset.getNumGlyphs()+k), 0);
					}else
					{
						builder.setuint16(newLAEntryOffset + 2 + 2*(i*classSubset.getNumGlyphs()+k), entrySize); // LigatureAnchor
						entrySize += writeAnchorFromOriginal(origOffset, newLAEntryOffset + entrySize);
					}
				}
				
			}
			
			return entrySize;
		}
	}

	int writeSubtable(int origSTOffset, int newSTOffset, Map newCoverages, int lookupType, Subset subset, LookupSubset lookupSubset) throws InvalidFontException, UnsupportedFontException {

		LookupSubtableGenerator generator;
		
	   	switch (lookupType) {
	      case 1: 
	      {
	    	  int origFormat = origTable.data.getuint16(origSTOffset);
	    	  int targetFormat;
	    	  int[] value;
	    	  if (origFormat == 2)
	    	  {
	    		  L1F2TargetFormatDeterminer consumer = new L1F2TargetFormatDeterminer(origSTOffset);
	    		  targetFormat = consumer.canUseFormat1(subset) ? 1:2;
	    		  value = consumer.bytes;
	    	  } else {
	    		  targetFormat = 1;
	    		  value = new int[((Gpos)origTable).getValueRecordSize(origTable.data.getuint16(origSTOffset + 4))/2];
	    		  for (int i = 0; i < value.length; i++)
				  {
	    			  value[i] = origTable.data.getint16(origSTOffset + 6 + 2*i);
				  }
	    	  }
	    	  
	    	  if (targetFormat == 1)
	    	  {
	    		  return writeLookup1Format1(origSTOffset, newSTOffset, value);
	    	  } else
	    	  {
	    		  generator = new Type1Format2Generator(origSTOffset, newSTOffset, subset);
	    	  }
    		  break;
	      }
	      case 2:
	      {
	    	  int origFormat = origTable.data.getuint16(origSTOffset);
	    	  switch(origFormat)
	    	  {
	    	  case 1: {
	    		  generator = new Type2Format1Generator(origTable, subset, builder, newSTOffset, origSTOffset, newCoverages);
	    		  break;
	    	  }
	    		  
	    	  case 2: {
	    		  generator = new Type2Format2Generator(origSTOffset, newSTOffset, subset);
	    		  break;

	    	  } 
	    	  default:
	    		  throw new InvalidFontException("Invalid type 2 gpos lookup: " + origFormat);
	    	  }
	    	  break;
	      }
	    	  
	      case 3:
	    	  generator = new Type3Generator(origSTOffset, newSTOffset, subset);
	    	  break;
	    	  
	      case 4: 
	      case 6:
	    	  generator = new Type4Or6Generator(origSTOffset, newSTOffset, subset);
	    	  break;
	    	  
	      case 5:
	    	  generator = new Type5Generator(origSTOffset, newSTOffset, subset);
	    	  break;
	      
	      case 7:
	    	  generator = ContextualGenerator.newContextualGenerator(origTable, builder,
	    			  origSTOffset, newSTOffset, subset, newCoverages, lookupSubset, origNumGlyphs);
	    	  break;
	      
	      case 8: {
	    	  generator = ChainingGenerator.newChainingInstance(origTable, builder, origSTOffset, newSTOffset, subset, 
	    				newCoverages, lookupSubset, origNumGlyphs);
	    	  break;
	    	  
	      }
	      case 9:
	    	  builder.setuint16(newSTOffset, 1);
	    	  builder.setuint16(newSTOffset + 2, origTable.data.getuint16(origSTOffset+2));
	    	  return 8;
	    	  
    	  default:
    		  throw new InvalidFontException ("Invalid GPOS lookup type (" + lookupType + ")");   
	   	}
		return generator.writeSubtable();
	}

	int getReferencedLookupType(int origExtSubOffset) throws InvalidFontException {
		return origTable.data.getuint16(origExtSubOffset + 2);
	}

	void patchExtensionOffset(int extensionSubtableOffset, int nonExtensionSubtableOffset) {
	  	  builder.setuint32(extensionSubtableOffset + 4, nonExtensionSubtableOffset - extensionSubtableOffset);		
	}

	int getReferencedSubtableOffset(int origSTOffset) throws InvalidFontException, UnsupportedFontException {
		return origSTOffset + origTable.data.getuint32asint(origSTOffset + 4, "Only handle extensions that fit in 31 bits");
	}

	int getExtensionLookupType() {		
		return 9;
	}

	int writeExtensionSubtable(int newSTOffset, int subTableLookupType) {
		builder.setuint16(newSTOffset, 1);
	  	builder.setuint16(newSTOffset + 2, subTableLookupType);
	  	return 8;
	}

	boolean mustKeepFeature(int origFeatureTagOffset) throws InvalidFontException {
		// If the 'kern' table exists in the subset font and the 'kern' feature exists
		// in the original font and the 'kern' feature has lookups associated with it, we
		// have to keep the 'kern' feature in the subset table. Otherwise, CTS will use the 'kern'
		// table, which it would not have done with the original font, and we could get different behavior
		// between the original and subset font.
		if (preserveNonEmptyKernFeature)
		{
			int[] kernTag = {'k','e','r','n'}; 
			for (int i = 0; i < 4; i++)
			{
				if (origTable.data.getuint8(origFeatureTagOffset + i) != kernTag[i])
					return false;
			}
			
			int featureOffset = origTable.data.getOffset(origFeatureTagOffset, 4);
			int numFeatures = origTable.data.getuint16(featureOffset + 2);
			return (numFeatures > 0);
		}
		return false;
	}
}
